from app import db
from sqlalchemy import JSON, Index, BigInteger, Text, Numeric
from datetime import datetime

class User(db.Model):
    """Пользователи Telegram бота - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'users'
    
    # Основные поля
    id = db.Column(BigInteger, primary_key=True)  # Telegram User ID
    telegram_username = db.Column(db.String(255))
    first_name = db.Column(db.String(255))
    last_name = db.Column(db.String(255))
    registration_date = db.Column(db.DateTime)
    user_status = db.Column(db.String(50))
    email = db.Column(db.String(255))
    phone = db.Column(db.String(50))
    referral_code = db.Column(db.String(50))
    referred_by = db.Column(BigInteger)
    total_paid = db.Column(Numeric)
    created_at = db.Column(db.DateTime)
    updated_at = db.Column(db.DateTime)
    username = db.Column(db.String(255))  # Дублирование для совместимости
    status = db.Column(db.String(50))     # Дублирование для совместимости
    vpn_email = db.Column(db.String(255)) # Дублирование для совместимости
    phone_number = db.Column(db.String(50)) # Дублирование для совместимости
    
    # Отношения
    subscriptions = db.relationship('Subscription', backref='user', lazy=True)
    payments = db.relationship('Payment', backref='user', lazy=True)
    
    def __repr__(self):
        return f'<User {self.id}: {self.username or self.first_name}>'

class Subscription(db.Model):
    """Подписки пользователей - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'subscriptions'
    
    # Основные поля
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(BigInteger, nullable=False)
    plan_id = db.Column(db.String(50), nullable=False)
    plan_status = db.Column(db.String(100))
    status = db.Column(db.String(50))
    expires_at = db.Column(db.DateTime)
    traffic_limit = db.Column(BigInteger)
    traffic_used = db.Column(BigInteger)
    traffic_limit_gb = db.Column(BigInteger)
    traffic_used_gb = db.Column(BigInteger)
    vpn_key = db.Column(Text)
    created_at = db.Column(db.DateTime)
    updated_at = db.Column(db.DateTime)
    vpn_user_id = db.Column(db.String(255))
    vpn_email = db.Column(db.String(255))
    
    def __repr__(self):
        return f'<Subscription {self.id}: {self.plan_id} for user {self.user_id}>'

class Payment(db.Model):
    """Платежи - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'payments'
    
    # Основные поля
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(BigInteger, nullable=False)
    subscription_id = db.Column(db.Integer)
    payment_id = db.Column(db.String(255), nullable=False)
    amount = db.Column(Numeric, nullable=False)
    currency = db.Column(db.String(10))
    status = db.Column(db.String(50))
    payment_method = db.Column(db.String(100))
    description = db.Column(Text)
    created_at = db.Column(db.DateTime)
    paid_at = db.Column(db.DateTime)
    plan_id = db.Column(db.String(50))
    completed_at = db.Column(db.DateTime)
    
    def __repr__(self):
        return f'<Payment {self.id}: {self.amount} {self.currency} - {self.status}>'

# Дополнительные таблицы для системы уведомлений
class NotificationSettings(db.Model):
    """Настройки уведомлений по тарифам - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'notification_settings'
    
    id = db.Column(db.Integer, primary_key=True)
    plan_id = db.Column(db.String, nullable=False)
    plan_name = db.Column(db.String, nullable=False)
    days_before = db.Column(JSON, nullable=False)
    send_time_start = db.Column(db.Integer)
    send_time_end = db.Column(db.Integer)
    check_interval_hours = db.Column(db.Integer)
    enabled = db.Column(db.Boolean)
    created_at = db.Column(db.DateTime)
    updated_at = db.Column(db.DateTime)

class NotificationQueue(db.Model):
    """Очередь уведомлений - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'notification_queue'
    
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(BigInteger, nullable=False)
    subscription_id = db.Column(db.Integer, nullable=False)
    notification_type = db.Column(db.String, nullable=False)
    days_before_expiry = db.Column(db.Integer)
    scheduled_at = db.Column(db.DateTime, nullable=False)
    sent = db.Column(db.Boolean)
    created_at = db.Column(db.DateTime)

class NotificationLog(db.Model):
    """Логи отправленных уведомлений - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'notification_log'
    
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(BigInteger, nullable=False)
    subscription_id = db.Column(db.Integer, nullable=False)
    notification_type = db.Column(db.String, nullable=False)
    days_before_expiry = db.Column(db.Integer)
    message_text = db.Column(Text)
    sent_at = db.Column(db.DateTime)
    success = db.Column(db.Boolean)
    error_message = db.Column(Text)

class AppSettings(db.Model):
    """Настройки приложения - полное соответствие PostgreSQL схеме"""
    __tablename__ = 'app_settings'
    
    id = db.Column(db.Integer, primary_key=True)
    key = db.Column(db.String, nullable=False)
    value = db.Column(Text)
    description = db.Column(Text)
    created_at = db.Column(db.DateTime)
    updated_at = db.Column(db.DateTime)

# Индексы для оптимизации запросов (соответствуют реальной PostgreSQL структуре)
Index('idx_users_telegram_username', User.telegram_username)
Index('idx_users_email', User.email)
Index('idx_subscriptions_user_id', Subscription.user_id)
Index('idx_subscriptions_expires_at', Subscription.expires_at)
Index('idx_subscriptions_status', Subscription.status)
Index('idx_subscriptions_vpn_email', Subscription.vpn_email)
Index('idx_payments_user_id', Payment.user_id)
Index('idx_payments_payment_id', Payment.payment_id)
Index('idx_payments_status', Payment.status)
Index('idx_notification_queue_scheduled_at', NotificationQueue.scheduled_at)
Index('idx_notification_queue_sent', NotificationQueue.sent)
Index('idx_notification_log_user_id', NotificationLog.user_id)
Index('idx_notification_log_sent_at', NotificationLog.sent_at)