import asyncio
import logging
import sys
from aiogram import Bot, Dispatcher
from aiogram.fsm.storage.memory import MemoryStorage
from aiogram.client.default import DefaultBotProperties

from config import BOT_TOKEN, PANEL_API_URL, PANEL_USERNAME, PANEL_PASSWORD
from bot.database.operations import init_db
from bot.handlers import main_menu, subscription, payment, personal, admin_handlers
from bot.services import traffic_manager
# Старый планировщик удален - используем только EnhancedNotificationScheduler
from bot.jobs.enhanced_notification_scheduler import EnhancedNotificationScheduler
from bot.services.simple_panel_api import simple_panel_api

# Импортируем Flask-приложение для Gunicorn
from app import app

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Initialize bot and dispatcher
bot = Bot(
    token=BOT_TOKEN,
    default=DefaultBotProperties(parse_mode="HTML")
)
dp = Dispatcher(storage=MemoryStorage())


async def on_startup():
    # Initialize database - SKIP SQLite, use PostgreSQL only
    try:
        from bot.database.postgres_operations import get_user_postgres
        logger.info("✓ PostgreSQL initialized - SQLite bypassed for testing")
    except Exception as e:
        logger.error(f"PostgreSQL initialization error: {e}")
    
    # SQLite config database инициализируется автоматически при первом обращении
    logger.info("✓ SQLite config database configured with lazy initialization")
    
    # Фоновая авторизация в панели для ускорения запуска
    async def background_panel_init():
        """Фоновая инициализация панели без блокировки запуска"""
        logger.info("Фоновая авторизация в панели 3X-UI...")
        try:
            from bot.services.working_panel_api import working_api
            login_success = await working_api.login()
            if login_success:
                logger.info("✓ Фоновая авторизация в панели успешна")
            else:
                logger.warning("⚠ Фоновая авторизация в панели неуспешна")
        except Exception as e:
            logger.warning(f"Фоновая авторизация в панели: {e}")
    
    # Запускаем авторизацию в фоне без ожидания
    asyncio.create_task(background_panel_init())
    
    # Установка команд бота
    from aiogram.types import BotCommand, BotCommandScopeDefault, BotCommandScopeChat
    from config import ADMIN_IDS
    
    # Общие команды для всех пользователей
    await bot.set_my_commands([
        BotCommand(command="start", description="Запустить бота / Вернуться в главное меню"),
        BotCommand(command="subscription", description="Оформить новую подписку"),
        BotCommand(command="extend", description="Продлить подписку"),
        BotCommand(command="account", description="Личный кабинет")
    ], scope=BotCommandScopeDefault())
    
    # Административные команды только для админов
    admin_commands = [
        BotCommand(command="start", description="Запустить бота / Вернуться в главное меню"),
        BotCommand(command="subscription", description="Оформить новую подписку"),
        BotCommand(command="extend", description="Продлить подписку"),
        BotCommand(command="account", description="Личный кабинет"),
        # Административные команды
        BotCommand(command="admin", description="Список административных команд"),
        BotCommand(command="stats", description="Статистика бота"),
        BotCommand(command="users", description="Список пользователей"),
        BotCommand(command="add_client", description="Добавить нового VPN-клиента"),
        BotCommand(command="reset_subscription", description="Сбросить подписку пользователя")
    ]
    
    # Устанавливаем админ-команды для каждого админа
    for admin_id in ADMIN_IDS:
        try:
            await bot.set_my_commands(
                admin_commands,
                scope=BotCommandScopeChat(chat_id=admin_id)
            )
        except Exception as e:
            logger.error(f"Ошибка при установке команд для админа {admin_id}: {e}")
    
    # Set up command handlers
    logger.info("Registering handlers...")
    
    # Register all handlers
    main_menu.register_handlers(dp)
    subscription.register_handlers(dp)

    payment.register_handlers(dp)  # payment должен быть последним из-за универсального обработчика
    personal.register_handlers(dp)
    admin_handlers.register_admin_handlers(dp)  # Регистрируем обработчики административных команд
    
    # Регистрируем обработчики дополнительных сервисов
    from bot.handlers import additional_services
    dp.include_router(additional_services.router)
    # optimized_refresh_link удален для упрощения кода
    
    # Старый планировщик удален - используем только EnhancedNotificationScheduler
    
    # Start notification service
    notification_scheduler = EnhancedNotificationScheduler(bot)
    await notification_scheduler.start()
    logger.info("Notification service started")
    
    # Register traffic management handlers
    traffic_manager.register_handlers(dp)
    logger.info("Traffic management handlers registered")
    
    logger.info("Bot started!")


async def on_shutdown():
    """Функция корректного завершения работы бота"""
    logger.info("Завершение работы бота...")
    
    # Закрываем все активные aiohttp сессии через глобальный менеджер
    try:
        from bot.services.session_manager import global_session_manager
        await global_session_manager.close_all()
        logger.info("✓ Все HTTP-сессии закрыты через глобальный менеджер")
    except Exception as e:
        logger.error(f"Ошибка закрытия глобального менеджера сессий: {e}")
    
    # Закрываем отдельные API объекты
    try:
        import sys
        if 'bot.services.simple_panel_api' in sys.modules:
            module = sys.modules['bot.services.simple_panel_api']
            if hasattr(module, 'simple_panel_api'):
                await module.simple_panel_api.close()
                logger.info("✓ Простой API панели закрыт")
    except Exception as e:
        logger.error(f"Ошибка закрытия простого API: {e}")
    
    try:
        if 'bot.services.working_panel_api' in sys.modules:
            module = sys.modules['bot.services.working_panel_api']
            if hasattr(module, 'working_api'):
                await module.working_api.close()
                logger.info("✓ Рабочий API панели закрыт")
    except Exception as e:
        logger.error(f"Ошибка закрытия рабочего API: {e}")
    
    try:
        # Закрываем глобальный PanelSessionManager
        if 'bot.services.panel_session_manager' in sys.modules:
            module = sys.modules['bot.services.panel_session_manager']
            if hasattr(module, 'panel_session_manager'):
                await module.panel_session_manager.close()
                logger.info("✓ Глобальный менеджер панели закрыт")
    except Exception as e:
        logger.error(f"Ошибка закрытия менеджера панели: {e}")
    
    logger.info("✓ Завершение работы выполнено")


async def main():
    logger.info("Starting bot...")
    
    # Set up startup and shutdown actions
    dp.startup.register(on_startup)
    dp.shutdown.register(on_shutdown)
    
    # Start polling
    await dp.start_polling(bot)


if __name__ == "__main__":
    try:
        asyncio.run(main())
    except (KeyboardInterrupt, SystemExit):
        logger.info("Bot stopped!")
