import os
from datetime import datetime, timedelta
import threading
import time
import asyncio
import sqlite3
import logging

from flask import Flask, render_template, request, redirect, url_for, jsonify, flash, make_response
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy.orm import DeclarativeBase
from sqlalchemy import text, create_engine
from werkzeug.middleware.proxy_fix import ProxyFix

# Настройка логирования для веб-приложения
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('web_app.log'),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger(__name__)


class Base(DeclarativeBase):
    pass


db = SQLAlchemy(model_class=Base)
# create the app
app = Flask(__name__)
app.secret_key = os.environ.get("SESSION_SECRET", "a_secret_key_for_development")
app.wsgi_app = ProxyFix(
    app.wsgi_app, x_proto=1, x_host=1
)  # needed for url_for to generate with https

# configure the database, relative to the app instance folder
from config import DATABASE_FILE

# Оптимизированная конфигурация PostgreSQL для быстрого запуска
app.config["SQLALCHEMY_DATABASE_URI"] = os.environ.get('DATABASE_URL', f"sqlite:///{DATABASE_FILE}")
app.config["SQLALCHEMY_ENGINE_OPTIONS"] = {
    "pool_recycle": 300,
    "pool_pre_ping": True,
    "pool_size": 5,
    "max_overflow": 10,
    "connect_args": {
        "connect_timeout": 5,
        "application_name": "RazDvaVPN_Web"
    }
}
# initialize the app with the extension, flask-sqlalchemy >= 3.0.x
db.init_app(app)

# Регистрируем blueprint для системы бекапов
from backup_web_interface import backup_bp
app.register_blueprint(backup_bp)

# Простая система кеширования
_cache = {}
_cache_lock = threading.Lock()

def get_cached_data(key, max_age_seconds=1):
    """Получить данные из кеша если они еще актуальны"""
    with _cache_lock:
        if key in _cache:
            data, timestamp = _cache[key]
            if datetime.now() - timestamp < timedelta(seconds=max_age_seconds):
                return data
    return None

def set_cached_data(key, data):
    """Сохранить данные в кеш"""
    with _cache_lock:
        _cache[key] = (data, datetime.now())
        # Очищаем старые записи (простая очистка)
        cutoff = datetime.now() - timedelta(minutes=30)
        keys_to_remove = [k for k, (_, ts) in _cache.items() if ts < cutoff]
        for k in keys_to_remove:
            del _cache[k]

def clear_cache():
    """Очистить весь кеш"""
    with _cache_lock:
        _cache.clear()
        logger.info("Кеш веб-приложения очищен")


def run_async_safely(coroutine):
    """Безопасное выполнение асинхронных функций в синхронном контексте"""
    try:
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        try:
            return loop.run_until_complete(coroutine)
        finally:
            loop.close()
    except Exception as e:
        logger.error(f"Ошибка выполнения асинхронной функции: {e}")
        return None

@app.route("/health")
def health_check():
    """Health check endpoint для деплоя"""
    return {"status": "healthy", "timestamp": datetime.now().isoformat()}, 200

@app.route("/")
def index():
    """Главная страница с информацией о VPN сервисе"""
    return render_template("index.html")


def send_sync_test_notification(user_id: int):
    """Синхронная отправка тестового уведомления через логирование"""
    try:
        import sqlite3
        from datetime import datetime
        
        # Записываем в лог базы данных
        conn = sqlite3.connect(DATABASE_FILE)
        cursor = conn.cursor()
        
        cursor.execute("""
            INSERT INTO notification_log 
            (user_id, subscription_id, notification_type, message_text, sent_at, success)
            VALUES (?, 0, 'test', 'Тестовое уведомление из веб-интерфейса', ?, 1)
        """, (user_id, datetime.now().isoformat()))
        
        conn.commit()
        conn.close()
        
        logger.info(f"Тестовое уведомление записано для пользователя {user_id}")
        return True
    except Exception as e:
        logger.error(f"Ошибка отправки тестового уведомления: {e}")
        return False


@app.route("/status")
def status():
    """Страница со статусом сервисов и статистикой системы"""
    from sync_web_stats import get_sync_web_statistics

    # Получаем реальную статистику из базы данных
    try:
        stats = get_sync_web_statistics()
    except Exception as e:
        # В случае ошибки используем базовую статистику
        from datetime import datetime

        stats = {
            "total_users": 0,
            "active_subscriptions": 0,
            "trial_subscriptions": 0,
            "expiring_soon": 0,
            "bot_version": "1.2.0",
            "last_restart": datetime.now().strftime("%d.%m.%Y %H:%M"),
            "uptime": "0:00:00",
            "last_sync": datetime.now().strftime("%d.%m.%Y %H:%M"),
            "sync_success_rate": 0,
            "error": str(e),
        }

    return render_template("status.html", **stats)


@app.route("/api/status")
def api_status():
    """API эндпоинт для проверки статуса приложения"""
    return jsonify({"status": "ok", "service": "VPN Bot API", "version": "1.0.0"})


@app.route("/api/panel-sync-status")
def api_panel_sync_status():
    """API эндпоинт для получения статуса синхронизации с панелью 3X-UI"""
    try:
        # Проверяем статус синхронизации
        sync_status = run_async_safely(check_panel_sync_status())
        return jsonify(sync_status)
    except Exception as e:
        return jsonify({
            "status": "error",
            "panel_connected": False,
            "last_sync": "Неизвестно",
            "sync_status": "Ошибка",
            "active_clients": 0,
            "error": str(e)
        })


@app.route("/api/trigger-panel-sync", methods=["POST"])
def api_trigger_panel_sync():
    """API эндпоинт для принудительной синхронизации с панелью 3X-UI"""
    try:
        # Запускаем синхронизацию трафика
        sync_result = run_async_safely(trigger_panel_sync())
        return jsonify(sync_result)
    except Exception as e:
        return jsonify({
            "status": "error",
            "message": f"Ошибка синхронизации: {str(e)}"
        })


async def trigger_panel_sync():
    """Принудительная синхронизация трафика с панелью 3X-UI"""
    try:
        import sys
        import os
        from datetime import datetime
        
        # Добавляем путь к сервисам панели
        sys.path.append(os.path.join(os.path.dirname(__file__), 'bot', 'services'))
        
        try:
            from working_panel_api import WorkingPanelAPI
        except ImportError:
            return {
                "status": "error", 
                "message": "Модуль синхронизации недоступен"
            }
        
        # Инициализируем API панели
        panel_api = WorkingPanelAPI()
        
        # Проверяем подключение
        if not await panel_api.login():
            await panel_api.close()
            return {
                "status": "error",
                "message": "Не удалось подключиться к панели 3X-UI"
            }
        
        # Получаем активные подписки из базы
        try:
            from sqlalchemy import create_engine, text
            
            database_url = os.environ.get('DATABASE_URL')
            if not database_url:
                return {
                    "status": "error",
                    "message": "URL базы данных не настроен"
                }
            
            engine = create_engine(database_url)
            
            with engine.connect() as conn:
                result = conn.execute(text("""
                    SELECT id, user_id, vpn_email, traffic_limit, traffic_used 
                    FROM subscriptions 
                    WHERE status = 'ACTIVE'
                    ORDER BY id
                """))
                
                subscriptions = result.fetchall()
                
                if not subscriptions:
                    await panel_api.close()
                    return {
                        "status": "success",
                        "message": "Нет активных подписок для синхронизации",
                        "updated": 0,
                        "errors": 0
                    }
                
                updated_count = 0
                error_count = 0
                
                # Синхронизируем каждую подписку
                for sub in subscriptions:
                    try:
                        # Получаем данные о трафике из панели
                        client_stats = await panel_api.get_traffic_by_email(sub.vpn_email)
                        
                        if client_stats and "total" in client_stats:
                            current_usage_bytes = client_stats["total"]
                            
                            # Обновляем только если есть изменения
                            if current_usage_bytes != sub.traffic_used:
                                conn.execute(text("""
                                    UPDATE subscriptions 
                                    SET traffic_used = :traffic_used
                                    WHERE id = :sub_id
                                """), {
                                    "traffic_used": current_usage_bytes,
                                    "sub_id": sub.id
                                })
                                conn.commit()
                                updated_count += 1
                        
                    except Exception as e:
                        logger.error(f"Ошибка синхронизации подписки {sub.id}: {e}")
                        error_count += 1
                
                await panel_api.close()
                
                return {
                    "status": "success",
                    "message": f"Синхронизация завершена: обновлено {updated_count}, ошибок {error_count}",
                    "updated": updated_count,
                    "errors": error_count,
                    "total": len(subscriptions)
                }
                
        except Exception as db_error:
            await panel_api.close()
            return {
                "status": "error",
                "message": f"Ошибка работы с базой данных: {str(db_error)}"
            }
            
    except Exception as e:
        return {
            "status": "error", 
            "message": f"Общая ошибка синхронизации: {str(e)}"
        }


async def check_panel_sync_status():
    """Проверка статуса синхронизации с панелью 3X-UI"""
    try:
        import sys
        import os
        from datetime import datetime
        
        # Добавляем путь к сервисам панели
        sys.path.append(os.path.join(os.path.dirname(__file__), 'bot', 'services'))
        
        try:
            from working_panel_api import WorkingPanelAPI
        except ImportError:
            # Если модуль не найден, возвращаем базовую информацию
            return {
                "status": "error", 
                "panel_connected": False,
                "last_sync": "Модуль не найден",
                "sync_status": "Недоступно",
                "active_clients": 0,
                "error": "Модуль working_panel_api не найден"
            }
        
        # Инициализируем API панели
        panel_api = WorkingPanelAPI()
        
        # Проверяем подключение
        panel_connected = await panel_api.login()
        
        if panel_connected:
            # Получаем информацию о клиентах и детали авторизации
            try:
                clients_data = await panel_api.get_clients()
                active_clients = len([c for c in clients_data if c.get('enable', False)]) if clients_data else 0
                
                # Проверяем наличие cookie
                cookie_status = "Активен" if hasattr(panel_api, 'auth_cookie') and panel_api.auth_cookie else "Отсутствует"
                
                # Время последнего успешного вызова API панели - это текущий момент
                # поскольку мы только что успешно получили клиентов
                current_time = datetime.now()
                last_sync = current_time.strftime("%d.%m.%Y %H:%M")
                sync_status = "Активна"  # Синхронизация только что произошла
                
            except Exception as e:
                active_clients = 0
                last_sync = "Ошибка API вызова"
                sync_status = "Ошибка"
                cookie_status = "Ошибка"
                logger.error(f"Ошибка получения клиентов: {e}")
            
            finally:
                await panel_api.close()
            
            return {
                "status": "success",
                "panel_connected": True,
                "last_sync": last_sync,
                "sync_status": sync_status,
                "active_clients": active_clients,
                "cookie_status": cookie_status,
                "login_success": True,
                "panel_url": "31.172.75.92:47773",
                "connection_time": datetime.now().strftime("%H:%M:%S")
            }
        else:
            await panel_api.close()
            return {
                "status": "error",
                "panel_connected": False,
                "last_sync": "Нет подключения",
                "sync_status": "Отключена",
                "active_clients": 0,
                "cookie_status": "Недоступен",
                "login_success": False,
                "panel_url": "31.172.75.92:47773",
                "connection_time": datetime.now().strftime("%H:%M:%S"),
                "error": "Не удалось подключиться к панели"
            }
            
    except Exception as e:
        logger.error(f"Ошибка проверки статуса панели: {e}")
        return {
            "status": "error", 
            "panel_connected": False,
            "last_sync": "Ошибка сети",
            "sync_status": "Ошибка",
            "active_clients": 0,
            "cookie_status": "Ошибка",
            "login_success": False,
            "panel_url": "31.172.75.92:47773",
            "connection_time": datetime.now().strftime("%H:%M:%S"),
            "error": str(e)
        }


@app.route("/api/users")
def api_users():
    """API эндпоинт для получения данных пользователей без перезагрузки страницы"""
    try:
        from sqlalchemy import create_engine, text
        from datetime import datetime
        import os

        engine = create_engine(os.environ.get('DATABASE_URL'))
        
        with engine.connect() as conn:
            query = text("""
            SELECT 
                u.id as user_id,
                u.username,
                u.first_name,
                u.last_name,
                u.created_at as user_created_at,
                u.status,
                u.vpn_email,
                COUNT(s.id) as subscriptions_count,
                SUM(CASE WHEN s.status = 'ACTIVE' THEN 1 ELSE 0 END) as active_subscriptions,
                MAX(s.expires_at) as latest_expiry,
                SUM(CASE WHEN s.status = 'ACTIVE' THEN s.traffic_used ELSE 0 END) as total_traffic_used,
                SUM(CASE WHEN s.status = 'ACTIVE' THEN s.traffic_limit ELSE 0 END) as total_traffic_limit
            FROM users u
            LEFT JOIN subscriptions s ON u.id = s.user_id
            GROUP BY u.id, u.username, u.first_name, u.last_name, u.created_at, u.status, u.vpn_email
            ORDER BY u.created_at DESC
            LIMIT 50
            """)

            result = conn.execute(query)
            users_data = result.fetchall()

        users_list = []
        for user in users_data:
            user_dict = dict(user._mapping)
            user_dict["subscriptions_count"] = user_dict["subscriptions_count"] or 0
            user_dict["active_subscriptions"] = user_dict["active_subscriptions"] or 0
            user_dict["total_traffic_used"] = user_dict["total_traffic_used"] or 0
            user_dict["total_traffic_limit"] = user_dict["total_traffic_limit"] or 0

            # Вычисляем процент использования трафика
            # traffic_limit в ГБ, traffic_used в байтах - приводим к одинаковым единицам
            if user_dict["total_traffic_limit"] > 0:
                traffic_limit_bytes = user_dict["total_traffic_limit"] * 1024 * 1024 * 1024
                user_dict["traffic_percentage"] = round(
                    (user_dict["total_traffic_used"] / traffic_limit_bytes) * 100, 1
                )
            else:
                user_dict["traffic_percentage"] = 0

            users_list.append(user_dict)

        return jsonify(
            {
                "status": "success",
                "users": users_list,
                "count": len(users_list),
                "last_update": datetime.now().strftime("%d.%m.%Y %H:%M:%S"),
            }
        )

    except Exception as e:
        return jsonify({"status": "error", "message": str(e)}), 500


@app.route("/users")
def users():
    """Страница со списком всех пользователей и их данными с пагинацией"""
    from sqlalchemy import create_engine, text
    from datetime import datetime
    import os

    # Получаем параметры пагинации
    page = request.args.get('page', 1, type=int)
    per_page = request.args.get('per_page', 50, type=int)
    per_page = min(per_page, 100)
    offset = (page - 1) * per_page

    try:
        logger.info(f"Загружаем данные страницы {page} из PostgreSQL")
        
        # Подключение к PostgreSQL
        engine = create_engine(os.environ.get('DATABASE_URL'))
        
        with engine.connect() as conn:
            # Обновляем статусы истекших подписок
            update_expired_subscription_statuses()
            
            # Подсчет общего количества пользователей
            count_result = conn.execute(text("SELECT COUNT(*) FROM users"))
            total_users = count_result.scalar()
            total_pages = (total_users + per_page - 1) // per_page

            # Получение пользователей с основной информацией
            query = text("""
            SELECT 
                u.id as user_id,
                u.username,
                u.first_name,
                u.last_name,
                u.created_at,
                u.status,
                u.vpn_email,
                COUNT(DISTINCT s.id) as subscriptions_count,
                COUNT(DISTINCT CASE WHEN s.status = :active_status THEN s.id END) as active_subscriptions,
                MAX(s.expires_at) as latest_expiry,
                COALESCE((SELECT SUM(amount) FROM payments WHERE user_id = u.id AND status = :payment_status), 0) as total_paid,
                COALESCE((SELECT COUNT(*) FROM payments WHERE user_id = u.id AND status = :payment_status), 0) as payments_count,
                COALESCE(MAX(CASE WHEN s.status = :active_status THEN s.traffic_used END), 0) as total_traffic_used,
                COALESCE(MAX(CASE WHEN s.status = :active_status THEN s.traffic_limit END), 0) as total_traffic_limit,
                MAX(CASE WHEN s.status = :active_status THEN s.plan_id END) as active_plan_id,
                MAX(CASE WHEN s.status = :active_status THEN s.plan_status END) as active_plan_status
            FROM users u
            LEFT JOIN subscriptions s ON u.id = s.user_id
            GROUP BY u.id, u.username, u.first_name, u.last_name, u.created_at, u.status, u.vpn_email
            ORDER BY u.created_at DESC
            LIMIT :per_page OFFSET :offset
            """)

            result = conn.execute(query, {
                "per_page": per_page, 
                "offset": offset,
                "active_status": "ACTIVE",
                "payment_status": "SUCCEEDED"
            })
            users_data = result.fetchall()

            # Обработка данных пользователей
            users_list = []
            for user_row in users_data:
                user_dict = dict(user_row._mapping)
                
                # Обработка None значений
                user_dict["total_paid"] = float(user_dict["total_paid"] or 0)
                user_dict["subscriptions_count"] = user_dict["subscriptions_count"] or 0
                user_dict["active_subscriptions"] = user_dict["active_subscriptions"] or 0
                user_dict["payments_count"] = user_dict["payments_count"] or 0
                user_dict["total_traffic_used"] = user_dict["total_traffic_used"] or 0
                user_dict["total_traffic_limit"] = user_dict["total_traffic_limit"] or 0
                
                # Карта тарифов для отображения названий
                plan_names = {
                    'basic': 'Базовый',
                    'weekly': 'Недельный',
                    'monthly': 'Месячный',
                    'premium': 'Премиум'
                }
                
                # Получение названия активного тарифа
                active_plan_name = "Нет подписки"
                if user_dict.get("active_plan_id"):
                    active_plan_name = plan_names.get(user_dict["active_plan_id"], user_dict["active_plan_id"])
                
                # Правильный расчет трафика в ГБ
                traffic_used_gb = 0
                traffic_limit_gb = 0
                traffic_percentage = 0
                
                if user_dict["total_traffic_limit"] > 0:
                    # Используем байты из базы данных и конвертируем в ГБ
                    traffic_used_gb = user_dict["total_traffic_used"] / (1024**3)
                    
                    # Проверяем, хранится ли лимит в байтах (большое число) или ГБ (маленькое)
                    if user_dict["total_traffic_limit"] > 1000:  # Если больше 1000, это байты
                        traffic_limit_gb = user_dict["total_traffic_limit"] / (1024**3)
                    else:  # Иначе это уже ГБ
                        traffic_limit_gb = user_dict["total_traffic_limit"]
                    
                    traffic_percentage = (traffic_used_gb / traffic_limit_gb) * 100 if traffic_limit_gb > 0 else 0
                
                # Расчет дней до окончания
                days_until_expiry = None
                expiry_status = "Нет данных"
                if user_dict["latest_expiry"]:
                    try:
                        # PostgreSQL возвращает datetime объекты напрямую (без микросекунд)
                        if isinstance(user_dict["latest_expiry"], datetime):
                            expiry_date = user_dict["latest_expiry"]
                        else:
                            # Простой формат без микросекунд: 2025-07-09 15:33:00
                            expiry_date = datetime.strptime(str(user_dict["latest_expiry"]), "%Y-%m-%d %H:%M:%S")
                        
                        now = datetime.now()
                        days_diff = (expiry_date - now).days
                        hours_diff = (expiry_date - now).seconds // 3600
                        
                        if days_diff > 0:
                            expiry_status = f"Осталось {days_diff} дн."
                        elif days_diff == 0 and hours_diff > 0:
                            expiry_status = f"Истекает сегодня (через {hours_diff}ч)"
                        elif days_diff == 0:
                            expiry_status = "Истекает сегодня"
                        else:
                            expiry_status = f"Истекла {abs(days_diff)} дн. назад"
                        
                        days_until_expiry = days_diff
                        
                    except Exception as e:
                        logger.error(f"Ошибка обработки даты для пользователя {user_dict['user_id']}: {e}")
                        expiry_status = "Ошибка даты"
                
                # Добавление полей для совместимости с шаблоном
                user_dict.update({
                    "traffic_used_gb": round(traffic_used_gb, 2),
                    "traffic_limit_gb": int(round(traffic_limit_gb)) if traffic_limit_gb > 0 else 0,
                    "traffic_percentage": round(traffic_percentage, 1),
                    "latest_subscription_id": active_plan_name,
                    "active_plan_name": active_plan_name,
                    "plan_status": user_dict.get("active_plan_status"),
                    "subscriptions": [],
                    "recent_payments": [],
                    "days_until_expiry": days_until_expiry,
                    "expiry_status": expiry_status,
                    "user_created_at": user_dict["created_at"],
                    "total_paid_formatted": int(user_dict["total_paid"]) if user_dict["total_paid"] == int(user_dict["total_paid"]) else user_dict["total_paid"]
                })
                
                users_list.append(user_dict)

        # Создание ответа
        response = make_response(
            render_template(
                "users_restored.html",
                users_data=users_list,
                current_time=datetime.now().strftime("%d.%m.%Y %H:%M:%S"),
                current_page=page,
                total_pages=total_pages,
                total_users=total_users,
                per_page=per_page,
                has_prev=page > 1,
                has_next=page < total_pages,
                prev_page=page - 1 if page > 1 else None,
                next_page=page + 1 if page < total_pages else None,
            )
        )

        # Отключение кеширования
        response.headers["Cache-Control"] = "no-cache, no-store, must-revalidate"
        response.headers["Pragma"] = "no-cache"
        response.headers["Expires"] = "0"

        return response

    except Exception as e:
        logger.error(f"Ошибка получения данных из PostgreSQL: {e}")
        return render_template(
            "users_restored.html",
            users_data=[],
            current_time=datetime.now().strftime("%d.%m.%Y %H:%M:%S"),
            current_page=1,
            total_pages=1,
            total_users=0,
            per_page=per_page,
            has_prev=False,
            has_next=False,
            prev_page=None,
            next_page=None,
        )


@app.route("/notifications")
def notifications():
    """Расширенная страница управления тарифами, сервисами и уведомлениями"""
    from database.sqlite_config import config_db
    
    # Получаем данные для управления
    try:
        # Тарифы из SQLite
        tariffs = config_db.get_all_tariffs()
        
        # Дополнительные услуги
        services = config_db.get_all_services()
        
        # История уведомлений
        admin_notifications = config_db.get_notifications('admin')
        promo_notifications = config_db.get_notifications('promo')
        
        # Количество пользователей для рассылки
        users_count = 0
        try:
            # Прямое подключение к PostgreSQL для подсчета пользователей
            from sqlalchemy import create_engine, text
            import os
            
            engine = create_engine(os.environ.get("DATABASE_URL"))
            with engine.connect() as conn:
                result = conn.execute(text("SELECT COUNT(*) FROM users"))
                users_count = result.scalar() or 0
        except Exception as e:
            logger.error(f"Ошибка подсчета пользователей: {e}")
            users_count = 0
            
    except Exception as e:
        logger.error(f"Ошибка загрузки данных управления: {e}")
        tariffs = []
        services = []
        admin_notifications = []
        promo_notifications = []
        users_count = 0

    return render_template("notifications_extended.html", 
                         tariffs=tariffs,
                         services=services,
                         admin_notifications=admin_notifications,
                         promo_notifications=promo_notifications,
                         users_count=users_count)


@app.route("/notifications/settings/<plan_id>", methods=["GET", "POST"])
def notification_settings(plan_id):
    """Настройки уведомлений для конкретного тарифа"""
    from config import NOTIFICATION_SETTINGS, DATABASE_FILE
    import sqlite3

    if request.method == "POST":
        # Получаем данные из формы
        days_before = request.form.getlist("days_before")
        days_before = [int(d) for d in days_before if d.isdigit()]

        send_time_start = int(request.form.get("send_time_start", 9))
        send_time_end = int(request.form.get("send_time_end", 21))
        check_interval_hours = int(request.form.get("check_interval_hours", 1))
        enabled = "enabled" in request.form

        # Сохраняем настройки в SQLite базу
        try:
            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()

            # Удаляем старые настройки и вставляем новые
            cursor.execute("DELETE FROM notification_settings WHERE plan_id = ?", (plan_id,))
            cursor.execute(
                """
                INSERT INTO notification_settings 
                (plan_id, plan_name, days_before, send_time_start, send_time_end, check_interval_hours, enabled)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """,
                (
                    plan_id,
                    "Тестовые периоды" if plan_id == "trial" else "Оплаченные подписки",
                    str(days_before),
                    send_time_start,
                    send_time_end,
                    check_interval_hours,
                    enabled,
                ),
            )
            conn.commit()
            conn.close()

            flash(f"Настройки уведомлений для тарифа {plan_id} обновлены", "success")
        except Exception as e:
            flash(f"Ошибка сохранения настроек: {e}", "error")

        return redirect(url_for("notifications"))

    # Получаем настройки из PostgreSQL или SQLite
    setting = get_notification_settings_postgres(plan_id)
    
    if not setting:
        # Fallback к SQLite
        try:
            import sqlite3
            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()
            cursor.execute("SELECT * FROM notification_settings WHERE plan_id = ?", (plan_id,))
            row = cursor.fetchone()

            if row:
                import json
                try:
                    days_before = json.loads(row[3]) if row[3] else []
                except:
                    days_before = []

                setting = {
                    "plan_id": row[1],
                    "plan_name": row[2],
                    "days_before": days_before,
                    "send_time_start": row[4],
                    "send_time_end": row[5],
                    "check_interval_hours": row[6],
                    "enabled": bool(row[7]),
                }
            conn.close()
        except Exception as e:
            logger.error(f"Ошибка получения настроек из SQLite: {e}")

    # Если нет настроек в базе, используем настройки по умолчанию
    if not setting and plan_id in NOTIFICATION_SETTINGS:
        config = NOTIFICATION_SETTINGS[plan_id]
        setting = {
            "plan_id": plan_id,
            "plan_name": "Тестовые периоды" if plan_id == "trial" else "Оплаченные подписки",
            "days_before": config["days_before"],
            "send_time_start": config["send_time_start"],
            "send_time_end": config["send_time_end"],
            "check_interval_hours": config["check_interval_hours"],
            "enabled": True,
        }

    return render_template("notification_settings.html", plan_id=plan_id, setting=setting)


@app.route("/notifications/send-test", methods=["POST"])
def send_test_notification():
    """Отправка тестового уведомления"""
    user_id = request.form.get("user_id")

    if not user_id or not user_id.isdigit():
        flash("Неверный ID пользователя", "error")
        return redirect(url_for("notifications"))

    # Используем синхронную версию отправки уведомления
    success = send_sync_test_notification(int(user_id))
    
    if success:
        flash(f"Тестовое уведомление записано для пользователя {user_id}", "success")
    else:
        flash(f"Ошибка отправки тестового уведомления пользователю {user_id}", "error")
        
    return redirect(url_for("notifications"))




@app.route("/scheduler")
def scheduler():
    """Страница мониторинга планировщика уведомлений"""
    import sqlite3
    from config import DATABASE_FILE
    from datetime import datetime, timedelta

    # Получаем состояние планировщика
    scheduler_status = get_scheduler_status()

    # Получаем предстоящие уведомления
    upcoming_notifications = get_upcoming_notifications()

    # Получаем статистику уведомлений за последние 24 часа
    from database.sqlite_config import config_db
    
    stats_24h = []
    try:
        with sqlite3.connect(config_db.db_path) as conn:
            cursor = conn.cursor()
            
            # Статистика за последние 24 часа
            yesterday = (datetime.now() - timedelta(days=1)).isoformat()
            cursor.execute(
                """
                SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN success = 1 THEN 1 ELSE 0 END) as successful,
                    SUM(CASE WHEN success = 0 THEN 1 ELSE 0 END) as failed,
                    notification_type
                FROM notification_log 
                WHERE sent_at > ?
                GROUP BY notification_type
                """,
                (yesterday,),
            )
            stats_24h = cursor.fetchall()
    except Exception as e:
        logger.error(f"Ошибка получения статистики уведомлений: {e}")
        stats_24h = []

    # Последние логи
    recent_logs = []
    try:
        with sqlite3.connect(config_db.db_path) as conn:
            cursor = conn.cursor()
            cursor.execute(
                """
                SELECT user_id, notification_type, message, 
                       success, error_message, sent_at 
                FROM notification_log 
                ORDER BY sent_at DESC 
                LIMIT 20
                """
            )
            recent_logs = cursor.fetchall()
    except Exception as e:
        logger.error(f"Ошибка получения логов уведомлений: {e}")
        recent_logs = []

    response = make_response(render_template(
        "scheduler.html",
        scheduler_status=scheduler_status,
        upcoming_notifications=upcoming_notifications,
        stats_24h=stats_24h,
        recent_logs=recent_logs,
    ))
    
    # Предотвращаем кэширование для актуальных данных
    response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    response.headers['Pragma'] = 'no-cache'
    response.headers['Expires'] = '0'
    
    return response


def get_scheduler_status():
    """Получить статус планировщика задач"""
    try:
        from datetime import datetime, timedelta
        import os
        import sqlite3
        from config import DATABASE_FILE

        status = {"is_running": False, "active_jobs": [], "last_check": None, "uptime": None}

        # Проверяем наличие недавних логов планировщика в базе данных
        try:
            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()

            # Проверяем любые недавние уведомления (за последние 20 минут)
            recent_time = (datetime.now() - timedelta(minutes=20)).isoformat()
            cursor.execute(
                """
                SELECT COUNT(*) FROM notification_log 
                WHERE sent_at > ?
            """,
                (recent_time,),
            )

            recent_activity = cursor.fetchone()[0]
            
            # Также проверяем недавние успешные отправки уведомлений
            cursor.execute(
                """
                SELECT COUNT(*) FROM notification_log 
                WHERE sent_at > ? AND success = 1
            """,
                (recent_time,),
            )
            
            recent_successful_notifications = cursor.fetchone()[0]
            conn.close()

            # Если есть любая недавняя активность уведомлений, планировщик работает
            if recent_activity > 0 or recent_successful_notifications > 0:
                status["is_running"] = True

        except Exception as db_error:
            logger.error(f"Ошибка проверки базы данных: {db_error}")

        # Проверяем процесс бота как основной индикатор
        try:
            import subprocess
            result = subprocess.run(["pgrep", "-f", "main.py"], capture_output=True, text=True)
            if result.returncode == 0 and result.stdout.strip():
                status["is_running"] = True
                logger.info("Планировщик активен: найден процесс main.py")
        except Exception as proc_error:
            logger.error(f"Ошибка проверки процесса: {proc_error}")

        # Дополнительная проверка через workflow статус
        try:
            import subprocess
            result = subprocess.run(["pgrep", "-f", "bot_execution"], capture_output=True, text=True)
            if result.returncode == 0:
                status["is_running"] = True
                logger.info("Планировщик активен: найден workflow bot_execution")
        except:
            pass

        # Если планировщик работает, определяем активные задачи
        if status["is_running"]:
            status["active_jobs"] = [
                "light_sync",
                "scheduler_health_check", 
                "trial_hourly_check",
                "traffic_check",
                "regular_daily_check",
            ]
            status["uptime"] = "Активен"

        status["last_check"] = datetime.now()
        return status

    except Exception as e:
        from datetime import datetime

        return {
            "is_running": False,
            "active_jobs": [],
            "last_check": datetime.now(),
            "error": str(e),
        }


def get_upcoming_notifications():
    """Получить предстоящие уведомления из PostgreSQL"""
    import json
    from datetime import datetime, timedelta
    from sqlalchemy import create_engine, text
    import os

    upcoming = []

    try:
        # Подключение к PostgreSQL
        database_url = os.environ.get("DATABASE_URL")
        if not database_url:
            logger.info("DATABASE_URL не найден")
            return upcoming
            
        engine = create_engine(database_url)

        with engine.connect() as conn:
            # Получаем все активные подписки с настройками уведомлений
            query = text("""
                SELECT 
                    u.id as user_id, 
                    u.username, 
                    s.id as subscription_id, 
                    s.expires_at, 
                    s.plan_id,
                    COALESCE(ns.days_before, '[1,3]') as days_before,
                    COALESCE(ns.send_time_start, 9) as start_time,
                    COALESCE(ns.send_time_end, 21) as end_time,
                    COALESCE(ns.check_interval_hours, 24) as interval_hours,
                    COALESCE(ns.plan_name, s.plan_id) as plan_name
                FROM users u
                JOIN subscriptions s ON u.id = s.user_id
                LEFT JOIN notification_settings ns ON 
                    CASE 
                        WHEN s.plan_id = 'trial' THEN ns.plan_id = 'trial'
                        ELSE ns.plan_id = 'paid'
                    END
                WHERE s.expires_at > NOW() 
                AND UPPER(s.status) = 'ACTIVE'
                AND (ns.enabled = true OR ns.enabled IS NULL)
                ORDER BY s.expires_at ASC
                LIMIT 50
            """)

            result = conn.execute(query)
            subscriptions = result.fetchall()

            for sub in subscriptions:
                user_id = sub.user_id
                username = sub.username
                sub_id = sub.subscription_id
                expires_at = sub.expires_at
                plan_type = sub.plan_id
                days_before_json = sub.days_before
                start_time = sub.start_time
                end_time = sub.end_time
                interval_hours = sub.interval_hours
                plan_name = sub.plan_name

                try:
                    # PostgreSQL возвращает datetime объекты напрямую
                    if isinstance(expires_at, datetime):
                        expires_date = expires_at
                    else:
                        # Если по какой-то причине пришла строка
                        expires_date = datetime.fromisoformat(str(expires_at))

                    days_until_expiry = (expires_date - datetime.now()).days

                    # Обрабатываем настройки дней уведомлений
                    if days_before_json:
                        try:
                            if isinstance(days_before_json, str):
                                notification_days = json.loads(days_before_json)
                            else:
                                notification_days = days_before_json
                        except:
                            notification_days = [1, 3] if plan_type != "trial" else [0, 1, 2, 3]
                    else:
                        notification_days = [1, 3] if plan_type != "trial" else [0, 1, 2, 3]

                    # Определяем частоту уведомлений
                    if plan_type == "trial":
                        frequency = f"каждый час с {start_time}:00 до {end_time}:00"
                    else:
                        frequency = f"ежедневно в {start_time}:00 МСК"

                    # Показываем предстоящие уведомления для каждого настроенного дня
                    for days_before in notification_days:
                        if days_until_expiry >= days_before:
                            # Определяем дату следующего уведомления
                            target_date = expires_date - timedelta(days=days_before)

                            # Для тестовых подписок - проверяем близость к дате уведомления
                            if plan_type == "trial":
                                if days_until_expiry == days_before:
                                    next_notification = "В ближайший час"
                                else:
                                    next_notification = f"{target_date.strftime('%d.%m')} с {start_time}:00"
                            else:
                                next_notification = f"{target_date.strftime('%d.%m')} в {start_time}:00"

                            notification_text = f"За {days_before} дн." if days_before > 0 else "В день истечения"

                            upcoming.append({
                                "user_id": user_id,
                                "username": username or f"ID{user_id}",
                                "subscription_id": sub_id,
                                "expires_at": expires_date.strftime("%d.%m.%Y %H:%M"),
                                "days_until_expiry": days_until_expiry,
                                "plan_type": plan_type,
                                "plan_name": plan_name or plan_type.title(),
                                "notification_type": notification_text,
                                "next_notification": next_notification,
                                "frequency": frequency,
                                "days_before": days_before,
                            })
                except Exception as date_error:
                    logger.error(f"Ошибка обработки даты для подписки {sub_id}: {date_error}")
                    continue

        # Сортируем по дате ближайшего уведомления
        upcoming.sort(key=lambda x: (x["days_until_expiry"], x["days_before"]))

    except Exception as e:
        logger.error(f"Ошибка получения предстоящих уведомлений из PostgreSQL: {e}")

    return upcoming


@app.route("/api/traffic-check/status")
def get_traffic_check_status():
    """Получить статус задачи проверки трафика"""
    try:
        from sqlalchemy import create_engine, text
        import os
        
        # Используем PostgreSQL
        database_url = os.environ.get("DATABASE_URL")
        engine = create_engine(database_url)
        
        with engine.connect() as conn:
            # Создаем таблицу если не существует
            conn.execute(text("""
                CREATE TABLE IF NOT EXISTS app_settings (
                    key VARCHAR(255) PRIMARY KEY,
                    value TEXT,
                    description TEXT,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """))
            
            result = conn.execute(text("SELECT value FROM app_settings WHERE key = 'traffic_check_enabled'"))
            row = result.fetchone()
        
        if row:
            enabled = row[0].lower() in ['true', '1', 'yes', 'on']
        else:
            enabled = True  # По умолчанию включено
            
        return jsonify({
            "enabled": enabled,
            "status": "success"
        })
    except Exception as e:
        return jsonify({
            "enabled": True,
            "status": "error",
            "message": str(e)
        }), 500


@app.route("/api/traffic-check/toggle", methods=["POST"])
def toggle_traffic_check():
    """Включить/отключить задачу проверки трафика"""
    try:
        from datetime import datetime
        from sqlalchemy import create_engine, text
        import os
        
        data = request.get_json()
        enabled = data.get('enabled', True)
        
        # Используем PostgreSQL
        database_url = os.environ.get("DATABASE_URL")
        engine = create_engine(database_url)
        
        with engine.connect() as conn:
            # Создаем таблицу если не существует
            conn.execute(text("""
                CREATE TABLE IF NOT EXISTS app_settings (
                    key VARCHAR(255) PRIMARY KEY,
                    value TEXT,
                    description TEXT,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            """))
            
            # Обновляем или создаем настройку
            conn.execute(text("""
                INSERT INTO app_settings (key, value, description, updated_at) 
                VALUES (:key, :value, :description, :updated_at)
                ON CONFLICT (key) DO UPDATE SET 
                    value = EXCLUDED.value,
                    updated_at = EXCLUDED.updated_at
            """), {
                'key': 'traffic_check_enabled',
                'value': 'true' if enabled else 'false',
                'description': 'Включение/отключение автоматической проверки низкого трафика',
                'updated_at': datetime.now()
            })
            
            conn.commit()
        
        return jsonify({
            "enabled": enabled,
            "status": "success",
            "message": f"Задача проверки трафика {'включена' if enabled else 'отключена'}"
        })
    except Exception as e:
        return jsonify({
            "enabled": False,
            "status": "error", 
            "message": str(e)
        }), 500


@app.route("/yookassa-webhook", methods=["POST"])
def yookassa_webhook():
    """
    Webhook endpoint для обработки уведомлений от YooKassa
    Реализует правильную обработку HTTP-кодов согласно документации YooKassa
    """
    import os
    import traceback
    import asyncio
    from datetime import datetime
    from sqlalchemy import text

    # Логируем все входящие webhook'ы для диагностики
    request_data = request.get_json()
    client_ip = request.headers.get('X-Forwarded-For', request.remote_addr)
    user_agent = request.headers.get('User-Agent', 'Unknown')
    
    logger.info(f"🔔 WEBHOOK ПОЛУЧЕН от {client_ip} (UA: {user_agent[:50]})")
    logger.info(f"📦 Данные webhook: {request_data}")
    
    # Проверяем, что это запрос от YooKassa
    if 'YooKassa' not in user_agent and 'yookassa' not in user_agent.lower():
        logger.warning(f"⚠️ Подозрительный webhook от {client_ip} - не YooKassa")
        # Но все равно обрабатываем на случай изменения User-Agent

    try:
        # Получаем данные от YooKassa
        notification_data = request.get_json()

        # Логируем входящий webhook
        timestamp = datetime.now().isoformat()
        logger.info(f"[{timestamp}] YooKassa Webhook получен: {notification_data}")

        if not notification_data:
            logger.error(f"[{timestamp}] ОШИБКА: Пустые данные от YooKassa")
            # Согласно документации YooKassa - возвращаем 400 для некорректного запроса
            return jsonify({"error": "Invalid request: no data received"}), 400

        # Извлекаем данные о платеже
        event = notification_data.get("event")
        payment_data = notification_data.get("object", {})
        payment_id = payment_data.get("id")
        status = payment_data.get("status")
        metadata = payment_data.get("metadata", {})
        amount_data = payment_data.get("amount", {})

        logger.info(f"[{timestamp}] Событие: {event}")
        logger.info(f"[{timestamp}] Статус платежа: {status}")
        logger.info(f"[{timestamp}] ID платежа: {payment_id}")
        logger.info(f"[{timestamp}] Metadata: {metadata}")
        logger.info(f"[{timestamp}] Сумма: {amount_data}")

        if not payment_id:
            logger.error(f"[timestamp] ОШИБКА: Отсутствует ID платежа")
            return jsonify({"error": "Payment ID required"}), 400

        # Обрабатываем только события успешных платежей
        if event == "payment.succeeded" and status == "succeeded":
            logger.info(f"[{timestamp}] Обработка успешного платежа {payment_id}")

            # Ищем платеж в базе данных
            try:
                # Создаем новое подключение для webhook с правильным управлением транзакциями
                engine = create_engine(os.environ.get('DATABASE_URL'), pool_pre_ping=True)
                
                logger.info(f"[{timestamp}] Подключаемся к базе данных...")
                
                # Получаем данные о платеже в отдельной транзакции
                with engine.connect() as conn:
                    # Ищем конкретный платеж
                    result = conn.execute(
                        text("SELECT id, user_id, subscription_id, plan_id, status FROM payments WHERE payment_id = :payment_id"),
                        {"payment_id": payment_id}
                    )
                    payment_record = result.fetchone()
                    
                    if payment_record:
                        payment_db_id, user_id, subscription_id, plan_id, current_status = payment_record
                        logger.info(f"[{timestamp}] Найден платеж в БД: user_id={user_id}, subscription_id={subscription_id}, plan_id={plan_id}")
                    else:
                        logger.info(f"[{timestamp}] Платеж {payment_id} не найден в БД")
                        # Извлекаем данные из metadata webhook
                        metadata = notification_data.get("object", {}).get("metadata", {})
                        user_id = metadata.get("user_id")
                        subscription_id = metadata.get("subscription_id") 
                        plan_id = metadata.get("plan_id")
                        amount_value = notification_data.get("object", {}).get("amount", {}).get("value", "0.00")
                        
                        if user_id:
                            logger.info(f"[{timestamp}] Создаем новый платеж из webhook: user_id={user_id}, plan_id={plan_id}")
                            
                            # Найдем актуальную подписку пользователя вместо использования metadata
                            subscription_result = conn.execute(
                                text("SELECT id FROM subscriptions WHERE user_id = :user_id ORDER BY created_at DESC LIMIT 1"),
                                {"user_id": int(user_id)}
                            )
                            subscription_record = subscription_result.fetchone()
                            actual_subscription_id = subscription_record[0] if subscription_record else None
                            
                            logger.info(f"[{timestamp}] Найдена актуальная подписка: subscription_id={actual_subscription_id}")
                            
                            # Создаем запись платежа из webhook данных с московским временем
                            from datetime import timezone, timedelta
                            moscow_tz = timezone(timedelta(hours=3))
                            now_moscow = datetime.now(moscow_tz).isoformat()
                            conn.execute(
                                text("""INSERT INTO payments (payment_id, user_id, subscription_id, amount, plan_id, status, created_at, paid_at, completed_at) 
                                        VALUES (:payment_id, :user_id, :subscription_id, :amount, :plan_id, 'SUCCEEDED', :created_at, :paid_at, :completed_at)"""),
                                {
                                    "payment_id": payment_id,
                                    "user_id": int(user_id),
                                    "subscription_id": actual_subscription_id,
                                    "amount": float(amount_value),
                                    "plan_id": plan_id,
                                    "created_at": now_moscow,
                                    "paid_at": now_moscow,
                                    "completed_at": now_moscow
                                }
                            )
                            conn.commit()
                            payment_db_id = None
                            current_status = "PENDING"
                            subscription_id = actual_subscription_id
                        else:
                            logger.error(f"[{timestamp}] Не найден user_id в webhook metadata")
                            return jsonify({"status": "missing_user_id", "message": "User ID not found in webhook"}), 200

                # Проверяем, не был ли платеж уже обработан (учитываем формат статуса)
                if payment_record and current_status in ["SUCCEEDED", "2"]:
                    logger.info(f"[{timestamp}] ИНФОРМАЦИЯ: Платеж {payment_id} уже был обработан ранее")
                    return jsonify({"status": "already_processed", "message": f"Payment {payment_id} already processed"}), 200

                logger.info(
                    f"[{timestamp}] Обрабатываем платеж: user_id={user_id}, subscription_id={subscription_id}, plan_id={plan_id}"
                )

                # Обновляем статус платежа в отдельной транзакции
                with engine.connect() as conn:
                    # Обновляем статус платежа на SUCCEEDED с заполнением paid_at (московское время)
                    from datetime import timezone, timedelta
                    moscow_tz = timezone(timedelta(hours=3))
                    now_moscow = datetime.now(moscow_tz)
                    conn.execute(
                        text("UPDATE payments SET status = 'SUCCEEDED', paid_at = :paid_at, completed_at = :completed_at WHERE payment_id = :payment_id"),
                        {"paid_at": now_moscow.isoformat(), "completed_at": now_moscow.isoformat(), "payment_id": payment_id}
                    )
                    conn.commit()

                    logger.info(f"[{timestamp}] Статус платежа {payment_id} обновлен на SUCCEEDED")

                # Автоматически обрабатываем платеж и обновляем подписку
                subscription_id_int = int(subscription_id) if subscription_id is not None else None
                
                # Получаем сумму платежа из данных webhook
                amount_data = notification_data.get("object", {}).get("amount", {})
                
                # Вызываем функцию обработки успешного платежа
                success = process_successful_payment_sync(
                    user_id=int(user_id),
                    subscription_id=subscription_id_int,
                    payment_id=payment_id,
                    plan_id=plan_id,
                    amount=amount_data.get("value", "0")
                )

            except Exception as db_error:
                logger.error(f"[{timestamp}] Ошибка подключения к БД: {db_error}")
                logger.error(f"[{timestamp}] Traceback: {traceback.format_exc()}")
                return jsonify({"error": "Database connection error"}), 500

            # Проверяем результат обработки
            try:

                if success:
                    logger.info(f"[{timestamp}] ✅ Платеж {payment_id} успешно обработан")
                    return (
                        jsonify(
                            {
                                "status": "success",
                                "message": f"Payment {payment_id} processed successfully",
                            }
                        ),
                        200,
                    )
                else:
                    logger.error(f"[{timestamp}] ❌ Ошибка при обработке платежа {payment_id}")
                    # Возвращаем 500 чтобы YooKassa повторила запрос
                    return jsonify({"error": "Payment processing failed"}), 500

            except Exception as processing_error:
                logger.info(f"[{timestamp}] ОШИБКА обработки платежа: {processing_error}")
                logger.info(f"[{timestamp}] Трассировка: {traceback.format_exc()}")
                # Возвращаем 500 для повторной попытки от YooKassa
                return jsonify({"error": "Payment processing exception"}), 500

        elif event == "payment.canceled":
            logger.info(f"[{timestamp}] Платеж {payment_id} отменен")
            # Обновляем статус в базе данных и получаем данные пользователя
            with db.engine.connect() as conn:
                # Обновляем статус платежа
                conn.execute(
                    text("UPDATE payments SET status = 'CANCELED', completed_at = :completed_at WHERE payment_id = :payment_id"),
                    {"completed_at": datetime.now().isoformat(), "payment_id": payment_id}
                )
                
                # Получаем данные пользователя для отправки уведомления
                result = conn.execute(
                    text("""
                        SELECT p.user_id, p.amount, p.subscription_id, p.plan_id 
                        FROM payments p 
                        WHERE p.payment_id = :payment_id
                    """),
                    {"payment_id": payment_id}
                )
                payment_data = result.fetchone()
                
                conn.commit()

            if payment_data:
                user_id, amount, subscription_id, plan_id = payment_data
                # Отправляем уведомление о неуспешном платеже в отдельном потоке
                import threading

                def send_notification():
                    try:
                        # Отправляем уведомление через Telegram API
                        import os
                        import requests
                        
                        bot_token = os.environ.get('BOT_TOKEN')
                        if bot_token:
                            message = f'''❌ Платеж отменен
                            
💳 ID платежа: {payment_id}
💰 Сумма: {amount} RUB
📋 Тариф: {plan_id}
❗ Причина: Недостаточно средств на карте

🔄 Попробуйте оплатить еще раз или используйте другую карту.'''
                            
                            try:
                                response = requests.post(
                                    f'https://api.telegram.org/bot{bot_token}/sendMessage',
                                    json={
                                        'chat_id': user_id,
                                        'text': message,
                                        'parse_mode': 'HTML'
                                    },
                                    timeout=10
                                )
                                if response.status_code == 200:
                                    logger.info(f"✅ Уведомление об отмене отправлено пользователю {user_id}")
                                else:
                                    logger.info(f"❌ Ошибка отправки уведомления об отмене: {response.status_code}")
                            except Exception as telegram_error:
                                logger.info(f"❌ Ошибка Telegram API: {telegram_error}")
                        
                        # Записываем уведомление в лог базы данных
                        conn = sqlite3.connect(DATABASE_FILE)
                        cursor = conn.cursor()
                        cursor.execute("""
                            INSERT INTO notification_log 
                            (user_id, subscription_id, notification_type, message_text, sent_at, success)
                            VALUES (?, ?, ?, ?, ?, 1)
                        """, (user_id, subscription_id or 0, 'payment_canceled', 
                              f'Платеж {payment_id} отменен на сумму {amount}', 
                              datetime.now().isoformat()))
                        conn.commit()
                        conn.close()
                        logger.info(f"Уведомление о неуспешном платеже записано для пользователя {user_id}")
                    except Exception as e:
                        logger.error(f"Ошибка записи уведомления о неуспешном платеже: {e}")

                thread = threading.Thread(target=send_notification)
                thread.daemon = True
                thread.start()

            return jsonify({"status": "canceled_processed"}), 200

        elif event == "payment.failed":
            logger.info(f"[{timestamp}] Платеж {payment_id} неуспешен")

            # Извлекаем причину неуспеха из cancellation_details
            cancellation_reason = "Неизвестная ошибка"
            if "cancellation_details" in payment_data:
                cancellation_details = payment_data["cancellation_details"]
                if cancellation_details.get("reason"):
                    cancellation_reason = get_payment_error_description(
                        cancellation_details["reason"]
                    )

            # Обновляем статус в базе данных
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()
            cursor.execute(
                "UPDATE payments SET status = 'FAILED', completed_at = ? WHERE payment_id = ?",
                (datetime.now().isoformat(), payment_id),
            )

            # Получаем данные пользователя для отправки уведомления
            cursor.execute(
                """
                SELECT p.user_id, p.amount, p.subscription_id, p.plan_id 
                FROM payments p 
                WHERE p.payment_id = ?
            """,
                (payment_id,),
            )
            payment_data = cursor.fetchone()

            conn.commit()
            conn.close()

            if payment_data:
                user_id, amount, subscription_id, plan_id = payment_data
                # Отправляем уведомление о неуспешном платеже в отдельном потоке
                import threading

                def send_notification():
                    try:
                        # Записываем уведомление в лог базы данных
                        conn = sqlite3.connect(DATABASE_FILE)
                        cursor = conn.cursor()
                        cursor.execute("""
                            INSERT INTO notification_log 
                            (user_id, subscription_id, notification_type, message_text, sent_at, success)
                            VALUES (?, ?, ?, ?, ?, 1)
                        """, (user_id, subscription_id or 0, 'payment_failed', 
                              f'Платеж {payment_id} неуспешен: {cancellation_reason}, сумма {amount}', 
                              datetime.now().isoformat()))
                        conn.commit()
                        conn.close()
                        logger.info(f"Уведомление о неуспешном платеже записано для пользователя {user_id}")
                    except Exception as e:
                        logger.error(f"Ошибка записи уведомления о неуспешном платеже: {e}")

                thread = threading.Thread(target=send_notification)
                thread.daemon = True
                thread.start()

            return jsonify({"status": "failed_processed"}), 200

        elif event == "payment.waiting_for_capture":
            logger.info(f"[{timestamp}] Платеж {payment_id} ожидает подтверждения")
            return jsonify({"status": "waiting_processed"}), 200

        else:
            logger.info(f"[{timestamp}] Неизвестное событие или статус: {event}/{status}")
            # Возвращаем 200 для неизвестных событий чтобы не засорять логи YooKassa
            return jsonify({"status": "event_ignored", "event": event, "status": status}), 200

    except Exception as e:
        error_trace = traceback.format_exc()
        timestamp = datetime.now().isoformat()
        logger.critical(f"[{timestamp}] КРИТИЧЕСКАЯ ОШИБКА webhook YooKassa: {e}")
        logger.info(f"[{timestamp}] Полная трассировка: {error_trace}")

        # Согласно документации YooKassa - возвращаем 500 для серверных ошибок
        # YooKassa будет повторять запрос при получении 5xx кодов
        return jsonify({"error": "Internal server error", "message": str(e)}), 500


def update_expired_subscription_statuses():
    """Обновляет статусы истекших подписок на 'Не продлена'"""
    try:
        from sqlalchemy import text
        from datetime import datetime
        
        with db.engine.connect() as conn:
            # Обновляем статус подписок:
            # 1. Истекшие подписки -> 'Не продлена'
            result1 = conn.execute(
                text("""
                    UPDATE subscriptions 
                    SET plan_status = 'Не продлена'
                    WHERE expires_at < NOW() 
                    AND status = 'ACTIVE'
                    AND (plan_status IS NULL OR plan_status != 'Не продлена')
                """)
            )
            
            # 2. Активные (не истекшие) подписки -> 'Оплачена'  
            result2 = conn.execute(
                text("""
                    UPDATE subscriptions 
                    SET plan_status = 'Оплачена'
                    WHERE expires_at >= NOW() 
                    AND status = 'ACTIVE'
                    AND plan_status IS NULL
                """)
            )
            conn.commit()
            
            expired_count = result1.rowcount
            paid_count = result2.rowcount
            
            if expired_count > 0:
                logger.info(f"Обновлено {expired_count} истекших подписок на статус 'Не продлена'")
            if paid_count > 0:
                logger.info(f"Обновлено {paid_count} активных подписок на статус 'Оплачена'")
    except Exception as e:
        logger.error(f"Ошибка обновления статусов истекших подписок: {e}")

def process_successful_payment_sync(
    user_id: int, subscription_id: int | None, payment_id: str, plan_id: str, amount: str
):
    """Обработка успешного платежа и автоматическое обновление подписки"""
    try:
        # Импортируем нужные модули
        import sys
        import os
        from datetime import datetime, timedelta
        from pytz import timezone
        from sqlalchemy import text

        # Добавляем путь к модулям бота
        sys.path.append(os.path.join(os.path.dirname(__file__)))

        from config import SUBSCRIPTION_PLANS
        from bot.utils.date_validation import validate_subscription_dates, log_date_calculation

        logger.info(
            f"Обработка платежа {payment_id} для пользователя {user_id}, подписка {subscription_id}"
        )

        # Получаем информацию о тарифе
        plan_info = SUBSCRIPTION_PLANS.get(plan_id)
        if not plan_info:
            logger.info(f"Тариф {plan_id} не найден в конфигурации")
            return False

        # Получаем текущую подписку для накопления времени и лимитов через PostgreSQL
        current_subscription = None
        if subscription_id is not None:
            with db.engine.connect() as conn:
                result = conn.execute(
                    text("SELECT expires_at, traffic_limit FROM subscriptions WHERE id = :subscription_id"),
                    {"subscription_id": subscription_id}
                )
                current_subscription = result.fetchone()
        else:
            logger.info(f"Создание новой подписки для пользователя {user_id} (subscription_id=None)")

        # Используем московское время (UTC+3) без timezone для consistency
        from pytz import timezone

        moscow_tz = timezone("Europe/Moscow")
        now_moscow = datetime.now(moscow_tz).replace(tzinfo=None)  # Убираем timezone info для простоты

        if current_subscription:
            current_expiry_str, current_traffic_limit = current_subscription

            # Парсим текущую дату окончания
            if current_expiry_str:
                try:
                    # Пробуем разные форматы даты и приводим к naive datetime
                    if "T" in current_expiry_str:
                        current_expiry = datetime.fromisoformat(
                            current_expiry_str.replace("Z", "").replace("+00:00", "")
                        )
                    else:
                        current_expiry = datetime.strptime(current_expiry_str, "%Y-%m-%d %H:%M:%S")
                    
                    # Убираем timezone info если есть
                    if current_expiry.tzinfo is not None:
                        current_expiry = current_expiry.replace(tzinfo=None)

                    logger.info(f"DEBUG: Текущая дата истечения: {current_expiry}")
                    logger.info(f"DEBUG: Текущая дата (МСК): {now_moscow}")

                    # ПРАВИЛЬНАЯ ЛОГИКА: 
                    # 1. Если подписка активна (дата истечения > даты оплаты) - добавляем к дате истечения
                    # 2. Если подписка истекла (дата истечения <= даты оплаты) - добавляем к дате оплаты
                    if current_expiry > now_moscow:
                        # Подписка еще активна - добавляем к дате истечения для накопления периода
                        new_expiry_date = current_expiry + timedelta(days=plan_info["duration_days"])
                        logger.info(f"DEBUG: Подписка активна до {current_expiry.strftime('%d.%m.%Y %H:%M')}, добавляем {plan_info['duration_days']} дней к дате истечения")
                    else:
                        # Подписка истекла - добавляем к дате оплаты для корректного восстановления
                        new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])
                        logger.info(f"DEBUG: Подписка истекла {current_expiry.strftime('%d.%m.%Y %H:%M')}, добавляем {plan_info['duration_days']} дней к дате оплаты {now_moscow.strftime('%d.%m.%Y %H:%M')}")
                    
                    logger.info(f"DEBUG: Новая дата истечения: {new_expiry_date}")
                    
                    # ВАЛИДАЦИЯ: Проверяем корректность расчета даты (мягкая проверка)
                    try:
                        is_valid = validate_subscription_dates(
                            payment_date=now_moscow,
                            plan_duration_days=plan_info["duration_days"],
                            calculated_expiry=new_expiry_date,
                            user_id=user_id,
                            plan_id=plan_id
                        )
                        
                        if not is_valid:
                            logger.info(f"ПРЕДУПРЕЖДЕНИЕ: Расхождение в расчете даты для пользователя {user_id}, но продолжаем обработку")
                    except Exception as e:
                        logger.error(f"Ошибка валидации дат: {e}, пропускаем проверку")
                    
                    # АУДИТ: Логируем расчет для отслеживания
                    log_date_calculation(
                        user_id=user_id,
                        plan_id=plan_id,
                        payment_date=now_moscow,
                        calculated_expiry=new_expiry_date,
                        is_extension=bool(current_subscription)
                    )
                    
                except Exception as e:
                    logger.info(f"DEBUG: Ошибка парсинга даты '{current_expiry_str}': {e}")
                    # Если ошибка парсинга, используем текущую дату
                    new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])
            else:
                new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])

            # Устанавливаем лимит трафика по тарифу (не накапливаем при обычном продлении)
            new_traffic_limit = plan_info["traffic_gb"]
        else:
            # Новая подписка
            new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])
            new_traffic_limit = plan_info["traffic_gb"]

        logger.info(
            f"Обновляем подписку {subscription_id}: тариф={plan_id}, до={new_expiry_date.strftime('%d.%m.%Y')}, лимит={new_traffic_limit}ГБ (накоплено)"
        )

        # Обновляем подписку в PostgreSQL с накопленными значениями
        try:
            if subscription_id is not None:
                with db.engine.connect() as conn:
                    # Выполняем обновление подписки с накопленными значениями и сбросом трафика
                    result = conn.execute(
                        text("""
                            UPDATE subscriptions 
                            SET plan_id = :plan_id, 
                                expires_at = :expires_at, 
                                traffic_limit = :traffic_limit, 
                                traffic_used = 0,
                                status = 'ACTIVE',
                                plan_status = 'Оплачена'
                            WHERE id = :subscription_id
                        """),
                        {
                            "plan_id": plan_id,
                            "expires_at": new_expiry_date.isoformat(),
                            "traffic_limit": new_traffic_limit,
                            "subscription_id": subscription_id
                        }
                    )

                    conn.commit()
                    
                    # Проверяем, что обновление прошло успешно
                    if result.rowcount > 0:
                        logger.info(f"Обновлена {result.rowcount} подписка, трафик сброшен на 0")
                        logger.info(f"Подписка {subscription_id} успешно обновлена в PostgreSQL")
                    else:
                        logger.info(f"ВНИМАНИЕ: Подписка {subscription_id} не найдена для обновления")
            else:
                logger.info(f"Платеж {payment_id} обработан, но подписка не указана (subscription_id=None)")
                # В случае subscription_id=None, просто логируем успешную обработку платежа
                return True

        except Exception as e:
            logger.error(f"Ошибка обновления подписки {subscription_id} в PostgreSQL: {e}")
            return False

        # Получаем данные подписки для обновления панели 3X-UI
        try:
            subscription_data = None
            if subscription_id is not None:
                with db.engine.connect() as conn:
                    result = conn.execute(
                        text("SELECT vpn_user_id, vpn_email FROM subscriptions WHERE id = :subscription_id"),
                        {"subscription_id": subscription_id}
                    )
                    subscription_data = result.fetchone()

            if subscription_data:
                vpn_user_id, vpn_email = subscription_data
                logger.info(f"Найдены данные клиента в панели: Email={vpn_email}")

                # Используем рабочий API для обновления клиента по email
                from bot.services.working_panel_api import WorkingPanelAPI

                # Обновляем клиента в панели через упрощенный API (только по email)
                logger.info(f"Планируется обновление клиента {vpn_email} в панели: {new_traffic_limit}ГБ до {new_expiry_date.strftime('%d.%m.%Y')}")
                
                # Используем асинхронный API для обновления
                import asyncio
                
                async def update_panel_client():
                    try:
                        panel = WorkingPanelAPI()
                        
                        # Сначала получаем данные клиента по email для получения UUID
                        client_data = await panel.get_client_by_email(vpn_email)
                        if not client_data:
                            logger.info(f"Клиент {vpn_email} не найден в панели")
                            return False
                        
                        client_uuid = client_data.get('id')
                        if not client_uuid:
                            logger.info(f"UUID клиента {vpn_email} не найден")
                            return False
                        
                        # Обновляем клиента с новыми параметрами, передавая tgId и subId
                        from datetime import datetime as dt
                        success = await panel.update_client(
                            client_uuid=str(client_uuid),
                            email=vpn_email,
                            traffic_gb=int(new_traffic_limit),
                            days=max(1, (new_expiry_date - dt.now()).days),  # Минимум 1 день
                            user_id=user_id,  # Telegram ID пользователя
                            subscription_id=subscription_id or 0  # ID подписки или 0 если None
                        )
                        
                        return success
                        
                    except Exception as e:
                        logger.error(f"Ошибка обновления клиента в панели: {e}")
                        return False
                
                # Запускаем обновление панели
                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                panel_update_success = loop.run_until_complete(update_panel_client())
                loop.close()

                if panel_update_success:
                    logger.info(f"Клиент {vpn_email} успешно обновлен в панели: {new_traffic_limit}ГБ до {new_expiry_date.strftime('%d.%m.%Y')}")
                else:
                    logger.error(f"Ошибка обновления клиента {vpn_email} в панели")
            else:
                logger.info(f"Не найдены данные клиента в панели для подписки {subscription_id}")

        except Exception as e:
            logger.error(f"Ошибка обновления клиента в панели: {e}")

        # Отправляем уведомление пользователю через Telegram API
        try:
            import asyncio
            import aiohttp
            
            async def send_payment_notification():
                bot_token = os.environ.get('BOT_TOKEN')
                if not bot_token:
                    logger.info("BOT_TOKEN не найден для отправки уведомления")
                    return False
                
                # Формируем сообщение с инструкцией для получения ключа
                message = f'''✅ Платеж успешно обработан!

💳 ID платежа: {payment_id}
📋 Тариф: {plan_info['name']}
💰 Сумма: {amount} RUB

🔄 Подписка продлена до {new_expiry_date.strftime('%d.%m.%Y')}
📊 Лимит трафика: {new_traffic_limit} ГБ
📊 Трафик сохранен

📱 Ваш VPN готов к использованию!
🔑 Получите ваш VPN ключ нажав кнопку ниже'''
                
                url = f'https://api.telegram.org/bot{bot_token}/sendMessage'
                
                # Создаем инлайн-клавиатуру с кнопкой "Получить ключ"
                keyboard = {
                    "inline_keyboard": [
                        [
                            {
                                "text": "🔑 Получить VPN ключ",
                                "callback_data": f"get_vpn_key_{subscription_id}"
                            }
                        ],
                        [
                            {
                                "text": "👤 Личный кабинет", 
                                "callback_data": "personal_account"
                            }
                        ]
                    ]
                }
                
                async with aiohttp.ClientSession() as session:
                    try:
                        async with session.post(url, json={
                            'chat_id': user_id,
                            'text': message,
                            'parse_mode': 'HTML',
                            'reply_markup': keyboard
                        }) as response:
                            if response.status == 200:
                                logger.info(f"✅ Уведомление отправлено пользователю {user_id}")
                                return True
                            else:
                                result = await response.text()
                                logger.info(f"❌ Ошибка отправки уведомления: {response.status} - {result}")
                                return False
                    except Exception as e:
                        logger.info(f"❌ Исключение при отправке уведомления: {e}")
                        return False
            
            # Запускаем отправку уведомления
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            notification_sent = loop.run_until_complete(send_payment_notification())
            loop.close()
            
            if not notification_sent:
                logger.info(f"Резервное уведомление: Платеж {payment_id} успешен для пользователя {user_id}, план {plan_info['name']}, сумма {amount}")
                
        except Exception as notification_error:
            logger.error(f"Ошибка отправки уведомления: {notification_error}")
            logger.info(f"Резервное уведомление: Платеж {payment_id} успешен для пользователя {user_id}, план {plan_info['name']}, сумма {amount}")

        logger.info(f"Платеж {payment_id} успешно обработан, подписка {subscription_id} обновлена")
        return True

    except Exception as e:
        logger.error(f"Ошибка обработки платежа {payment_id}: {e}")
        return False


# Async notification functions removed from Flask app to maintain sync/async separation
# Notifications are handled by the Telegram bot asynchronously


def get_payment_error_description(error_code: str) -> str:
    """Возвращает понятное описание ошибки платежа по коду YooKassa"""
    error_descriptions = {
        # Ошибки карт
        "card_expired": "Истек срок действия карты",
        "insufficient_funds": "Недостаточно средств на карте",
        "invalid_card_number": "Неверный номер карты",
        "invalid_csc": "Неверный CVV/CVC код",
        "issuer_unavailable": "Банк-эмитент недоступен",
        "payment_method_limit_exceeded": "Превышен лимит платежей",
        "payment_method_restricted": "Платежи с данной карты ограничены",
        "rejected_by_issuer": "Банк отклонил операцию",
        "lost_card": "Карта заблокирована (утеряна)",
        "stolen_card": "Карта заблокирована (украдена)",
        "expired_card": "Истек срок действия карты",
        # Ошибки 3DS
        "three_d_secure_failed": "Ошибка подтверждения 3D-Secure",
        "three_d_secure_not_supported": "3D-Secure не поддерживается",
        # Технические ошибки
        "processing_error": "Техническая ошибка обработки",
        "general_decline": "Платеж отклонен банком",
        "invalid_processing_request": "Некорректный запрос",
        # YooMoney ошибки
        "account_blocked": "Кошелек заблокирован",
        "account_closed": "Кошелек закрыт",
        "fraud_suspected": "Подозрение на мошенничество",
        # Временные ошибки
        "expired_on_capture": "Превышено время подтверждения платежа",
        "expired_on_confirmation": "Превышено время ожидания подтверждения",
        "payment_timeout": "Превышено время ожидания",
        "timeout": "Истекло время ожидания ответа",
        # Sberbank ошибки
        "sberbank_online_rejected": "Sberbank Online отклонил платеж",
        # Общие ошибки
        "canceled_by_merchant": "Платеж отменен продавцом",
        "permission_revoked": "Отозвано разрешение на списание",
        "invalid_data": "Неверные данные платежа",
        "merchant_account_error": "Ошибка аккаунта продавца",
        "payment_method_unavailable": "Способ оплаты временно недоступен",
        "unsupported_mobile_operator": "Мобильный оператор не поддерживается",
        # Ошибки интернет-банкинга
        "alfabank_online_rejected": "Альфа-Банк отклонил платеж",
        "payment_method_not_available": "Способ оплаты недоступен",
        # Ошибки валидации
        "invalid_request": "Некорректный запрос",
        "forbidden": "Операция запрещена",
        "not_supported": "Операция не поддерживается",
    }

    return error_descriptions.get(error_code, f"Ошибка платежной системы ({error_code})")


@app.route("/api/refresh-users", methods=["POST"])
def api_refresh_users():
    """API эндпоинт для принудительного обновления данных пользователей с синхронизацией трафика"""
    try:
        logger.info("Принудительное обновление: синхронизация пользователей и трафика...")
        
        # Синхронизируем пользователей из базы бота
        from sync_web_stats import sync_users_from_bot_db
        users_synced = sync_users_from_bot_db()
        logger.info(f"Синхронизировано пользователей: {users_synced}")
        
        # Синхронизируем трафик
        from sync_web_stats import sync_refresh_traffic
        sync_refresh_traffic()
        
        # Очищаем кеш пользователей для принудительного обновления
        cache_key = "users_data"
        with _cache_lock:
            if cache_key in _cache:
                del _cache[cache_key]
                logger.info("Кеш пользователей очищен")
        
        return jsonify({
            "status": "success", 
            "message": f"Синхронизировано {users_synced} пользователей, трафик обновлен"
        })
        
    except Exception as e:
        logger.error(f"Ошибка при обновлении данных: {e}")
        return jsonify({"status": "error", "message": str(e)}), 500


@app.route("/api/sync-traffic", methods=["POST"])
def api_sync_traffic():
    """API эндпоинт для синхронизации трафика с панелью 3X-UI"""
    try:
        from sync_web_stats import sync_refresh_traffic

        # Запускаем синхронизацию трафика
        result = sync_refresh_traffic()

        return jsonify(
            {"success": True, "message": "Синхронизация трафика завершена", "data": result}
        )

    except Exception as e:
        return jsonify({"success": False, "error": f"Ошибка синхронизации: {str(e)}"}), 500


@app.route("/api/scheduler/status")
def api_scheduler_status():
    """API endpoint для получения актуального статуса планировщика"""
    status = get_scheduler_status()
    return jsonify(status)

@app.route("/api/get-panel-id", methods=["POST"])
def api_get_panel_id():
    """API эндпоинт для получения ID клиента из панели 3X-UI"""
    try:
        import asyncio
        import sys
        import os

        # Добавляем путь к модулям бота
        sys.path.append(os.path.join(os.path.dirname(__file__), "bot"))

        from bot.services.working_panel_api import WorkingPanelAPI
        from config_keys import get_panel_credentials

        data = request.json
        if not data:
            return jsonify({"success": False, "error": "Не получены данные запроса"})
        
        user_id = data.get("user_id")
        if not user_id:
            return jsonify({"success": False, "error": "User ID не указан"})

        # Получаем данные панели
        panel_url, panel_username, panel_password = get_panel_credentials()

        # Получаем реальный UUID из PostgreSQL базы данных
        try:
            from bot.database.postgres_operations import get_user_active_subscription_postgres
            
            subscription = run_async_safely(get_user_active_subscription_postgres(user_id))
            
            if subscription and subscription.get('vpn_user_id'):
                panel_id = subscription['vpn_user_id']
                return jsonify({"success": True, "panel_id": panel_id, "note": "Real UUID from PostgreSQL database"})
            else:
                return jsonify({"success": False, "error": "UUID не найден в базе данных"})
        except Exception as e:
            return jsonify({"success": False, "error": str(e)})

    except Exception as e:
        return jsonify({"success": False, "error": str(e)}), 500


@app.route("/api/delete-user/<int:user_id>", methods=["DELETE"])
def api_delete_user(user_id):
    """API эндпоинт для полного удаления пользователя из базы данных и панели"""
    try:
        import asyncio
        import sys
        import os
        from sqlalchemy import create_engine, text

        # Добавляем путь к модулям бота
        sys.path.append(os.path.join(os.path.dirname(__file__), "bot"))

        from config_keys import get_panel_credentials

        # Получаем данные пользователя перед удалением из PostgreSQL
        engine = create_engine(os.environ['DATABASE_URL'])
        
        # Получаем данные пользователя
        with engine.connect() as conn:
            result = conn.execute(text("SELECT vpn_email FROM users WHERE id = :user_id"), {"user_id": user_id})
            user_result = result.fetchone()

            if not user_result:
                return jsonify({"success": False, "error": "Пользователь не найден"})

            vpn_email = user_result[0]

        # Удаляем из панели 3X-UI если есть VPN email
        panel_deletion_success = False
        if vpn_email:
            try:
                logger.info(f"Удаляем пользователя {user_id} с email {vpn_email} из панели...")
                
                # Создаем асинхронную функцию для удаления из панели
                async def delete_from_panel():
                    from config_keys import get_panel_credentials
                    import aiohttp
                    import ssl
                    import json
                    import re
                    
                    panel_url, panel_username, panel_password = get_panel_credentials()
                    base_url = panel_url.rstrip('/').removesuffix('/panel')
                    
                    ssl_context = ssl.create_default_context()
                    ssl_context.check_hostname = False
                    ssl_context.verify_mode = ssl.CERT_NONE
                    
                    try:
                        async with aiohttp.ClientSession(
                            connector=aiohttp.TCPConnector(ssl=ssl_context)
                        ) as session:
                            # Авторизация
                            login_url = f'{base_url}/login'
                            login_data = f'username={panel_username}&password={panel_password}'
                            
                            async with session.post(
                                login_url,
                                data=login_data,
                                headers={'Content-Type': 'application/x-www-form-urlencoded'}
                            ) as response:
                                if response.status != 200:
                                    return False
                                
                                cookie_3x_ui = None
                                for cookie_header in response.headers.getall('Set-Cookie', []):
                                    if '3x-ui=' in cookie_header:
                                        match = re.search(r'3x-ui=([^;]+)', cookie_header)
                                        if match:
                                            cookie_3x_ui = match.group(1)
                                            break
                                
                                if not cookie_3x_ui:
                                    return False
                            
                            # Получаем список всех клиентов
                            list_url = f'{base_url}/panel/api/inbounds/list'
                            headers = {'Cookie': f'3x-ui={cookie_3x_ui}'}
                            
                            async with session.post(list_url, headers=headers) as response:
                                if response.status != 200:
                                    return False
                                
                                result = await response.json()
                                if not result.get('success'):
                                    return False
                                
                                # Ищем клиента по email
                                client_uuid = None
                                for inbound in result.get('obj', []):
                                    settings = json.loads(inbound.get('settings', '{}'))
                                    clients = settings.get('clients', [])
                                    for client in clients:
                                        if client.get('email') == vpn_email:
                                            client_uuid = client.get('id')
                                            break
                                    if client_uuid:
                                        break
                                
                                if not client_uuid:
                                    logger.info(f"Клиент с email {vpn_email} не найден в панели")
                                    return False
                            
                            # Удаляем клиента из панели
                            delete_url = f'{base_url}/panel/api/inbounds/delClient/{client_uuid}'
                            delete_data = {'id': 1}  # ID inbound
                            
                            async with session.post(
                                delete_url, 
                                json=delete_data, 
                                headers={
                                    'Cookie': f'3x-ui={cookie_3x_ui}',
                                    'Content-Type': 'application/json'
                                }
                            ) as response:
                                if response.status == 200:
                                    try:
                                        result = await response.json()
                                        return result.get('success', False)
                                    except Exception as json_error:
                                        # Если ответ не JSON, читаем как текст
                                        text_response = await response.text()
                                        logger.info(f"Панель вернула не-JSON ответ: {text_response[:200]}")
                                        # Считаем успешным, если статус 200
                                        return True
                                return False
                                
                    except Exception as e:
                        logger.error(f"Ошибка API панели при удалении: {e}")
                        return False
                
                # Выполняем асинхронное удаление
                panel_deletion_success = asyncio.run(delete_from_panel())
                
                if panel_deletion_success:
                    logger.info(f"✅ Пользователь {vpn_email} успешно удален из панели")
                else:
                    logger.info(f"❌ Не удалось удалить пользователя {vpn_email} из панели")
                    
            except Exception as panel_error:
                logger.error(f"Ошибка при удалении из панели: {panel_error}")
                panel_deletion_success = False

        # Удаляем все связанные данные из PostgreSQL базы
        with engine.connect() as conn:
            conn.execute(text("DELETE FROM payments WHERE user_id = :user_id"), {"user_id": user_id})
            conn.execute(text("DELETE FROM subscriptions WHERE user_id = :user_id"), {"user_id": user_id})
            conn.execute(text("DELETE FROM users WHERE id = :user_id"), {"user_id": user_id})
            conn.commit()

        # Формируем сообщение результата
        if vpn_email:
            if panel_deletion_success:
                message = f"Пользователь {user_id} полностью удален из базы данных и панели 3X-UI"
            else:
                message = f"Пользователь {user_id} удален из базы данных, но возможны проблемы с удалением из панели 3X-UI"
        else:
            message = f"Пользователь {user_id} удален из базы данных (не имел VPN ключа)"

        return jsonify({"success": True, "message": message})

    except Exception as e:
        return jsonify({"success": False, "error": str(e)}), 500


# === API ДЛЯ УПРАВЛЕНИЯ ТАРИФАМИ ===

@app.route("/api/tariffs", methods=["GET", "POST"])
def api_tariffs():
    """API для получения и создания тарифов"""
    from database.sqlite_config import config_db
    
    if request.method == "GET":
        try:
            tariffs = config_db.get_all_tariffs()
            return jsonify({"success": True, "tariffs": tariffs})
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})
    
    elif request.method == "POST":
        try:
            data = request.get_json()
            
            # Валидация данных
            required_fields = ['id', 'name', 'price', 'traffic_gb', 'duration_days']
            for field in required_fields:
                if field not in data:
                    return jsonify({"success": False, "message": f"Поле {field} обязательно"})
            
            # Вычисляем цену за 30 дней если указана скидка
            if 'discount_percent' in data and data['discount_percent'] > 0:
                price_per_30_days = round((data['price'] / data['duration_days']) * 30, 1)
                data['price_per_30_days'] = price_per_30_days
            
            success = config_db.create_or_update_tariff(data)
            
            if success:
                return jsonify({"success": True, "message": "Тариф сохранен"})
            else:
                return jsonify({"success": False, "message": "Ошибка при сохранении тарифа"})
                
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})

@app.route("/api/tariffs/<tariff_id>", methods=["PUT", "DELETE"])
def api_tariff_by_id(tariff_id):
    """API для обновления и удаления тарифа"""
    from database.sqlite_config import config_db
    
    if request.method == "PUT":
        try:
            data = request.get_json()
            data['id'] = tariff_id
            
            # Вычисляем цену за 30 дней если указана скидка
            if 'discount_percent' in data and data['discount_percent'] > 0:
                price_per_30_days = round((data['price'] / data['duration_days']) * 30, 1)
                data['price_per_30_days'] = price_per_30_days
            
            success = config_db.create_or_update_tariff(data)
            
            if success:
                return jsonify({"success": True, "message": "Тариф обновлен"})
            else:
                return jsonify({"success": False, "message": "Ошибка при обновлении тарифа"})
                
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})
    
    elif request.method == "DELETE":
        try:
            success = config_db.delete_tariff(tariff_id)
            
            if success:
                return jsonify({"success": True, "message": "Тариф удален"})
            else:
                return jsonify({"success": False, "message": "Ошибка при удалении тарифа"})
                
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})

# === API ДЛЯ УПРАВЛЕНИЯ ДОПОЛНИТЕЛЬНЫМИ УСЛУГАМИ ===

@app.route("/api/services", methods=["GET", "POST"])
def api_services():
    """API для получения и создания дополнительных услуг"""
    from database.sqlite_config import config_db
    
    if request.method == "GET":
        try:
            services = config_db.get_all_services()
            return jsonify({"success": True, "services": services})
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})
    
    elif request.method == "POST":
        try:
            data = request.get_json()
            
            # Валидация данных
            required_fields = ['title', 'description']
            for field in required_fields:
                if field not in data:
                    return jsonify({"success": False, "message": f"Поле {field} обязательно"})
            
            success = config_db.create_or_update_service(data)
            
            if success:
                return jsonify({"success": True, "message": "Услуга сохранена"})
            else:
                return jsonify({"success": False, "message": "Ошибка при сохранении услуги"})
                
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})

@app.route("/api/services/<int:service_id>", methods=["PUT", "DELETE"])
def api_service_by_id(service_id):
    """API для обновления и удаления услуги"""
    from database.sqlite_config import config_db
    
    if request.method == "PUT":
        try:
            data = request.get_json()
            data['id'] = service_id
            
            success = config_db.create_or_update_service(data)
            
            if success:
                return jsonify({"success": True, "message": "Услуга обновлена"})
            else:
                return jsonify({"success": False, "message": "Ошибка при обновлении услуги"})
                
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})
    
    elif request.method == "DELETE":
        try:
            success = config_db.delete_service(service_id)
            
            if success:
                return jsonify({"success": True, "message": "Услуга удалена"})
            else:
                return jsonify({"success": False, "message": "Ошибка при удалении услуги"})
                
        except Exception as e:
            return jsonify({"success": False, "message": str(e)})

# === API ДЛЯ ОТПРАВКИ АДМИНСКИХ УВЕДОМЛЕНИЙ ===

@app.route("/api/send_admin_notification", methods=["POST"])
def api_send_admin_notification():
    """API для отправки админского уведомления всем пользователям"""
    from database.sqlite_config import config_db
    
    try:
        data = request.get_json()
        
        # Валидация данных
        required_fields = ['title', 'message', 'type']
        for field in required_fields:
            if field not in data:
                return jsonify({"success": False, "message": f"Поле {field} обязательно"})
        
        if data['type'] not in ['admin', 'promo']:
            return jsonify({"success": False, "message": "Тип должен быть 'admin' или 'promo'"})
        
        # Сохраняем уведомление в базу
        notification_data = {
            'type': data['type'],
            'title': data['title'],
            'message': data['message']
        }
        
        success = config_db.create_notification(notification_data)
        
        if not success:
            return jsonify({"success": False, "message": "Ошибка при сохранении уведомления"})
        
        # Получаем список пользователей для отправки из PostgreSQL
        try:
            from sqlalchemy import create_engine, text
            import os
            
            engine = create_engine(os.environ.get("DATABASE_URL"))
            with engine.connect() as conn:
                result = conn.execute(text("SELECT id FROM users WHERE id IS NOT NULL"))
                user_ids = [row[0] for row in result.fetchall()]
            
            if not user_ids:
                return jsonify({"success": False, "message": "Нет пользователей для отправки"})
            
            # Отправляем уведомление через бота
            async def send_to_all_users():
                from main import bot
                sent_count = 0
                
                for user_id in user_ids:
                    try:
                        message_text = f"<b>{data['title']}</b>\n\n{data['message']}"
                        
                        if data['type'] == 'admin':
                            message_text = f"🔧 <b>Оперативное уведомление от Администратора сервиса RazDvaVPN</b>\n\n{message_text}"
                        elif data['type'] == 'promo':
                            message_text = f"🎉 <b>Информационное сообщение</b>\n\n{message_text}"
                        
                        await bot.send_message(user_id, message_text)
                        sent_count += 1
                        
                        # Небольшая задержка между сообщениями
                        import asyncio
                        await asyncio.sleep(0.1)
                        
                    except Exception as e:
                        logger.error(f"Ошибка отправки уведомления пользователю {user_id}: {e}")
                
                return sent_count
            
            sent_count = run_async_safely(send_to_all_users())
            
            # Обновляем статус уведомления
            # Получаем ID последнего созданного уведомления
            notifications = config_db.get_notifications(data['type'])
            if notifications:
                last_notification_id = notifications[0]['id']
                config_db.mark_notification_sent(last_notification_id, sent_count)
            
            return jsonify({
                "success": True, 
                "message": f"Уведомление отправлено {sent_count} пользователям"
            })
            
        except Exception as e:
            logger.error(f"Ошибка при отправке уведомлений: {e}")
            return jsonify({"success": False, "message": f"Ошибка отправки: {str(e)}"})
        
    except Exception as e:
        logger.error(f"Ошибка API отправки уведомлений: {e}")
        return jsonify({"success": False, "message": str(e)})

@app.route("/api/notifications", methods=["GET"])
def api_get_notifications():
    """API для получения истории уведомлений"""
    from database.sqlite_config import config_db
    
    try:
        notification_type = request.args.get('type')
        notifications = config_db.get_notifications(notification_type)
        return jsonify({"success": True, "notifications": notifications})
    except Exception as e:
        return jsonify({"success": False, "message": str(e)})

# === API ДЛЯ УПРАВЛЕНИЯ УВЕДОМЛЕНИЯМИ ПОДПИСЧИКОВ ===

@app.route("/api/notification-settings", methods=["GET"])
def api_get_notification_settings():
    """API для получения настроек уведомлений подписчиков"""
    from database.sqlite_config import config_db
    
    try:
        plan_type = request.args.get('plan_type')
        settings = config_db.get_notification_settings(plan_type)
        return jsonify({"success": True, "settings": settings})
    except Exception as e:
        return jsonify({"success": False, "message": str(e)})

@app.route("/api/notification-settings/<int:setting_id>", methods=["PUT"])
def api_update_notification_setting(setting_id):
    """API для обновления настройки уведомления"""
    from database.sqlite_config import config_db
    
    try:
        data = request.get_json()
        success = config_db.update_notification_setting(setting_id, **data)
        if success:
            return jsonify({"success": True, "message": "Настройка обновлена"})
        else:
            return jsonify({"success": False, "message": "Ошибка обновления настройки"})
    except Exception as e:
        return jsonify({"success": False, "message": str(e)})

@app.route("/api/notification-logs", methods=["GET"])
def api_get_notification_logs():
    """API для получения логов уведомлений"""
    from database.sqlite_config import config_db
    
    try:
        notification_type = request.args.get('type')
        limit = int(request.args.get('limit', 100))
        logs = config_db.get_notification_logs(limit, notification_type)
        return jsonify({"success": True, "logs": logs})
    except Exception as e:
        return jsonify({"success": False, "message": str(e)})


if __name__ == "__main__":
    app.run(host="0.0.0.0", port=5000, debug=True)
