"""
SQLite база данных для конфигурации тарифов, дополнительных услуг и админских уведомлений
"""
import sqlite3
import os
from datetime import datetime
from typing import Dict, List, Optional, Any
import json
import logging

logger = logging.getLogger(__name__)

class SQLiteConfig:
    def __init__(self, db_path: str = "database/config.db"):
        self.db_path = db_path
        self._initialized = False
        # Создаем директорию если её нет
        os.makedirs(os.path.dirname(db_path), exist_ok=True)
        # Ленивая инициализация - только при первом обращении
        
    def _ensure_initialized(self):
        """Обеспечивает подключение к базе данных при первом обращении"""
        if not self._initialized:
            # Если база не существует - создаем, иначе просто подключаемся
            if not os.path.exists(self.db_path):
                self.init_database()
            self._initialized = True

    def init_database(self):
        """Инициализация базы данных и создание таблиц"""
        with sqlite3.connect(self.db_path) as conn:
            cursor = conn.cursor()
            
            # Таблица тарифов
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS tariffs (
                    id TEXT PRIMARY KEY,
                    name TEXT NOT NULL,
                    price INTEGER NOT NULL,
                    traffic_gb INTEGER NOT NULL,
                    duration_days INTEGER NOT NULL,
                    description TEXT,
                    discount_percent REAL DEFAULT 0,
                    price_per_30_days REAL DEFAULT 0,
                    is_active BOOLEAN DEFAULT 1,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            ''')
            
            # Таблица дополнительных услуг
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS additional_services (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    title TEXT NOT NULL,
                    description TEXT NOT NULL,
                    link TEXT,
                    button_text TEXT DEFAULT 'Перейти',
                    is_active BOOLEAN DEFAULT 1,
                    sort_order INTEGER DEFAULT 0,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            ''')
            
            # Таблица админских уведомлений
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS admin_notifications (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    type TEXT NOT NULL, -- 'admin', 'promo'
                    title TEXT NOT NULL,
                    message TEXT NOT NULL,
                    is_sent BOOLEAN DEFAULT 0,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    sent_at TIMESTAMP,
                    recipients_count INTEGER DEFAULT 0
                )
            ''')
            
            # Таблица логов уведомлений для мониторинга
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS notification_log (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    notification_type TEXT NOT NULL, -- 'trial', 'expiry', 'traffic', 'admin', 'promo'
                    user_id INTEGER,
                    message TEXT,
                    success BOOLEAN DEFAULT 0,
                    error_message TEXT,
                    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            ''')
            
            # Таблица настроек уведомлений для Trial и платных подписчиков
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS notification_settings (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    plan_type TEXT NOT NULL, -- 'trial', 'paid'
                    notification_type TEXT NOT NULL, -- 'expiry_1day', 'expiry_3day', 'traffic_80', 'traffic_90'
                    is_enabled BOOLEAN DEFAULT 1,
                    message_template TEXT,
                    hours_before INTEGER, -- для уведомлений об истечении
                    traffic_percent INTEGER, -- для уведомлений о трафике
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            ''')
            
            # Таблица расписания уведомлений
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS notification_schedule (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    user_id INTEGER NOT NULL,
                    notification_type TEXT NOT NULL,
                    plan_type TEXT NOT NULL,
                    scheduled_time TIMESTAMP NOT NULL,
                    is_sent BOOLEAN DEFAULT 0,
                    sent_at TIMESTAMP,
                    message TEXT,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )
            ''')
            
            conn.commit()
            logger.info("SQLite база данных конфигурации инициализирована")

    # === ОПЕРАЦИИ С ТАРИФАМИ ===
    
    def get_all_tariffs(self) -> List[Dict]:
        """Получить все активные тарифы"""
        self._ensure_initialized()
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            cursor.execute('SELECT * FROM tariffs WHERE is_active = 1 ORDER BY price')
            return [dict(row) for row in cursor.fetchall()]
    
    def get_tariff(self, tariff_id: str) -> Optional[Dict]:
        """Получить тариф по ID"""
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            cursor.execute('SELECT * FROM tariffs WHERE id = ?', (tariff_id,))
            row = cursor.fetchone()
            return dict(row) if row else None
    
    def create_or_update_tariff(self, tariff_data: Dict) -> bool:
        """Создать или обновить тариф"""
        self._ensure_initialized()
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                
                # Проверяем существует ли тариф
                cursor.execute('SELECT id FROM tariffs WHERE id = ?', (tariff_data['id'],))
                exists = cursor.fetchone()
                
                if exists:
                    # Обновляем существующий
                    cursor.execute('''
                        UPDATE tariffs SET 
                            name = ?, price = ?, traffic_gb = ?, duration_days = ?,
                            description = ?, discount_percent = ?, price_per_30_days = ?,
                            is_active = ?, updated_at = CURRENT_TIMESTAMP
                        WHERE id = ?
                    ''', (
                        tariff_data['name'], tariff_data['price'], tariff_data['traffic_gb'],
                        tariff_data['duration_days'], tariff_data.get('description', ''),
                        tariff_data.get('discount_percent', 0), tariff_data.get('price_per_30_days', 0),
                        tariff_data.get('is_active', 1), tariff_data['id']
                    ))
                else:
                    # Создаем новый
                    cursor.execute('''
                        INSERT INTO tariffs (id, name, price, traffic_gb, duration_days, 
                                           description, discount_percent, price_per_30_days, is_active)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ''', (
                        tariff_data['id'], tariff_data['name'], tariff_data['price'],
                        tariff_data['traffic_gb'], tariff_data['duration_days'],
                        tariff_data.get('description', ''), tariff_data.get('discount_percent', 0),
                        tariff_data.get('price_per_30_days', 0), tariff_data.get('is_active', 1)
                    ))
                
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Ошибка при сохранении тарифа: {e}")
            return False
    
    def delete_tariff(self, tariff_id: str) -> bool:
        """Удалить тариф (деактивировать)"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                cursor.execute('UPDATE tariffs SET is_active = 0 WHERE id = ?', (tariff_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Ошибка при удалении тарифа: {e}")
            return False

    # === ОПЕРАЦИИ С ДОПОЛНИТЕЛЬНЫМИ УСЛУГАМИ ===
    
    def get_all_services(self) -> List[Dict]:
        """Получить все активные дополнительные услуги"""
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            cursor.execute('SELECT * FROM additional_services WHERE is_active = 1 ORDER BY sort_order, title')
            return [dict(row) for row in cursor.fetchall()]
    
    def create_or_update_service(self, service_data: Dict) -> bool:
        """Создать или обновить дополнительную услугу"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                
                if 'id' in service_data and service_data['id']:
                    # Обновляем существующий
                    cursor.execute('''
                        UPDATE additional_services SET 
                            title = ?, description = ?, link = ?, button_text = ?,
                            is_active = ?, sort_order = ?, updated_at = CURRENT_TIMESTAMP
                        WHERE id = ?
                    ''', (
                        service_data['title'], service_data['description'], service_data.get('link', ''),
                        service_data.get('button_text', 'Перейти'), service_data.get('is_active', 1),
                        service_data.get('sort_order', 0), service_data['id']
                    ))
                else:
                    # Создаем новый
                    cursor.execute('''
                        INSERT INTO additional_services (title, description, link, button_text, is_active, sort_order)
                        VALUES (?, ?, ?, ?, ?, ?)
                    ''', (
                        service_data['title'], service_data['description'], service_data.get('link', ''),
                        service_data.get('button_text', 'Перейти'), service_data.get('is_active', 1),
                        service_data.get('sort_order', 0)
                    ))
                
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Ошибка при сохранении услуги: {e}")
            return False
    
    def delete_service(self, service_id: int) -> bool:
        """Удалить дополнительную услугу"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                cursor.execute('UPDATE additional_services SET is_active = 0 WHERE id = ?', (service_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Ошибка при удалении услуги: {e}")
            return False

    # === ОПЕРАЦИИ С АДМИНСКИМИ УВЕДОМЛЕНИЯМИ ===
    
    def get_notifications(self, notification_type: str = None) -> List[Dict]:
        """Получить уведомления"""
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            
            if notification_type:
                cursor.execute('SELECT * FROM admin_notifications WHERE type = ? ORDER BY created_at DESC', (notification_type,))
            else:
                cursor.execute('SELECT * FROM admin_notifications ORDER BY created_at DESC')
            
            return [dict(row) for row in cursor.fetchall()]
    
    def create_notification(self, notification_data: Dict) -> bool:
        """Создать уведомление"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                cursor.execute('''
                    INSERT INTO admin_notifications (type, title, message)
                    VALUES (?, ?, ?)
                ''', (
                    notification_data['type'], notification_data['title'], notification_data['message']
                ))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Ошибка при создании уведомления: {e}")
            return False
    
    def mark_notification_sent(self, notification_id: int, recipients_count: int) -> bool:
        """Отметить уведомление как отправленное"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                cursor.execute('''
                    UPDATE admin_notifications SET 
                        is_sent = 1, sent_at = CURRENT_TIMESTAMP, recipients_count = ?
                    WHERE id = ?
                ''', (recipients_count, notification_id))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Ошибка при обновлении статуса уведомления: {e}")
            return False

    def migrate_current_tariffs(self):
        """Миграция текущих тарифов из config.py в SQLite"""
        self._ensure_initialized()
        from config import SUBSCRIPTION_PLANS
        
        logger.info("Миграция тарифов в SQLite...")
        
        for plan_id, plan_data in SUBSCRIPTION_PLANS.items():
            tariff_data = {
                'id': plan_id,
                'name': plan_data['name'],
                'price': plan_data['price'],
                'traffic_gb': plan_data['traffic_gb'],
                'duration_days': plan_data['duration_days'],
                'description': plan_data.get('description', ''),
                'discount_percent': plan_data.get('discount_percent', 0),
                'price_per_30_days': plan_data.get('price_per_30_days', 0),
                'is_active': 1
            }
            
            self.create_or_update_tariff(tariff_data)
        
        # Дополнительные услуги создаются только вручную через веб-интерфейс
        
        logger.info("Миграция завершена")
        
        # Создаем базовые настройки уведомлений
        self._create_default_notification_settings()
    
    def _create_default_notification_settings(self):
        """Создает базовые настройки уведомлений для Trial и платных подписчиков"""
        default_settings = [
            # Trial подписки
            {
                'plan_type': 'trial',
                'notification_type': 'expiry_1day',
                'is_enabled': 1,
                'message_template': '⚠️ Ваша пробная подписка истекает завтра!\n\nПродлите подписку, чтобы продолжить пользоваться VPN.',
                'hours_before': 24,
                'traffic_percent': None
            },
            {
                'plan_type': 'trial',
                'notification_type': 'traffic_80',
                'is_enabled': 1,
                'message_template': '📊 Внимание! Использовано 80% трафика пробной подписки.\n\nРекомендуем продлить подписку.',
                'hours_before': None,
                'traffic_percent': 80
            },
            # Платные подписки
            {
                'plan_type': 'paid',
                'notification_type': 'expiry_3day',
                'is_enabled': 1,
                'message_template': '⏰ Ваша подписка истекает через 3 дня.\n\nНе забудьте продлить подписку для бесперебойного доступа.',
                'hours_before': 72,
                'traffic_percent': None
            },
            {
                'plan_type': 'paid',
                'notification_type': 'expiry_1day',
                'is_enabled': 1,
                'message_template': '🚨 Ваша подписка истекает завтра!\n\nПродлите подписку прямо сейчас.',
                'hours_before': 24,
                'traffic_percent': None
            },
            {
                'plan_type': 'paid',
                'notification_type': 'traffic_90',
                'is_enabled': 1,
                'message_template': '📈 Использовано 90% трафика!\n\nРекомендуем продлить подписку или докупить трафик.',
                'hours_before': None,
                'traffic_percent': 90
            }
        ]
        
        with sqlite3.connect(self.db_path) as conn:
            cursor = conn.cursor()
            for setting in default_settings:
                # Проверяем, существует ли уже такая настройка
                cursor.execute('''
                    SELECT id FROM notification_settings 
                    WHERE plan_type = ? AND notification_type = ?
                ''', (setting['plan_type'], setting['notification_type']))
                
                if not cursor.fetchone():
                    cursor.execute('''
                        INSERT INTO notification_settings 
                        (plan_type, notification_type, is_enabled, message_template, hours_before, traffic_percent)
                        VALUES (?, ?, ?, ?, ?, ?)
                    ''', (
                        setting['plan_type'], setting['notification_type'], setting['is_enabled'],
                        setting['message_template'], setting['hours_before'], setting['traffic_percent']
                    ))
            conn.commit()

    # === ОПЕРАЦИИ С НАСТРОЙКАМИ УВЕДОМЛЕНИЙ ===
    
    def get_notification_settings(self, plan_type: str = None) -> List[Dict]:
        """Получить настройки уведомлений"""
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            if plan_type:
                cursor.execute('SELECT * FROM notification_settings WHERE plan_type = ? ORDER BY notification_type', (plan_type,))
            else:
                cursor.execute('SELECT * FROM notification_settings ORDER BY plan_type, notification_type')
            return [dict(row) for row in cursor.fetchall()]
    
    def update_notification_setting(self, setting_id: int, **kwargs) -> bool:
        """Обновить настройку уведомления"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                
                # Формируем SQL запрос для обновления
                fields = []
                values = []
                for key, value in kwargs.items():
                    if key in ['is_enabled', 'message_template', 'hours_before', 'traffic_percent']:
                        fields.append(f"{key} = ?")
                        values.append(value)
                
                if fields:
                    fields.append("updated_at = CURRENT_TIMESTAMP")
                    values.append(setting_id)
                    
                    sql = f"UPDATE notification_settings SET {', '.join(fields)} WHERE id = ?"
                    cursor.execute(sql, values)
                    conn.commit()
                    return True
                return False
        except Exception as e:
            logger.error(f"Ошибка при обновлении настройки уведомления: {e}")
            return False
    
    def log_notification(self, notification_type: str, user_id: int = None, 
                        message: str = None, success: bool = False, 
                        error_message: str = None) -> None:
        """Записать лог уведомления"""
        try:
            with sqlite3.connect(self.db_path) as conn:
                cursor = conn.cursor()
                cursor.execute('''
                    INSERT INTO notification_log 
                    (notification_type, user_id, message, success, error_message)
                    VALUES (?, ?, ?, ?, ?)
                ''', (notification_type, user_id, message, success, error_message))
                conn.commit()
        except Exception as e:
            logger.error(f"Ошибка при записи лога уведомления: {e}")
    
    def get_notification_logs(self, limit: int = 100, notification_type: str = None) -> List[Dict]:
        """Получить логи уведомлений"""
        with sqlite3.connect(self.db_path) as conn:
            conn.row_factory = sqlite3.Row
            cursor = conn.cursor()
            if notification_type:
                cursor.execute('''
                    SELECT * FROM notification_log 
                    WHERE notification_type = ? 
                    ORDER BY sent_at DESC LIMIT ?
                ''', (notification_type, limit))
            else:
                cursor.execute('SELECT * FROM notification_log ORDER BY sent_at DESC LIMIT ?', (limit,))
            return [dict(row) for row in cursor.fetchall()]

# Глобальный экземпляр
config_db = SQLiteConfig()