import logging
from aiogram import Bot

from bot.database.models import User, Subscription

logger = logging.getLogger(__name__)


async def send_expiry_notification(bot: Bot, user: User, subscription: Subscription, days_left: int) -> None:
    """Send notification about subscription expiry"""
    from config import SUBSCRIPTION_PLANS
    from bot.utils.formatting import format_bytes_to_gb
    
    # Получаем информацию о тарифе
    plan_id = subscription.plan_id or "basic"
    plan = SUBSCRIPTION_PLANS.get(plan_id, {"name": "Стандартный"})
    plan_name = plan.get("name", "Стандартный")
    
    # Специальное сообщение для тестового тарифа
    if plan_id == "trial":
        # Форматируем данные о трафике
        remaining_gb = format_bytes_to_gb(subscription.traffic_remaining)
        percent_used = int(100 - (subscription.traffic_remaining / subscription.traffic_limit * 100))
        
        if days_left == 0:
            message = (
                f"⚠️ Тестовый период подходит к концу\n\n"
                f"Здравствуйте, {user.first_name or 'Пользователь'}!\n\n"
                f"📱 Осталось трафика: {remaining_gb:.1f} ГБ\n"
                f"📅 Тестовый период: еще менее 1 дня\n\n"
                f"Рекомендуем оформить полную подписку для непрерывного доступа."
            )
        else:
            message = (
                f"ℹ️ Тестовый период подходит к концу\n\n"
                f"Здравствуйте, {user.first_name or 'Пользователь'}!\n\n"
                f"📱 Осталось трафика: {remaining_gb:.1f} ГБ\n"
                f"📅 Тестовый период: еще {days_left} дня\n\n"
                f"Рекомендуем оформить полную подписку для непрерывного доступа."
            )
    else:
        # Для других тарифов
        message = (
            f"ℹ️ Напоминание о подписке\n\n"
            f"Здравствуйте, {user.first_name or 'Пользователь'}!\n\n"
            f"Ваша подписка '{plan_name}' истекает через {days_left} дней.\n\n"
            f"Для продления воспользуйтесь главным меню бота."
        )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message
        )
        
    except Exception as e:
        logger.error(f"Error sending expiry notification to user {user.id}: {e}")


async def send_traffic_warning(bot: Bot, user: User, subscription: Subscription) -> None:
    """Send notification about low traffic"""
    from config import SUBSCRIPTION_PLANS
    from bot.utils.formatting import format_bytes_to_gb
    
    # Получаем информацию о тарифе
    plan_id = subscription.plan_id or "basic"
    plan = SUBSCRIPTION_PLANS.get(plan_id, {"name": "Стандартный"})
    plan_name = plan.get("name", "Стандартный")
    
    # Форматируем данные о трафике
    remaining_gb = format_bytes_to_gb(subscription.traffic_remaining)
    used_gb = format_bytes_to_gb(subscription.traffic_used)
    limit_gb = format_bytes_to_gb(subscription.traffic_limit)
    
    message = (
        f"⚠️ Мало трафика\n\n"
        f"Здравствуйте, {user.first_name or 'Пользователь'}!\n\n"
        f"📊 Использовано: {used_gb:.1f} ГБ из {limit_gb:.1f} ГБ\n"
        f"📱 Осталось: {remaining_gb:.1f} ГБ\n"
        f"📦 Тариф: {plan_name}\n\n"
        f"Рекомендуем следить за расходом трафика."
    )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message
        )
        
    except Exception as e:
        logger.error(f"Error sending traffic warning to user {user.id}: {e}")


async def send_subscription_renewed(bot: Bot, user: User, subscription: Subscription, old_expiry, new_expiry) -> None:
    """Send notification about subscription renewal"""
    from config import SUBSCRIPTION_PLANS
    
    # Получаем информацию о тарифе
    plan_id = subscription.plan_id or "basic"
    plan = SUBSCRIPTION_PLANS.get(plan_id, {"name": "Стандартный"})
    plan_name = plan.get("name", "Стандартный")
    
    message = (
        f"✅ Подписка продлена\n\n"
        f"Здравствуйте, {user.first_name or 'Пользователь'}!\n\n"
        f"Ваша подписка '{plan_name}' успешно продлена.\n\n"
        f"📅 Действует до: {new_expiry.strftime('%d.%m.%Y')}\n\n"
        f"Спасибо за использование нашего сервиса!"
    )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message
        )
        
    except Exception as e:
        logger.error(f"Error sending renewal notification to user {user.id}: {e}")