"""
Утилиты для форматирования вывода
"""

def get_user_name(user):
    """Получает отформатированное имя пользователя"""
    if hasattr(user, 'full_name'):
        return user.full_name
    
    first = user.first_name or ""
    last = user.last_name or ""
    return f"{first} {last}".strip()


def format_bytes_to_gb(bytes_value):
    """
    Преобразует байты в гигабайты и форматирует значение
    
    Args:
        bytes_value: Значение в байтах
        
    Returns:
        str: Отформатированное значение в ГБ
    """
    if bytes_value is None:
        return "0.00"
    gb_value = bytes_value / (1024 * 1024 * 1024)
    return f"{gb_value:.2f}"

def format_traffic_info(traffic_limit_gb, traffic_used_bytes):
    """
    Правильно форматирует информацию о трафике с учетом разных единиц измерения
    
    Args:
        traffic_limit_gb: Лимит трафика в ГБ (float)
        traffic_used_bytes: Использованный трафик в байтах (float)
        
    Returns:
        tuple: (total_gb, used_gb, remaining_gb) как строки
    """
    if traffic_limit_gb is None:
        traffic_limit_gb = 0.0
    if traffic_used_bytes is None:
        traffic_used_bytes = 0.0
        
    # Проверяем, не хранится ли traffic_limit_gb случайно в байтах
    # Если значение больше 10000, вероятно это байты, конвертируем в ГБ
    if traffic_limit_gb > 10000:
        traffic_limit_gb = traffic_limit_gb / (1024 * 1024 * 1024)
    
    # traffic_limit_gb теперь точно в ГБ, traffic_used_bytes в байтах
    total_gb = f"{traffic_limit_gb:.2f}"
    used_gb_float = traffic_used_bytes / (1024 * 1024 * 1024)
    used_gb = f"{used_gb_float:.2f}"
    
    # Рассчитываем оставшийся трафик
    remaining_gb_float = max(0, traffic_limit_gb - used_gb_float)
    remaining_gb = f"{remaining_gb_float:.2f}"
    
    return total_gb, used_gb, remaining_gb