#!/usr/bin/env python3
"""
Рабочий API для панели 3X-UI на основе успешного теста
"""

import json
import asyncio
import ssl
import uuid
import aiohttp
import logging
from datetime import datetime, timedelta
from typing import Optional, Dict, Any
import sys
import os

# Добавляем корневую директорию в путь для импорта
sys.path.append(os.path.dirname(os.path.dirname(os.path.dirname(__file__))))

from config_keys import get_panel_credentials
from url_helper import get_correct_panel_urls, get_api_endpoint

logger = logging.getLogger(__name__)


class WorkingPanelAPI:
    """Рабочий API для панели 3X-UI"""

    def __init__(self, base_url=None, username=None, password=None):
        # Используем переданные параметры или получаем из конфигурации
        if base_url and username and password:
            self.panel_url = base_url
            self.username = username
            self.password = password
        else:
            # Загружаем из config_keys.py
            panel_url, username, password = get_panel_credentials()
            self.panel_url = panel_url
            self.username = username
            self.password = password
        self.session = None

    async def close(self):
        """Закрыть HTTP сессию"""
        if self.session:
            await self.session.close()
            self.session = None

    def _get_login_url(self):
        """Получить URL для авторизации (без /panel/)"""
        _, login_url, _ = get_correct_panel_urls(self.panel_url)
        return login_url

    def _get_api_url(self, endpoint):
        """Получить URL для API запросов (с /panel/api/)"""
        return get_api_endpoint(self.panel_url, endpoint)

    async def _get_session(self):
        """Получить сессию с правильными настройками"""
        if not self.session:
            ssl_context = ssl.create_default_context()
            ssl_context.check_hostname = False
            ssl_context.verify_mode = ssl.CERT_NONE

            jar = aiohttp.CookieJar(unsafe=True)
            connector = aiohttp.TCPConnector(ssl=False)
            timeout = aiohttp.ClientTimeout(total=10, connect=5)

            self.session = aiohttp.ClientSession(
                connector=connector, timeout=timeout, cookie_jar=jar
            )
        return self.session

    async def login(self) -> bool:
        """Авторизация в панели согласно PANEL_API_INSTRUCTIONS.md"""
        try:
            session = await self._get_session()

            # URL авторизации согласно инструкциям
            login_url = self._get_login_url()

            # Данные в формате application/x-www-form-urlencoded
            credentials = {"username": self.username, "password": self.password}

            headers = {"Content-Type": "application/x-www-form-urlencoded"}

            async with session.post(login_url, data=credentials, headers=headers) as response:
                if response.status == 200:
                    try:
                        response_data = await response.json()
                        if response_data.get("success"):
                            logger.info(f"✅ Авторизация успешна: {response.status}")

                            # Извлекаем cookie '3x-ui' из заголовков
                            set_cookie_headers = response.headers.getall("Set-Cookie", [])
                            import re

                            cookie_3x_ui = None

                            for cookie_header in set_cookie_headers:
                                if "3x-ui=" in cookie_header:
                                    match = re.search(r"3x-ui=([^;]+)", cookie_header)
                                    if match:
                                        cookie_3x_ui = match.group(1)
                                        break

                            if cookie_3x_ui:
                                self.auth_cookie = f"3x-ui={cookie_3x_ui}"
                                logger.info("🔐 Cookie '3x-ui' успешно извлечен")
                                return True
                            else:
                                logger.error("❌ Cookie '3x-ui' не найден")
                                return False
                        else:
                            logger.error(
                                f"❌ Ошибка авторизации: {response_data.get('msg', 'Неизвестная ошибка')}"
                            )
                            return False
                    except Exception as e:
                        response_text = await response.text()
                        logger.error(
                            f"❌ Ошибка парсинга ответа: {e}, текст: {response_text[:200]}..."
                        )
                        return False
                else:
                    logger.error(f"❌ HTTP ошибка авторизации: {response.status}")
                    return False

        except Exception as e:
            logger.error(f"❌ Ошибка при авторизации: {e}")
            return False

    async def create_client(self, email: str, traffic_gb: int = 15, days: int = 3, user_id: int = 0) -> Optional[str]:
        """
        Создать клиента в панели используя рабочий формат API

        Args:
            email: Email клиента
            traffic_gb: Лимит трафика в ГБ
            days: Срок действия в днях

        Returns:
            Optional[str]: ID созданного клиента или None при ошибке
        """
        try:
            # Авторизация
            if not await self.login():
                logger.error("Не удалось авторизоваться в панели")
                return None

            # Генерируем временный ID для создания, потом получим реальный
            temp_client_id = str(uuid.uuid4())
            expiry_time = int((datetime.now() + timedelta(days=days)).timestamp() * 1000)
            traffic_bytes = traffic_gb * 1024 * 1024 * 1024

            # Данные для VLESS клиента в inbound с ID = 1
            client_data = {
                "id": temp_client_id,
                "email": email,
                "limitIp": 0,
                "totalGB": traffic_bytes,
                "expiryTime": expiry_time,
                "enable": True,
                "tgId": str(user_id) if user_id else "",
                "subId": "",
                "flow": "xtls-rprx-vision",
            }

            # Формат для добавления клиента в существующий inbound
            api_data = {
                "id": 1,  # ID inbound для VLESS
                "settings": json.dumps({"clients": [client_data]}),
            }

            # API URL (правильный путь)
            api_url = self._get_api_url("inbounds/addClient")

            api_headers = {"Content-Type": "application/json", "Accept": "application/json"}

            session = await self._get_session()

            logger.info(f"🔧 Создание клиента: {email} (temp ID: {temp_client_id})")
            logger.info(f"📡 URL: {api_url}")
            logger.info(f"📦 Данные: {json.dumps(api_data, indent=2)}")

            async with session.post(api_url, json=api_data, headers=api_headers) as response:

                if response.status == 200:
                    # Сначала получаем текст ответа
                    text_response = await response.text()
                    logger.info(f"📨 Ответ панели: {text_response}")

                    # Проверяем, что в ответе содержится признак успеха
                    if '"success":true' in text_response or "success: true" in text_response:
                        logger.info(f"✅ Клиент успешно создан в панели: {email}")

                        # Получаем реальный ID клиента из панели
                        real_client_id = await self.get_real_client_id(email, temp_client_id)
                        if real_client_id:
                            logger.info(f"📝 Реальный ID клиента из панели: {real_client_id}")
                            return real_client_id
                        else:
                            logger.warning(
                                f"⚠️ Клиент создан, но не найден реальный ID. Используем временный: {temp_client_id}"
                            )
                            return temp_client_id
                    else:
                        # Пытаемся распарсить как JSON для получения ошибки
                        try:
                            result = json.loads(text_response)
                            logger.error(
                                f"❌ API вернул ошибку: {result.get('msg', 'Неизвестная ошибка')}"
                            )
                        except:
                            logger.error(f"❌ Неожиданный ответ панели: {text_response[:200]}")
                        return None
                else:
                    logger.error(f"❌ Ошибка API: статус {response.status}")
                    text_response = await response.text()
                    logger.error(f"Ответ: {text_response[:200]}")
                    return None

        except Exception as e:
            logger.error(f"❌ Ошибка при создании клиента: {e}")
            return None

    async def _is_html_response(self, text: str) -> bool:
        """Проверяет, является ли ответ HTML (означает устаревшую сессию)"""
        html_indicators = ['<html', '<div', '<form', 'text/html', 'login', '<title']
        text_lower = text.lower()
        return any(indicator in text_lower for indicator in html_indicators)

    async def _ensure_valid_session(self, max_retries: int = 2) -> bool:
        """
        Обеспечивает валидную сессию с автоматической переавторизацией
        
        Args:
            max_retries: Максимальное количество попыток авторизации
            
        Returns:
            bool: True если сессия валидна, иначе False
        """
        for attempt in range(max_retries + 1):
            if attempt > 0:
                logger.info(f"Попытка переавторизации {attempt}/{max_retries}")
                # Закрываем старую сессию и создаем новую
                if self.session:
                    await self.session.close()
                    self.session = None
                self.auth_cookie = None
            
            if await self.login():
                # Проверяем сессию простым API запросом
                try:
                    session = await self._get_session()
                    test_url = self._get_api_url("inbounds/list")
                    headers = {"Cookie": self.auth_cookie} if self.auth_cookie else {}
                    
                    async with session.post(test_url, headers=headers) as response:
                        if response.status == 200:
                            text = await response.text()
                            if not self._is_html_response(text):
                                logger.info(f"✅ Сессия валидна после попытки {attempt + 1}")
                                return True
                            else:
                                logger.warning(f"⚠️ Получен HTML ответ, сессия невалидна")
                        else:
                            logger.warning(f"⚠️ Тестовый запрос вернул статус {response.status}")
                except Exception as e:
                    logger.error(f"❌ Ошибка проверки сессии: {e}")
            
            logger.warning(f"❌ Попытка авторизации {attempt + 1} неудачна")
        
        logger.error("❌ Не удалось установить валидную сессию после всех попыток")
        return False

    async def update_client(self, client_uuid: str, email: str, days: int = 30, 
                          user_id: int = 0, subscription_id: int = 0) -> bool:
        """
        Надежное обновление клиента в панели с автоматической переавторизацией
        
        Args:
            client_uuid: UUID клиента в панели
            email: Email клиента
            days: Новый срок действия в днях
            user_id: ID пользователя Telegram
            subscription_id: ID подписки в базе данных
            
        Returns:
            bool: True если обновление успешно, иначе False
        """
        MAX_ATTEMPTS = 3
        
        for attempt in range(MAX_ATTEMPTS):
            try:
                logger.info(f"🔧 Попытка обновления клиента {attempt + 1}/{MAX_ATTEMPTS}: {client_uuid}")
                
                # Обеспечиваем валидную сессию
                if not await self._ensure_valid_session():
                    logger.error("❌ Не удалось установить валидную сессию")
                    continue

                # Вычисляем новое время истечения в миллисекундах
                from datetime import datetime, timedelta
                expiry_time = int((datetime.now() + timedelta(days=days)).timestamp() * 1000)

                # Данные для обновления клиента (точно как в Postman тесте)
                update_data = {
                    "id": client_uuid,
                    "email": email,
                    "limitIp": 5,
                    "totalGB": 0,
                    "expiryTime": expiry_time,
                    "enable": True,
                    "tgId": str(user_id) if user_id else "",
                    "subId": str(subscription_id) if subscription_id else "",
                    "flow": "xtls-rprx-vision",
                    "reset": 1
                }

                # Прямой API URL с UUID
                api_url = self._get_api_url(f"inbounds/updateClient/{client_uuid}")
                
                session = await self._get_session()
                headers = {
                    "Content-Type": "application/json", 
                    "Accept": "application/json",
                    "Cookie": self.auth_cookie
                }

                logger.info(f"📡 URL: {api_url}")
                logger.info(f"📦 Данные: {json.dumps(update_data, indent=2)}")

                async with session.post(api_url, json=update_data, headers=headers) as response:
                    if response.status == 200:
                        text_response = await response.text()
                        logger.info(f"📨 Ответ панели: {text_response}")

                        # Проверяем, не HTML ли это (означает устаревшую сессию)
                        if self._is_html_response(text_response):
                            logger.warning(f"⚠️ Получен HTML ответ, сессия устарела, попытка {attempt + 1}")
                            # Принудительно сбрасываем сессию для переавторизации
                            if self.session:
                                await self.session.close()
                                self.session = None
                            self.auth_cookie = None
                            continue

                        # Проверяем JSON успешность
                        try:
                            result = json.loads(text_response)
                            if result.get("success"):
                                logger.info(f"✅ Клиент успешно обновлен в панели: {client_uuid}")
                                return True
                            else:
                                logger.error(f"❌ API вернул ошибку: {result.get('msg', 'Неизвестная ошибка')}")
                                return False
                        except json.JSONDecodeError:
                            # Проверяем простым поиском строки в случае проблем с JSON
                            if '"success":true' in text_response or "success: true" in text_response:
                                logger.info(f"✅ Клиент успешно обновлен (по тексту): {client_uuid}")
                                return True
                            else:
                                logger.error(f"❌ Неожиданный ответ панели: {text_response[:200]}")
                                continue
                    else:
                        text_response = await response.text()
                        logger.error(f"❌ Ошибка API: статус {response.status}, ответ: {text_response[:200]}")
                        
                        # Если статус 401/403 - проблема с авторизацией
                        if response.status in [401, 403]:
                            logger.warning("⚠️ Ошибка авторизации, сбрасываем сессию")
                            if self.session:
                                await self.session.close()
                                self.session = None
                            self.auth_cookie = None
                            continue
                        else:
                            return False

            except Exception as e:
                logger.error(f"❌ Ошибка при попытке {attempt + 1}: {e}")
                if attempt == MAX_ATTEMPTS - 1:
                    return False
                continue
        
        logger.error(f"❌ Не удалось обновить клиента {client_uuid} после {MAX_ATTEMPTS} попыток")
        return False

    async def get_real_client_id(self, email: str, temp_id: str) -> Optional[str]:
        """
        Получает реальный ID клиента из панели по email используя правильный endpoint
        Повторяет попытки, так как панель может не успеть обновиться
        """
        max_attempts = 3
        for attempt in range(max_attempts):
            try:
                # Небольшая задержка для обновления панели
                if attempt > 0:
                    await asyncio.sleep(1)
                
                session = await self._get_session()

                # Используем правильный GET endpoint для получения inbound данных
                inbound_url = self._get_api_url("inbounds/get/1")
                headers = {"Content-Type": "application/json"}

                async with session.get(inbound_url, headers=headers) as response:
                    if response.status == 200:
                        try:
                            inbound_data = await response.json()

                            if inbound_data.get("success"):
                                obj = inbound_data.get("obj", {})
                                settings = obj.get("settings", "{}")

                                # Парсим settings как JSON
                                settings_json = json.loads(settings)
                                clients = settings_json.get("clients", [])

                                # Ищем клиента по email
                                for client in clients:
                                    if client.get("email") == email:
                                        real_id = client.get("id")
                                        sub_id = client.get("subId", "")

                                        logger.info(f"🎯 Найден клиент в панели:")
                                        logger.info(f"   Email: {email}")
                                        logger.info(f"   Real ID: {real_id}")
                                        logger.info(f"   Sub ID: {sub_id}")
                                        logger.info(f"   Enabled: {client.get('enable')}")

                                        return real_id

                                logger.warning(
                                    f"Клиент {email} не найден среди {len(clients)} клиентов (попытка {attempt + 1}/{max_attempts})"
                                )
                            else:
                                logger.error(
                                    f"API ошибка: {inbound_data.get('msg', 'Неизвестная ошибка')}"
                                )

                        except json.JSONDecodeError as e:
                            logger.error(f"Ошибка парсинга JSON: {e}")

                    else:
                        logger.error(f"HTTP ошибка: {response.status}")

            except Exception as e:
                logger.error(f"Ошибка при получении реального ID клиента (попытка {attempt + 1}/{max_attempts}): {e}")
        
        # Если все попытки неудачны, возвращаем None
        logger.error(f"Не удалось получить реальный ID клиента {email} после {max_attempts} попыток")
        return None

    async def get_client_by_email(self, email: str) -> Optional[dict]:
        """
        Получает информацию о клиенте по email через поиск в списке всех клиентов
        """
        try:
            clients = await self.get_clients()
            if clients:
                for client in clients:
                    if client.get("email") == email:
                        logger.info(f"Найден клиент с email {email}")
                        return client

                logger.info(f"Клиент с email {email} не найден")
                return None
            else:
                logger.error("Не удалось получить список клиентов")
                return None

        except Exception as e:
            logger.error(f"Ошибка при получении клиента по email {email}: {e}")
            return None

    async def get_client_by_uuid(self, uuid: str) -> Optional[dict]:
        """
        Получает информацию о клиенте по UUID через поиск в списке всех клиентов
        """
        try:
            clients = await self.get_clients()
            if clients:
                for client in clients:
                    if client.get("id") == uuid:
                        logger.info(f"Найден клиент с UUID {uuid}")
                        return client

                logger.info(f"Клиент с UUID {uuid} не найден")
                return None
            else:
                logger.error("Не удалось получить список клиентов")
                return None

        except Exception as e:
            logger.error(f"Ошибка при получении клиента по UUID {uuid}: {e}")
            return None

    async def get_clients(self) -> Optional[list]:
        """
        Получает список всех клиентов из панели с проверкой авторизации
        """
        try:
            # Сначала убеждаемся что авторизованы
            if not await self.login():
                logger.error("Не удалось авторизоваться в панели")
                return None

            session = await self._get_session()

            # Используем правильный endpoint согласно документации API
            inbound_url = self._get_api_url("inbounds/list")
            headers = {
                "Accept": "application/json",
                "Content-Type": "application/json"
            }
            
            # Добавляем cookie авторизации если есть
            if hasattr(self, 'auth_cookie') and self.auth_cookie:
                headers["Cookie"] = self.auth_cookie

            async with session.get(inbound_url, headers=headers) as response:
                if response.status == 200:
                    # HTML ответ = ошибка авторизации, НЕ ОБРАБАТЫВАЕМ HTML!
                    content_type = response.headers.get('content-type', '')
                    if 'application/json' not in content_type:
                        logger.error(f"❌ ОШИБКА АВТОРИЗАЦИИ: получен {content_type} вместо JSON!")
                        logger.error("❌ ПАНЕЛЬ ВЕРНУЛА HTML - АВТОРИЗАЦИЯ НЕДЕЙСТВИТЕЛЬНА")
                        return None
                        
                    try:
                        inbound_data = await response.json()
                    except Exception as e:
                        logger.error(f"❌ Ошибка парсинга JSON: {e}")
                        return None

                    if inbound_data.get("success"):
                        inbounds = inbound_data.get("obj", [])
                        all_clients = []

                        # Собираем клиентов из всех inbound'ов
                        for inbound in inbounds:
                            settings = inbound.get("settings", "{}")

                            # Парсим settings как JSON
                            try:
                                settings_json = json.loads(settings)
                                clients = settings_json.get("clients", [])

                                # Добавляем информацию о порте и протоколе к каждому клиенту
                                for client in clients:
                                    client["inbound_port"] = inbound.get("port")
                                    client["inbound_protocol"] = inbound.get("protocol")
                                    client["inbound_id"] = inbound.get("id")

                                all_clients.extend(clients)
                            except json.JSONDecodeError as e:
                                logger.warning(
                                    f"Ошибка парсинга settings для inbound {inbound.get('id')}: {e}"
                                )
                                continue

                        logger.info(
                            f"Получено {len(all_clients)} клиентов из {len(inbounds)} inbound'ов"
                        )
                        return all_clients
                    else:
                        logger.error(
                            f"API ошибка: {inbound_data.get('msg', 'Неизвестная ошибка')}"
                        )

                elif response.status == 403 or response.status == 401:
                    logger.error("Ошибка авторизации, пробуем переавторизоваться")
                    if await self.login():
                        # Рекурсивно вызываем себя один раз после переавторизации
                        return await self.get_clients()
                    else:
                        logger.error("Не удалось переавторизоваться")
                        return None
                else:
                    logger.error(f"HTTP ошибка: {response.status}")
                    response_text = await response.text()
                    logger.error(f"Ответ сервера: {response_text[:200]}")

            return None

        except Exception as e:
            logger.error(f"Ошибка при получении списка клиентов: {e}")
            return None

    async def get_client_vpn_key(self, client_id: str, email: str) -> Optional[str]:
        """
        Получить VPN ключ для клиента

        Args:
            client_id: UUID клиента
            email: Email клиента для проверки

        Returns:
            Optional[str]: VPN ключ в формате vless:// или None при ошибке
        """
        try:
            # Авторизуемся заново для получения актуальных cookies
            login_success = await self.login()
            if not login_success:
                logger.error("❌ Не удалось авторизоваться для получения VPN ключа")
                return None

            session = await self._get_session()

            # Получаем настройки inbound для формирования ключа
            inbound_url = self._get_api_url("inbounds/get/1")

            async with session.get(inbound_url) as response:
                if response.status == 200:
                    try:
                        inbound_data = await response.json()

                        if inbound_data.get("success"):
                            obj = inbound_data.get("obj", {})

                            # Получаем настройки Reality
                            stream_settings = json.loads(obj.get("streamSettings", "{}"))
                            reality_settings = stream_settings.get("realitySettings", {})

                            # Получаем данные клиента
                            settings = json.loads(obj.get("settings", "{}"))
                            clients = settings.get("clients", [])

                            # Находим нашего клиента
                            client_data = None
                            for client in clients:
                                if client.get("id") == client_id and client.get("email") == email:
                                    client_data = client
                                    break

                            if not client_data:
                                logger.error(f"❌ Клиент {email} с ID {client_id} не найден")
                                return None

                            # Формируем VPN ключ
                            server_ip = "31.172.75.92"  # IP сервера
                            port = obj.get("port", 443)

                            # СТРОГИЕ ПРАВИЛА: Используем фиксированные параметры Reality для всех VLESS ключей
                            # pbk всегда один и тот же для всех пользователей
                            public_key = "J7O8WbpqHE4mrSwOZZSb0zWc5Wx5ptCJzyfnVAi0pRw"
                            sni = "yahoo.com"
                            short_id = "cec02ca306"

                            # Формируем VLESS ключ с фиксированными параметрами согласно строгим правилам
                            vless_key = (
                                f"vless://{client_id}@{server_ip}:{port}"
                                f"?type=tcp&security=reality&pbk={public_key}"
                                f"&fp=chrome&sni={sni}&sid={short_id}"
                                f"&spx=%2F&flow=xtls-rprx-vision"
                                f"#RazDvaVPN%20Vless-{email.replace('@', '%40')}"
                            )

                            logger.info(f"✅ VPN ключ создан для {email}")
                            return vless_key
                        else:
                            logger.error(
                                f"❌ Ошибка API при получении ключа: {inbound_data.get('msg')}"
                            )
                            return None

                    except json.JSONDecodeError as e:
                        logger.error(f"❌ Ошибка парсинга JSON: {e}")
                        return None
                else:
                    logger.error(f"❌ HTTP ошибка при получении ключа: {response.status}")
                    return None

        except Exception as e:
            logger.error(f"❌ Ошибка при создании VPN ключа: {e}")
            return None

    async def get_client_traffic_stats(self, client_id: str) -> Optional[dict]:
        """
        Получить статистику трафика клиента

        Args:
            client_id: ID клиента

        Returns:
            Optional[dict]: Статистика трафика или None при ошибке
        """
        try:
            # Авторизуемся заново для получения актуальных cookies
            login_success = await self.login()
            if not login_success:
                logger.error("❌ Не удалось авторизоваться для получения статистики")
                return None

            session = await self._get_session()

            # Используем endpoint для получения статистики трафика
            traffic_url = self._get_api_url("inbounds/getClientTrafficsById/1")

            async with session.get(traffic_url) as response:
                if response.status == 200:
                    try:
                        # Сначала получаем текст ответа
                        response_text = await response.text()

                        # Пытаемся парсить как JSON
                        try:
                            traffic_data = json.loads(response_text)
                        except json.JSONDecodeError:
                            # Если не JSON, логируем и возвращаем None
                            logger.warning(
                                f"Ответ панели не в формате JSON: {response_text[:200]}..."
                            )
                            return None

                        if traffic_data.get("success"):
                            traffics = traffic_data.get("obj", [])

                            # Ищем статистику для нашего клиента
                            for traffic in traffics:
                                if traffic.get("email") == client_id:  # Проверяем по ID
                                    return {
                                        "up": traffic.get("up", 0),
                                        "down": traffic.get("down", 0),
                                        "total": traffic.get("total", 0),
                                        "enable": traffic.get("enable", True),
                                    }

                            # Если клиент не найден в статистике, возвращаем нулевые значения
                            return {"up": 0, "down": 0, "total": 0, "enable": True}
                        else:
                            logger.error(
                                f"❌ Ошибка API при получении статистики: {traffic_data.get('msg')}"
                            )
                            return None

                    except json.JSONDecodeError as e:
                        logger.error(f"❌ Ошибка парсинга JSON статистики: {e}")
                        return None
                else:
                    logger.error(f"❌ HTTP ошибка при получении статистики: {response.status}")
                    return None

        except Exception as e:
            logger.error(f"❌ Ошибка при получении статистики трафика: {e}")
            return None

    async def _get_traffic_alternative_method(self, session, client_email: str):
        """Альтернативный метод получения трафика через список inbounds"""
        try:
            # Получаем список inbounds
            inbound_url = self._get_api_url("inbounds/list")
            
            async with session.post(inbound_url) as response:
                if response.status == 200:
                    response_text = await response.text()
                    
                    try:
                        inbound_data = json.loads(response_text)
                    except json.JSONDecodeError:
                        # Если JSON не парсится, возвращаем дефолтные значения
                        logger.warning("Не удалось получить данные трафика из панели")
                        return {"up": 0, "down": 0, "total": 0, "enable": True}
                    
                    if inbound_data.get("success"):
                        inbounds = inbound_data.get("obj", [])
                        
                        # Ищем клиента в статистике
                        for inbound in inbounds:
                            if 'clientStats' in inbound:
                                client_stats = inbound['clientStats']
                                for client in client_stats:
                                    if client.get('email') == client_email:
                                        return {
                                            "up": client.get('up', 0),
                                            "down": client.get('down', 0),
                                            "total": client.get('up', 0) + client.get('down', 0),
                                            "enable": client.get('enable', True)
                                        }
            
            # Если клиент не найден, возвращаем нулевые значения
            return {"up": 0, "down": 0, "total": 0, "enable": True}
            
        except Exception as e:
            logger.error(f"Ошибка альтернативного метода получения трафика: {e}")
            return {"up": 0, "down": 0, "total": 0, "enable": True}

    async def get_traffic_by_email(self, email: str) -> Optional[dict]:
        """
        Получить статистику трафика клиента по email согласно PANEL_API_INSTRUCTIONS
        URL: /panel/api/inbounds/getClientTraffics/{email}
        """
        try:
            # Авторизуемся заново для получения актуальных cookies
            login_success = await self.login()
            if not login_success:
                logger.error("❌ Не удалось авторизоваться для получения статистики")
                return None

            session = await self._get_session()

            # Используем правильный endpoint из PANEL_API_INSTRUCTIONS
            traffic_url = self._get_api_url(f"inbounds/getClientTraffics/{email}")

            async with session.get(traffic_url) as response:
                if response.status == 200:
                    try:
                        traffic_data = await response.json()

                        if traffic_data.get("success"):
                            obj = traffic_data.get("obj")
                            if obj:
                                up = obj.get("up", 0)
                                down = obj.get("down", 0)
                                total = up + down
                                
                                return {
                                    "up": up,
                                    "down": down,
                                    "total": total,
                                    "enable": obj.get("enable", True),
                                }
                            else:
                                # Нет данных о трафике - возвращаем нули
                                return {"up": 0, "down": 0, "total": 0, "enable": True}
                        else:
                            logger.warning(f"API ошибка для {email}: {traffic_data.get('msg', 'Unknown')}")
                            return {"up": 0, "down": 0, "total": 0, "enable": True}

                    except json.JSONDecodeError as e:
                        logger.error(f"❌ Ошибка парсинга JSON для {email}: {e}")
                        return None
                else:
                    logger.error(f"❌ HTTP ошибка при получении статистики для {email}: {response.status}")
                    return None

        except Exception as e:
            logger.error(f"❌ Ошибка при получении статистики трафика для {email}: {e}")
            return None

    async def update_client(
        self,
        client_uuid: str,
        email: str,
        traffic_gb: int = None,
        days: int = None,
        enabled: bool = None,
        user_id: int = None,
        subscription_id: int = None,
    ) -> bool:
        """
        Обновить настройки клиента (трафик, срок действия, статус) с улучшенной авторизацией

        Args:
            client_uuid: UUID клиента
            email: Email клиента
            traffic_gb: Новый лимит трафика в ГБ (если None - не изменяется)
            days: Продлить на количество дней (если None - не изменяется)
            enabled: Включить/отключить клиента (если None - не изменяется)

        Returns:
            bool: Успешность обновления
        """
        try:
            # Авторизуемся заново для получения актуальных cookies
            login_success = await self.login()
            if not login_success:
                logger.error("❌ Не удалось авторизоваться для обновления клиента")
                return False

            session = await self._get_session()

            # Получаем текущие настройки inbound с правильной авторизацией
            inbound_url = self._get_api_url("inbounds/get/1")
            headers = {
                "Accept": "application/json",
                "Content-Type": "application/json"
            }
            
            # Добавляем cookie авторизации
            if hasattr(self, 'auth_cookie') and self.auth_cookie:
                headers["Cookie"] = self.auth_cookie

            async with session.get(inbound_url, headers=headers) as response:
                if response.status == 200:
                    # HTML ответ = ошибка авторизации, НЕ ОБРАБАТЫВАЕМ HTML!
                    content_type = response.headers.get('content-type', '')
                    if 'application/json' not in content_type:
                        logger.error(f"❌ ОШИБКА АВТОРИЗАЦИИ inbound: получен {content_type} вместо JSON!")
                        logger.error("❌ ПАНЕЛЬ ВЕРНУЛА HTML - АВТОРИЗАЦИЯ НЕДЕЙСТВИТЕЛЬНА")
                        return False
                        
                    try:
                        inbound_data = await response.json()
                    except Exception as e:
                        logger.error(f"❌ Ошибка парсинга JSON inbound: {e}")
                        return False

                    if inbound_data.get("success"):
                        obj = inbound_data.get("obj", {})
                        settings_str = obj.get("settings", "{}")
                        settings = json.loads(settings_str)
                        clients = settings.get("clients", [])

                        # Находим нашего клиента
                        target_client = None
                        for client in clients:
                            if client.get("id") == client_uuid and client.get("email") == email:
                                target_client = client.copy()  # Копируем клиента

                                # Обновляем трафик если указан
                                if traffic_gb is not None:
                                    target_client["totalGB"] = (
                                        traffic_gb * 1024 * 1024 * 1024
                                    )  # Конвертируем GB в байты

                                # Обновляем срок действия если указан
                                if days is not None:
                                    from datetime import datetime, timedelta
                                    # Используем правильный расчет времени
                                    expiry_datetime = datetime.now() + timedelta(days=days)
                                    expiry_time = int(expiry_datetime.timestamp() * 1000)
                                    target_client["expiryTime"] = expiry_time

                                # Обновляем статус если указан
                                if enabled is not None:
                                    target_client["enable"] = enabled

                                # Обязательные поля согласно PANEL_API_INSTRUCTIONS.md
                                target_client["limitIp"] = 5
                                target_client["reset"] = 1  # ОБНУЛЯЕТ трафик при обновлении
                                target_client["flow"] = "xtls-rprx-vision"  # ОБЯЗАТЕЛЬНО
                                
                                # Передаем tgId и subId согласно инструкции
                                if user_id is not None:
                                    target_client["tgId"] = str(user_id)
                                else:
                                    target_client["tgId"] = ""  # Пустая строка по умолчанию
                                    
                                if subscription_id is not None:
                                    target_client["subId"] = str(subscription_id)
                                # Если subId не передан, оставляем существующий

                                break

                        if target_client is None:
                            logger.error(f"❌ Клиент {email} с UUID {client_uuid} не найден")
                            return False

                        # Формируем правильные данные для обновления согласно API
                        update_data = {
                            "id": 1,  # ID inbound
                            "settings": json.dumps({"clients": [target_client]}, separators=(",", ":"))
                        }

                        # Обновляем клиента через правильный API endpoint
                        update_url = self._get_api_url(f"inbounds/updateClient/{client_uuid}")
                        update_headers = {
                            "Content-Type": "application/json",
                            "Accept": "application/json"
                        }
                        
                        # Добавляем cookie авторизации
                        if hasattr(self, 'auth_cookie') and self.auth_cookie:
                            update_headers["Cookie"] = self.auth_cookie

                        async with session.post(
                            update_url, json=update_data, headers=update_headers
                        ) as update_response:
                            if update_response.status == 200:
                                # HTML ответ = ошибка авторизации, НЕ ОБРАБАТЫВАЕМ HTML!
                                content_type = update_response.headers.get('content-type', '')
                                if 'application/json' not in content_type:
                                    logger.error(f"❌ ОШИБКА АВТОРИЗАЦИИ update: получен {content_type} вместо JSON!")
                                    logger.error("❌ ПАНЕЛЬ ВЕРНУЛА HTML - АВТОРИЗАЦИЯ НЕДЕЙСТВИТЕЛЬНА")
                                    return False
                                    
                                try:
                                    result = await update_response.json()
                                    if result.get("success"):
                                        logger.info(f"✅ Клиент {email} успешно обновлен в панели")
                                        return True
                                    else:
                                        logger.error(
                                            f"❌ Ошибка API при обновлении: {result.get('msg')}"
                                        )
                                        return False
                                except json.JSONDecodeError as e:
                                    logger.error(f"❌ Ошибка парсинга JSON update: {e}")
                                    return False
                                    logger.info(f"✅ Клиент {email} обновлен (ответ не JSON)")
                                    return True
                            else:
                                logger.error(
                                    f"❌ HTTP ошибка при обновлении: {update_response.status}"
                                )
                                return False
                    else:
                        logger.error(
                            f"❌ Ошибка получения настроек inbound: {inbound_data.get('msg')}"
                        )
                        return False
                else:
                    logger.error(f"❌ HTTP ошибка при получении inbound: {response.status}")
                    return False

        except Exception as e:
            logger.error(f"❌ Ошибка при обновлении клиента: {e}")
            return False

    async def reset_client_traffic(self, email: str) -> bool:
        """
        Сбросить статистику трафика клиента (обнулить использованный трафик)

        Args:
            email: Email клиента

        Returns:
            bool: Успешность сброса
        """
        try:
            # Авторизуемся заново для получения актуальных cookies
            login_success = await self.login()
            if not login_success:
                logger.error("❌ Не удалось авторизоваться для сброса трафика")
                return False

            session = await self._get_session()

            # Формируем URL для сброса трафика
            reset_url = self._get_api_url("inbounds/1/resetClientTraffic/{email}")
            headers = {"Content-Type": "application/json"}

            logger.info(f"🔄 Сбрасываем трафик для клиента: {email}")

            async with session.post(reset_url, headers=headers) as response:
                if response.status == 200:
                    try:
                        result = await response.json()
                        if result.get("success"):
                            logger.info(f"✅ Трафик клиента {email} успешно сброшен")
                            return True
                        else:
                            logger.error(f"❌ Ошибка API при сбросе трафика: {result.get('msg')}")
                            return False
                    except json.JSONDecodeError:
                        # Если не JSON, но статус 200 - считаем успехом
                        logger.info(f"✅ Трафик клиента {email} сброшен (ответ не JSON)")
                        return True
                else:
                    logger.error(f"❌ HTTP ошибка при сбросе трафика: {response.status}")
                    return False

        except Exception as e:
            logger.error(f"❌ Ошибка при сбросе трафика клиента: {e}")
            return False

    async def delete_client_by_email(self, email: str) -> bool:
        """
        Удалить клиента по email без избыточного парсинга JSON
        
        Args:
            email: Email клиента для удаления
            
        Returns:
            bool: Успешность удаления
        """
        try:
            # Авторизуемся
            login_success = await self.login()
            if not login_success:
                logger.error("❌ Не удалось авторизоваться для удаления клиента")
                return False

            session = await self._get_session()
            
            # Используем прямой API endpoint для удаления клиента по email
            # Согласно API панели 3X-UI: /panel/api/inbounds/delDepletedClients/{inbound_id}
            # Но для конкретного клиента используем /panel/api/inbounds/{inbound_id}/delClient/{email}
            delete_url = self._get_api_url("inbounds/1/delClient/{email}")
            headers = {"Content-Type": "application/json"}
            
            logger.info(f"🗑️ Удаляем клиента по email: {email}")
            
            async with session.post(delete_url, headers=headers) as response:
                if response.status == 200:
                    try:
                        result = await response.json()
                        if result.get("success"):
                            logger.info(f"✅ Клиент {email} успешно удален из панели")
                            return True
                        else:
                            logger.warning(f"⚠️ API ответ: {result.get('msg', 'Неизвестная ошибка')}")
                            return False
                    except json.JSONDecodeError:
                        # Если не JSON, но статус 200 - считаем успехом
                        logger.info(f"✅ Клиент {email} удален (ответ не JSON)")
                        return True
                else:
                    logger.error(f"❌ HTTP ошибка при удалении: {response.status}")
                    return False
                    
        except Exception as e:
            logger.error(f"❌ Ошибка при удалении клиента {email}: {e}")
            return False

    async def close(self):
        """Закрыть сессию"""
        if self.session:
            await self.session.close()


# Глобальный экземпляр рабочего API
working_api = WorkingPanelAPI()


async def create_client_in_panel(email: str, traffic_gb: int = 15, days: int = 3) -> Optional[str]:
    """
    Удобная функция для создания клиента в панели

    Args:
        email: Email клиента
        traffic_gb: Лимит трафика в ГБ
        days: Срок действия в днях

    Returns:
        Optional[str]: ID созданного клиента или None при ошибке
    """
    return await working_api.create_client(email, traffic_gb, days)


async def update_client_in_panel(
    email: str = None, 
    expiry_days: int = None,
    client_id: str = None, 
    traffic_limit: int = None, 
    expiry_time: int = None, 
    reset_traffic: bool = False,
    user_id: int = None,
    subscription_id: int = None
) -> bool:
    """
    Обновить клиента в панели с новыми лимитами и сроком действия
    Поддерживает как старый формат (client_id, traffic_limit) так и новый (email, expiry_days)

    Args:
        email: Email клиента для обновления (новый формат)
        expiry_days: Срок действия в днях (новый формат)
        client_id: ID клиента в панели (старый формат)
        traffic_limit: Новый лимит трафика в байтах (старый формат)
        expiry_time: Время истечения в миллисекундах (старый формат)
        reset_traffic: Сбросить использованный трафик
        user_id: Telegram ID пользователя для передачи в панель
        subscription_id: ID подписки для передачи в панель

    Returns:
        bool: True если обновление успешно
    """
    try:
        # Обработка нового формата (email, expiry_days)
        if email and expiry_days is not None:
            logger.info(f"Обновление клиента через новый API: {email}, {expiry_days} дней")
            
            # Получаем данные клиента из панели для получения UUID
            client_data = await working_api.get_client_by_email(email)
            if not client_data:
                logger.error(f"❌ Клиент {email} не найден в панели")
                return False
                
            client_uuid = client_data.get('id')
            if not client_uuid:
                logger.error(f"❌ UUID клиента {email} не найден")
                return False
            
            # Используем новый API с передачей user_id и subscription_id
            success = await working_api.update_client(
                client_uuid=str(client_uuid),
                email=email,
                days=expiry_days,
                user_id=user_id,
                subscription_id=subscription_id
            )
            
            return success
            
        # Обработка старого формата (client_id, traffic_limit, expiry_time)
        elif client_id and traffic_limit is not None and expiry_time is not None:
            logger.info(f"Обновление клиента через старый API: {client_id}")
            
            # Получаем email клиента по его ID
            from bot.services.client_info import get_client_info

            client_info = await get_client_info(client_id)
            if not client_info:
                logger.error(f"❌ Не удалось получить информацию о клиенте {client_id}")
                return False

            client_email = client_info.get("email")
            if not client_email:
                logger.error(f"❌ Email не найден для клиента {client_id}")
                return False

            # Конвертируем байты в ГБ для working_api
            traffic_gb = traffic_limit // (1024 * 1024 * 1024)

            # Для старого API используем прямое обновление через updateClient API
            import time
            success = await working_api.update_client(
                client_uuid=client_id,
                email=client_email,
                traffic_gb=traffic_gb,
                # Конвертируем expiry_time из миллисекунд в дни от сейчас
                days=max(1, int((expiry_time - int(time.time() * 1000)) / (24 * 60 * 60 * 1000))),
                user_id=user_id,
                subscription_id=subscription_id
            )
            
            return success
        else:
            logger.error("❌ Неподдерживанные параметры для update_client_in_panel")
            return False
            
    except Exception as e:
        logger.error(f"❌ Ошибка при обновлении клиента в панели: {e}")
        return False
