"""
Модуль для управления трафиком VPN-подписок
"""
import logging
from datetime import datetime, timedelta
from typing import Dict, Optional

from aiogram import Bot, Router, F
from aiogram.types import CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext

from bot.database.models import User, Subscription
from bot.database.operations import (
    get_user, 
    get_user_subscription,
    update_subscription
)
from bot.database.postgres_operations import (
    get_user_postgres,
    get_user_subscription_postgres,
    update_subscription_postgres
)

logger = logging.getLogger(__name__)

# Пакеты дополнительного трафика
TRAFFIC_PACKAGES = {
    '10g': {'name': '10 ГБ', 'price': 150, 'size': 10},
    '25g': {'name': '25 ГБ', 'price': 350, 'size': 25},
    '50g': {'name': '50 ГБ', 'price': 600, 'size': 50},
    '100g': {'name': '100 ГБ', 'price': 1000, 'size': 100},
}

router = Router()


class TrafficManager:
    """Менеджер управления дополнительным трафиком"""
    
    def __init__(self):
        self.packages = TRAFFIC_PACKAGES
    
    def get_packages(self):
        """Возвращает доступные пакеты трафика"""
        return self.packages
    
    async def add_traffic(self, user_id: int, package_id: str) -> bool:
        """Добавляет трафик к подписке пользователя"""
        return await add_traffic_to_subscription(user_id, package_id)


@router.callback_query(F.data.startswith("buy_traffic:"))
async def process_buy_traffic(callback: CallbackQuery, state: FSMContext):
    """
    Обработчик нажатия на кнопку покупки дополнительного трафика
    
    Формат callback_data: "buy_traffic:<package_id>"
    Например: "buy_traffic:10g"
    """
    # Извлекаем ID пакета из callback_data
    package_id = callback.data.split(":")[1]
    
    if package_id not in TRAFFIC_PACKAGES:
        await callback.answer("Выбран недопустимый пакет трафика", show_alert=True)
        return
    
    package = TRAFFIC_PACKAGES[package_id]
    
    # Сохраняем выбранный пакет в состоянии
    await state.update_data(selected_traffic_package=package_id)
    
    # Информируем пользователя о выбранном пакете
    confirmation_text = (
        f"📦 <b>Дополнительный трафик: {package['name']}</b>\n\n"
        f"💰 Стоимость: <b>{package['price']} руб.</b>\n\n"
        f"Оплатив этот пакет, вы получите дополнительно {package['name']} трафика "
        f"к вашей текущей подписке. Трафик будет добавлен сразу после оплаты."
    )
    
    # Создаем клавиатуру для подтверждения покупки
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text=f"💳 Оплатить {package['price']} руб.", callback_data=f"confirm_traffic:{package_id}")],
        [InlineKeyboardButton(text="🔙 Назад", callback_data="cancel_traffic_purchase")]
    ])
    
    await callback.message.edit_text(confirmation_text, reply_markup=keyboard)
    await callback.answer()


@router.callback_query(F.data.startswith("confirm_traffic:"))
async def confirm_traffic_purchase(callback: CallbackQuery, state: FSMContext):
    """
    Обработчик подтверждения покупки дополнительного трафика
    
    Формат callback_data: "confirm_traffic:<package_id>"
    """
    # Извлекаем ID пакета из callback_data
    package_id = callback.data.split(":")[1]
    
    if package_id not in TRAFFIC_PACKAGES:
        await callback.answer("Выбран недопустимый пакет трафика", show_alert=True)
        return
    
    package = TRAFFIC_PACKAGES[package_id]
    
    # Информируем о начале процесса оплаты
    await callback.answer("Переходим к оплате...")
    
    # Создаем текст для экрана оплаты
    payment_text = (
        f"💳 <b>Оплата дополнительного трафика</b>\n\n"
        f"📦 Пакет: <b>{package['name']}</b>\n"
        f"💰 Сумма: <b>{package['price']} руб.</b>\n\n"
        f"Для оплаты воспользуйтесь одним из следующих способов:\n"
        f"• Банковская карта\n"
        f"• СБП (Система быстрых платежей)\n"
        f"• ЮMoney\n\n"
        f"После успешной оплаты трафик будет добавлен автоматически."
    )
    
    # Здесь будет интеграция с платежной системой
    # После успешной оплаты нужно будет вызывать метод add_traffic_to_subscription
    
    # Временная заглушка - считаем, что оплата прошла успешно
    # В реальной системе этот вызов должен происходить после подтверждения оплаты
    await add_traffic_to_subscription(callback.from_user.id, package_id)
    
    success_text = (
        f"✅ <b>Трафик успешно добавлен!</b>\n\n"
        f"Вы приобрели дополнительно {package['name']} трафика. "
        f"Ваша подписка обновлена, вы можете продолжать пользоваться VPN."
    )
    
    # Создаем клавиатуру с кнопкой возврата в меню
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🔙 Вернуться в меню", callback_data="back_to_main")]
    ])
    
    await callback.message.edit_text(success_text, reply_markup=keyboard)


@router.callback_query(F.data == "cancel_traffic_purchase")
async def cancel_traffic_purchase(callback: CallbackQuery, state: FSMContext):
    """Отмена покупки дополнительного трафика"""
    await callback.answer("Покупка трафика отменена")
    
    # Очищаем данные о выбранном пакете
    await state.update_data(selected_traffic_package=None)
    
    # Возвращаемся в меню выбора пакетов
    traffic_packages_text = (
        "📦 <b>Выберите пакет дополнительного трафика:</b>\n\n"
        "Вы можете приобрести дополнительный трафик для вашей VPN-подписки. "
        "Трафик будет добавлен к вашему текущему лимиту."
    )
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text=f"{TRAFFIC_PACKAGES['10g']['name']} - {TRAFFIC_PACKAGES['10g']['price']} руб.", callback_data="buy_traffic:10g"),
            InlineKeyboardButton(text=f"{TRAFFIC_PACKAGES['25g']['name']} - {TRAFFIC_PACKAGES['25g']['price']} руб.", callback_data="buy_traffic:25g")
        ],
        [
            InlineKeyboardButton(text=f"{TRAFFIC_PACKAGES['50g']['name']} - {TRAFFIC_PACKAGES['50g']['price']} руб.", callback_data="buy_traffic:50g"),
            InlineKeyboardButton(text=f"{TRAFFIC_PACKAGES['100g']['name']} - {TRAFFIC_PACKAGES['100g']['price']} руб.", callback_data="buy_traffic:100g")
        ],
        [InlineKeyboardButton(text="🔙 Назад", callback_data="back_to_main")]
    ])
    
    await callback.message.edit_text(traffic_packages_text, reply_markup=keyboard)


async def add_traffic_to_subscription(user_id: int, package_id: str) -> bool:
    """
    Добавляет дополнительный трафик к подписке пользователя
    
    Args:
        user_id: ID пользователя
        package_id: ID пакета трафика (например, '10g')
        
    Returns:
        bool: True если трафик успешно добавлен, False в случае ошибки
    """
    if package_id not in TRAFFIC_PACKAGES:
        logger.error(f"Недопустимый ID пакета трафика: {package_id}")
        return False
    
    try:
        # Получаем текущую подписку пользователя
        subscription = await get_user_subscription_postgres(user_id)
        if not subscription:
            logger.error(f"Не найдена активная подписка для пользователя {user_id}")
            return False
        
        # Добавляем трафик к текущему лимиту
        package = TRAFFIC_PACKAGES[package_id]
        additional_traffic_bytes = package['size'] * 1024 * 1024 * 1024  # Преобразуем ГБ в байты
        
        logger.info(
            f"Добавление {package['size']} ГБ трафика для пользователя {user_id}. "
            f"Текущий лимит: {subscription.traffic_limit / (1024*1024*1024):.2f} ГБ, "
            f"Использовано: {subscription.traffic_used / (1024*1024*1024):.2f} ГБ"
        )
        
        subscription.traffic_limit += additional_traffic_bytes
        
        # Обновляем подписку в базе данных
        await update_subscription_postgres(subscription)
        
        logger.info(
            f"Успешно добавлен трафик для пользователя {user_id}. "
            f"Новый лимит: {subscription.traffic_limit / (1024*1024*1024):.2f} ГБ"
        )
        
        return True
    except Exception as e:
        logger.error(f"Ошибка при добавлении трафика для пользователя {user_id}: {e}")
        return False


def register_handlers(dp):
    """Регистрация обработчиков для управления трафиком"""
    dp.include_router(router)