"""
Сервис для обновления временных sync_id на реальные UUID из панели 3X-UI
"""

import asyncio
import logging
from typing import Optional

from bot.database.operations import get_all_subscriptions, update_subscription
from bot.services.working_panel_api import WorkingPanelAPI
from config_keys import get_panel_credentials

logger = logging.getLogger(__name__)


async def update_sync_ids_to_real_uuids():
    """
    Обновляет все временные sync_id на реальные UUID из панели
    """
    try:
        logger.info("Начинаем обновление временных sync_id...")
        
        # Получаем все подписки
        subscriptions = await get_all_subscriptions()
        
        if not subscriptions:
            logger.info("Нет подписок для обновления")
            return
        
        # Инициализируем API панели
        panel_url, panel_username, panel_password = get_panel_credentials()
        panel_api = WorkingPanelAPI()
        
        if not await panel_api.login():
            logger.error("Не удалось авторизоваться в панели для обновления sync_id")
            return
        
        updated_count = 0
        
        for subscription in subscriptions:
            try:
                # Проверяем, является ли sync_id временным
                if subscription.vpn_user_id and subscription.vpn_user_id.startswith("sync_id_"):
                    logger.info(f"Найдена подписка с временным sync_id: {subscription.vpn_user_id}")
                    
                    # Получаем реальный UUID из панели по email
                    real_uuid = await get_real_uuid_from_panel(panel_api, subscription.vpn_email)
                    
                    if real_uuid and real_uuid != subscription.vpn_user_id:
                        # Обновляем запись в базе данных
                        subscription.vpn_user_id = real_uuid
                        await update_subscription(subscription)
                        
                        logger.info(f"✅ Обновлен sync_id для подписки {subscription.id}: {subscription.vpn_user_id} → {real_uuid}")
                        updated_count += 1
                    else:
                        logger.warning(f"⚠️ Не удалось получить реальный UUID для {subscription.vpn_email}")
                        
            except Exception as e:
                logger.error(f"Ошибка при обновлении подписки {subscription.id}: {e}")
                continue
        
        await panel_api.close()
        logger.info(f"Завершено обновление sync_id. Обновлено записей: {updated_count}")
        
    except Exception as e:
        logger.error(f"Ошибка при обновлении sync_id: {e}")


async def get_real_uuid_from_panel(panel_api: WorkingPanelAPI, email: str) -> Optional[str]:
    """
    Получает реальный UUID клиента из панели по email
    """
    try:
        # Получаем клиента по email
        client = await panel_api.get_client_by_email(email)
        
        if client and client.get("id"):
            logger.info(f"Найден реальный UUID для {email}: {client['id']}")
            return client["id"]
        else:
            logger.warning(f"Клиент с email {email} не найден в панели")
            return None
            
    except Exception as e:
        logger.error(f"Ошибка при получении UUID для {email}: {e}")
        return None


async def update_single_subscription_sync_id(subscription_id: int, email: str) -> bool:
    """
    Обновляет sync_id для одной подписки
    """
    try:
        # Инициализируем API панели
        panel_url, panel_username, panel_password = get_panel_credentials()
        panel_api = WorkingPanelAPI()
        
        if not await panel_api.login():
            logger.error("Не удалось авторизоваться в панели")
            return False
        
        # Получаем реальный UUID
        real_uuid = await get_real_uuid_from_panel(panel_api, email)
        
        if real_uuid:
            # Получаем подписку для обновления
            from bot.database.operations import get_subscription_by_id
            subscription = await get_subscription_by_id(subscription_id)
            
            if subscription:
                # Обновляем запись в базе данных
                subscription.vpn_user_id = real_uuid
                await update_subscription(subscription)
            
            logger.info(f"✅ Обновлен sync_id для подписки {subscription_id}: → {real_uuid}")
            await panel_api.close()
            return True
        else:
            logger.error(f"Не удалось получить реальный UUID для {email}")
            await panel_api.close()
            return False
            
    except Exception as e:
        logger.error(f"Ошибка при обновлении sync_id для подписки {subscription_id}: {e}")
        return False