"""
Строгий API для панели 3X-UI с обработкой ошибок
Используется в админских обработчиках
"""

import logging
import aiohttp
import json
from config import PANEL_API_URL, PANEL_USERNAME, PANEL_PASSWORD

logger = logging.getLogger(__name__)

class StrictPanelAPI:
    def __init__(self):
        self.base_url = PANEL_API_URL.rstrip('/')
        self.username = PANEL_USERNAME
        self.password = PANEL_PASSWORD
        self.session = None
        self.cookies = None

    async def _get_session(self):
        """Получить сессию с авторизацией"""
        if not self.session:
            self.session = aiohttp.ClientSession()
        
        if not self.cookies:
            success = await self._login()
            if not success:
                raise Exception("Не удалось авторизоваться в панели")
        
        return self.session

    async def _login(self):
        """Авторизация в панели"""
        try:
            login_data = {
                "username": self.username,
                "password": self.password
            }
            
            async with self.session.post(f"{self.base_url}/login", data=login_data) as response:
                if response.status == 200:
                    self.cookies = response.cookies
                    logger.info("Успешная авторизация в панели")
                    return True
                else:
                    logger.error(f"Ошибка авторизации: {response.status}")
                    return False
        except Exception as e:
            logger.error(f"Ошибка при авторизации: {e}")
            return False

    async def create_client(self, email: str, protocol: str = "vless", traffic_limit_gb: int = 5, days: int = 30, user_id: int = 0):
        """Создать нового клиента в панели"""
        try:
            session = await self._get_session()
            
            # Данные для создания клиента
            client_data = {
                "id": 1,  # ID инбаунда
                "settings": json.dumps({
                    "clients": [{
                        "id": f"client-{email}",
                        "email": email,
                        "limitIp": 2,
                        "totalGB": traffic_limit_gb * 1024 * 1024 * 1024,  # Конвертируем ГБ в байты
                        "expiryTime": days * 24 * 60 * 60 * 1000,  # Конвертируем дни в миллисекунды
                        "enable": True,
                        "tgId": str(user_id) if user_id else "",
                        "subId": ""
                    }]
                })
            }
            
            async with session.post(f"{self.base_url}/panel/api/inbounds/addClient", 
                                  json=client_data, 
                                  cookies=self.cookies) as response:
                if response.status == 200:
                    result = await response.json()
                    if result.get("success"):
                        logger.info(f"Клиент {email} успешно создан")
                        return {"success": True, "email": email}
                    else:
                        logger.error(f"Ошибка создания клиента: {result}")
                        return {"success": False, "error": result.get("msg", "Неизвестная ошибка")}
                else:
                    logger.error(f"HTTP ошибка при создании клиента: {response.status}")
                    return {"success": False, "error": f"HTTP {response.status}"}
                    
        except Exception as e:
            logger.error(f"Ошибка при создании клиента: {e}")
            return {"success": False, "error": str(e)}

    async def get_client_link(self, email: str, protocol: str = "vless"):
        """Получить ссылку клиента по email"""
        try:
            session = await self._get_session()
            
            # Получаем список инбаундов
            async with session.get(f"{self.base_url}/panel/api/inbounds/list", cookies=self.cookies) as response:
                if response.status == 200:
                    data = await response.json()
                    
                    if data.get("success"):
                        inbounds = data.get("obj", [])
                        
                        # Ищем клиента по email
                        for inbound in inbounds:
                            if inbound.get("settings"):
                                settings = json.loads(inbound["settings"])
                                clients = settings.get("clients", [])
                                
                                for client in clients:
                                    if client.get("email") == email:
                                        # Формируем ссылку
                                        client_id = client.get("id")
                                        if protocol.lower() == "vless":
                                            port = inbound.get("port")
                                            server = PANEL_API_URL.split('//')[1].split(':')[0]
                                            return f"vless://{client_id}@{server}:{port}?type=tcp&security=none#{email}"
                                        
                        return None
                    else:
                        logger.error("API вернул ошибку при получении инбаундов")
                        return None
                else:
                    logger.error(f"Ошибка запроса инбаундов: {response.status}")
                    return None
                    
        except Exception as e:
            logger.error(f"Ошибка при получении ссылки клиента: {e}")
            return None

    async def get_traffic_stats(self, email: str):
        """Получить статистику трафика по email"""
        try:
            session = await self._get_session()
            
            async with session.get(f"{self.base_url}/panel/api/inbounds/list", cookies=self.cookies) as response:
                if response.status == 200:
                    data = await response.json()
                    
                    if data.get("success"):
                        inbounds = data.get("obj", [])
                        
                        for inbound in inbounds:
                            if inbound.get("clientStats"):
                                for stat in inbound["clientStats"]:
                                    if stat.get("email") == email:
                                        return {
                                            "up": stat.get("up", 0),
                                            "down": stat.get("down", 0),
                                            "total": stat.get("total", 0),
                                            "enable": stat.get("enable", True)
                                        }
                        
                        return None
                    else:
                        return None
                else:
                    return None
                    
        except Exception as e:
            logger.error(f"Ошибка при получении статистики трафика: {e}")
            return None

    async def close(self):
        """Закрыть сессию"""
        if self.session:
            await self.session.close()

# Глобальный экземпляр
strict_panel_api = StrictPanelAPI()

# Функции для обратной совместимости
async def create_client(email: str, protocol: str = "vless", traffic_limit_gb: int = 5, days: int = 30):
    return await strict_panel_api.create_client(email, protocol, traffic_limit_gb, days)

async def get_client_link(email: str, protocol: str = "vless"):
    return await strict_panel_api.get_client_link(email, protocol)

async def get_traffic_stats(email: str):
    return await strict_panel_api.get_traffic_stats(email)