"""
Глобальный менеджер HTTP-сессий для предотвращения утечек памяти
"""

import aiohttp
import logging
from typing import Dict, Optional
import weakref

logger = logging.getLogger(__name__)

class GlobalSessionManager:
    """Глобальный менеджер для управления всеми aiohttp сессиями"""
    
    def __init__(self):
        self._sessions: Dict[str, aiohttp.ClientSession] = {}
        self._session_refs: Dict[str, weakref.ref] = {}
    
    async def get_session(self, name: str, **kwargs) -> aiohttp.ClientSession:
        """Получить или создать именованную сессию"""
        if name in self._sessions and not self._sessions[name].closed:
            return self._sessions[name]
        
        # Создаем новую сессию
        session = aiohttp.ClientSession(**kwargs)
        self._sessions[name] = session
        
        # Создаем weak reference для автоочистки
        def cleanup_callback(ref):
            if name in self._session_refs:
                del self._session_refs[name]
        
        self._session_refs[name] = weakref.ref(session, cleanup_callback)
        logger.debug(f"Создана HTTP-сессия: {name}")
        return session
    
    async def close_session(self, name: str):
        """Закрыть конкретную сессию"""
        if name in self._sessions:
            session = self._sessions[name]
            if not session.closed:
                await session.close()
                logger.debug(f"Закрыта HTTP-сессия: {name}")
            del self._sessions[name]
        
        if name in self._session_refs:
            del self._session_refs[name]
    
    async def close_all(self):
        """Закрыть все активные сессии"""
        for name, session in list(self._sessions.items()):
            if not session.closed:
                try:
                    await session.close()
                    logger.debug(f"Закрыта HTTP-сессия: {name}")
                except Exception as e:
                    logger.error(f"Ошибка закрытия сессии {name}: {e}")
        
        self._sessions.clear()
        self._session_refs.clear()
        logger.info("Все HTTP-сессии закрыты")
    
    def get_active_sessions(self) -> list:
        """Получить список активных сессий"""
        active = []
        for name, session in self._sessions.items():
            if not session.closed:
                active.append(name)
        return active
    
    async def __aenter__(self):
        return self
    
    async def __aexit__(self, exc_type, exc_val, exc_tb):
        await self.close_all()

# Глобальный экземпляр
global_session_manager = GlobalSessionManager()

async def get_global_session(name: str, **kwargs) -> aiohttp.ClientSession:
    """Удобная функция для получения глобальной сессии"""
    return await global_session_manager.get_session(name, **kwargs)

async def close_global_session(name: str):
    """Удобная функция для закрытия глобальной сессии"""
    await global_session_manager.close_session(name)

async def close_all_sessions():
    """Удобная функция для закрытия всех сессий"""
    await global_session_manager.close_all()