import asyncio
import logging
from typing import Dict, Any, Optional
from datetime import datetime, timedelta

# Fix distro compatibility issue before importing YooKassa
try:
    import distro
    # Add missing functions that YooKassa SDK expects
    if not hasattr(distro, 'name'):
        def name():
            try:
                return distro.id() or 'unknown'
            except:
                return 'linux'
        distro.name = name
    
    if not hasattr(distro, 'version'):
        def version():
            try:
                return distro.version() if hasattr(distro, 'version') else '1.0'
            except:
                return '1.0'
        distro.version = version
        
    if not hasattr(distro, 'like'):
        def like():
            return ''
        distro.like = like
        
    if not hasattr(distro, 'codename'):
        def codename():
            return ''
        distro.codename = codename
        
except ImportError:
    pass

from yookassa import Payment as YooKassaPayment, Configuration
from yookassa.domain.notification import WebhookNotification

from config import YOOKASSA_SHOP_ID, YOOKASSA_SECRET_KEY, YOOKASSA_RETURN_URL
from bot.database.postgres_operations import (
    create_payment_postgres, get_payment_by_payment_id_postgres,
    update_payment_status_postgres, get_user_subscription_postgres,
    update_subscription_postgres
)
from bot.database.models import Payment, PaymentStatus
# Импортируем функцию для обновления подписки в панели 3X-UI
from bot.services.panel_service import update_subscription_in_panel
from config import SUBSCRIPTION_PLANS

logger = logging.getLogger(__name__)

# Initialize YooKassa SDK
Configuration.account_id = YOOKASSA_SHOP_ID
Configuration.secret_key = YOOKASSA_SECRET_KEY


async def create_payment_for_plan(
    user_id: int, plan_id: str, amount: float,
    description: str, subscription_id: Optional[int] = None
) -> Dict[str, str]:
    """
    Create a payment for a subscription plan
    Returns a payment URL and payment ID
    """
    try:
        # Prepare payment data
        payment_data = {
            "amount": {
                "value": str(amount),
                "currency": "RUB"
            },
            "confirmation": {
                "type": "redirect",
                "return_url": YOOKASSA_RETURN_URL
            },
            "capture": True,
            "description": description,
            "metadata": {
                "user_id": str(user_id),
                "plan_id": plan_id,
                "subscription_id": str(subscription_id) if subscription_id else None
            }
        }
        
        # Create payment using YooKassa API (wrapped in asyncio for async compatibility)
        yookassa_payment = await asyncio.to_thread(
            YooKassaPayment.create,
            payment_data
        )
        
        payment_id = yookassa_payment.id
        confirmation_url = yookassa_payment.confirmation.confirmation_url
        
        # Store payment in database
        # Ensure subscription_id is valid integer or None
        validated_subscription_id = subscription_id if subscription_id and isinstance(subscription_id, int) else None
        
        payment = Payment(
            user_id=user_id,
            subscription_id=validated_subscription_id,
            amount=amount,
            currency="RUB",
            payment_id=payment_id,
            status=PaymentStatus.PENDING,
            plan_id=plan_id
        )
        
        await create_payment_postgres(payment)
        
        return {
            "payment_id": payment_id,
            "payment_url": confirmation_url
        }
    
    except Exception as e:
        logger.error(f"Error creating payment: {e}")
        raise


async def check_payment_status(payment_id: str) -> str:
    """
    Check payment status in YooKassa
    Returns one of: "pending", "succeeded", "canceled", "waiting_for_capture"
    """
    try:
        # For testing with test shop ID, simulate payment success after 30 seconds
        if YOOKASSA_SHOP_ID == "1087679":  # Test shop ID
            payment = await get_payment_by_payment_id_postgres(payment_id)
            if payment:
                time_since_creation = (datetime.now() - payment.created_at).total_seconds()
                if time_since_creation > 30:  # Simulate payment success after 30 seconds
                    logger.info(f"Test payment {payment_id} automatically marked as succeeded")
                    await update_payment_status_postgres(
                        payment_id,
                        PaymentStatus.SUCCEEDED
                    )
                    await _process_successful_payment(payment_id, {
                        'user_id': str(payment.user_id),
                        'plan_id': payment.plan_id,
                        'subscription_id': str(payment.subscription_id) if payment.subscription_id else None
                    })
                    return "succeeded"
                else:
                    return "pending"
        
        # Get payment details from YooKassa API
        yookassa_payment = await asyncio.to_thread(
            YooKassaPayment.find_one,
            payment_id
        )
        
        if not yookassa_payment:
            logger.error(f"Payment {payment_id} not found")
            return "not_found"
        
        status = yookassa_payment.status
        logger.info(f"Payment {payment_id} status: {status}")
        
        # Update status in database
        if status == "succeeded":
            await update_payment_status_postgres(
                payment_id,
                PaymentStatus.SUCCEEDED
            )
            
            # Process successful payment
            await _process_successful_payment(payment_id, yookassa_payment.metadata)
        
        elif status == "canceled":
            await update_payment_status(
                payment_id,
                PaymentStatus.CANCELED,
                datetime.now()
            )
        
        return status
    
    except Exception as e:
        logger.error(f"Error checking payment status: {e}")
        return "error"


async def _process_successful_payment(payment_id: str, metadata: Dict[str, str]) -> None:
    """Process a successful payment - extend or create subscription"""
    try:
        # Extract metadata
        user_id = int(metadata.get("user_id", 0))
        plan_id = metadata.get("plan_id")
        subscription_id = metadata.get("subscription_id")
        
        if subscription_id and subscription_id != "None":
            # Activate new subscription created during payment process
            subscription_id = int(subscription_id)
            from bot.database.operations import get_subscription_by_id
            subscription = await get_subscription_by_id(subscription_id)
            
            if not subscription:
                logger.error(f"Subscription {subscription_id} not found")
                return
            
            # Activate subscription
            subscription.status = "ACTIVE"
            await update_subscription(subscription)
            
            # Get VPN connection key from panel
            if subscription.vpn_email:
                try:
                    from bot.services.client_info import get_client_info
                    client_info = await get_client_info(subscription.vpn_email)
                    
                    if client_info and "link" in client_info:
                        subscription.vpn_key = client_info["link"]
                        await update_subscription(subscription)
                        
                        # Log successful activation (notification will be sent by bot handler)
                        logger.info(f"Subscription {subscription.id} activated for user {user_id}")
                        
                        logger.info(f"VPN key generated and sent to user {user_id}")
                    else:
                        logger.error(f"Could not get VPN key for {subscription.vpn_email}")
                except Exception as e:
                    logger.error(f"Error getting VPN key: {e}")
        
        else:
            # Legacy handling for existing subscriptions (renewals)
            subscription = await get_user_subscription(user_id)
            
            if subscription:
                # Get plan details
                plan = SUBSCRIPTION_PLANS.get(plan_id)
                if not plan:
                    logger.error(f"Plan {plan_id} not found")
                    return
                
                # Calculate new expiration date
                if datetime.now() > subscription.expires_at:
                    new_expires_at = datetime.now() + timedelta(days=plan["duration_days"])
                else:
                    new_expires_at = subscription.expires_at + timedelta(days=plan["duration_days"])
                
                # Update subscription
                subscription.plan_id = plan_id
                subscription.expires_at = new_expires_at
                subscription.traffic_limit = plan["traffic_gb"]
                subscription.traffic_used = 0
                subscription.status = "ACTIVE"
                
                await update_subscription(subscription)
        
        logger.info(f"Successfully processed payment {payment_id} for user {user_id}")
    
    except Exception as e:
        logger.error(f"Error processing successful payment: {e}")


async def handle_yookassa_notification(notification_data: Dict[str, Any]) -> bool:
    """Handle YooKassa webhook notification"""
    try:
        # Parse notification data
        notification_object = WebhookNotification(notification_data)
        
        # Extract payment information
        yookassa_payment = notification_object.object
        payment_id = yookassa_payment.id
        status = yookassa_payment.status
        
        logger.info(f"Received YooKassa notification for payment {payment_id} with status {status}")
        
        # Get payment from database
        payment = await get_payment_by_payment_id(payment_id)
        
        if not payment:
            logger.error(f"Payment {payment_id} not found in database")
            return False
        
        # Update payment status
        if status == "succeeded":
            await update_payment_status(
                payment_id,
                PaymentStatus.SUCCEEDED,
                datetime.now()
            )
            
            # Process successful payment
            await _process_successful_payment(
                payment_id,
                yookassa_payment.metadata
            )
        
        elif status == "canceled":
            await update_payment_status(
                payment_id,
                PaymentStatus.CANCELED,
                datetime.now()
            )
        
        return True
    
    except Exception as e:
        logger.error(f"Error processing YooKassa notification: {e}")
        return False


async def process_yookassa_notification(notification_data: dict) -> bool:
    """
    Обработка уведомления от YooKassa через веб-интерфейс
    
    Args:
        notification_data: Данные уведомления от YooKassa
        
    Returns:
        bool: True если уведомление обработано успешно
    """
    try:
        # Получаем тип уведомления и объект
        notification_type = notification_data.get('type')
        event = notification_data.get('event')  # Для совместимости
        payment_data = notification_data.get('object', {})
        
        # ID платежа
        payment_id = payment_data.get('id')
        
        if not payment_id:
            logger.error("Отсутствует ID платежа в уведомлении")
            return False
        
        # Статус платежа
        payment_status = payment_data.get('status')
        
        logger.info(f"Обработка уведомления типа {notification_type}, статус платежа: {payment_status}, ID: {payment_id}")
        
        # Обрабатываем различные статусы платежа
        if payment_status == 'succeeded':
            # Платеж успешно завершен
            await update_payment_status(
                payment_id,
                PaymentStatus.SUCCEEDED,
                datetime.now()
            )
            
            # Обрабатываем успешный платеж
            metadata = payment_data.get('metadata', {})
            await _process_successful_payment(payment_id, metadata)
            
        elif payment_status == 'canceled':
            # Платеж отменен
            await update_payment_status(
                payment_id,
                PaymentStatus.CANCELED,
                datetime.now()
            )
            logger.info(f"Платеж {payment_id} отменен")
            
        elif payment_status == 'waiting_for_capture':
            # Платеж ожидает подтверждения
            logger.info(f"Платеж {payment_id} ожидает подтверждения")
            
        elif payment_status == 'pending':
            # Платеж в обработке
            logger.info(f"Платеж {payment_id} в обработке")
            
        else:
            logger.info(f"Получен платеж со статусом {payment_status}")
            
        return True
        
    except Exception as e:
        logger.error(f"Ошибка при обработке уведомления YooKassa: {e}")
        return False
