import asyncio
import logging
import aiohttp
import time
from typing import Dict, Any, Optional
from datetime import datetime, timedelta

# Fix distro compatibility issue before importing YooKassa
try:
    import distro
    # Add missing functions that YooKassa SDK expects
    if not hasattr(distro, 'name'):
        def name():
            try:
                return distro.id() or 'unknown'
            except:
                return 'linux'
        distro.name = name
    
    if not hasattr(distro, 'version'):
        def version():
            try:
                return distro.version() if hasattr(distro, 'version') else '1.0'
            except:
                return '1.0'
        distro.version = version
        
    if not hasattr(distro, 'like'):
        def like():
            return ''
        distro.like = like
        
    if not hasattr(distro, 'codename'):
        def codename():
            return ''
        distro.codename = codename
        
except ImportError:
    pass

from yookassa import Payment as YooKassaPayment, Configuration
from yookassa.domain.notification import WebhookNotification

import sys
import os
sys.path.append(os.path.dirname(os.path.dirname(os.path.dirname(__file__))))
from config import YOOKASSA_SHOP_ID, YOOKASSA_SECRET_KEY, YOOKASSA_RETURN_URL, SUBSCRIPTION_PLANS
from bot.database.postgres_operations import (
    create_payment_postgres, get_payment_by_payment_id_postgres,
    update_payment_status_postgres, get_user_subscription_postgres,
    update_subscription_postgres, get_subscription_by_id_postgres
)
from bot.database.models import Payment, PaymentStatus

logger = logging.getLogger(__name__)

# Configure YooKassa
Configuration.account_id = YOOKASSA_SHOP_ID
Configuration.secret_key = YOOKASSA_SECRET_KEY


async def send_payment_retry_notification(user_id: int, attempt: int, max_retries: int, delay: int):
    """Отправляет уведомление пользователю о повторной попытке обработки платежа"""
    try:
        from main import bot
        
        if attempt == 1:
            message = (
                "⏳ <b>Обработка платежа</b>\n\n"
                "Возникли временные проблемы с соединением к платежной системе.\n"
                f"Автоматически повторяем запрос (попытка {attempt}/{max_retries})\n\n"
                f"⏱ Следующая попытка через {delay//60} мин {delay%60} сек\n"
                "Пожалуйста, ожидайте..."
            )
        else:
            message = (
                "🔄 <b>Повторная попытка обработки платежа</b>\n\n"
                f"Попытка {attempt}/{max_retries}\n"
                f"⏱ Следующая попытка через {delay//60} мин {delay%60} сек\n\n"
                "Система автоматически обрабатывает ваш платеж..."
            )
        
        await bot.send_message(user_id, message, parse_mode="HTML")
        logger.info(f"Sent payment retry notification to user {user_id} (attempt {attempt})")
        
    except Exception as e:
        logger.error(f"Failed to send payment retry notification: {e}")


async def retry_yookassa_request(func, *args, max_retries=3, base_delay=60.0, user_id=None, **kwargs):
    """
    Retry YooKassa API requests with exponential backoff
    Handles connection errors, timeouts, and other network issues
    Sends notifications to user about retry attempts
    """
    last_exception = None
    
    for attempt in range(max_retries):
        try:
            # Execute the function in a thread to avoid blocking
            result = await asyncio.to_thread(func, *args, **kwargs)
            logger.info(f"YooKassa request successful on attempt {attempt + 1}")
            
            # If there were retries and now success, notify user
            if attempt > 0 and user_id:
                try:
                    from main import bot
                    message = (
                        "✅ <b>Платеж успешно обработан!</b>\n\n"
                        "Соединение с платежной системой восстановлено.\n"
                        "Ваш платеж обработан корректно."
                    )
                    await bot.send_message(user_id, message, parse_mode="HTML")
                except Exception as e:
                    logger.error(f"Failed to send success notification: {e}")
            
            return result
            
        except Exception as e:
            last_exception = e
            error_msg = str(e).lower()
            
            # Check for connection-related errors
            if any(keyword in error_msg for keyword in [
                'connection is closed', 'connection closed', 'connection reset',
                'timeout', 'network', 'connection error', 'ssl error',
                'read timeout', 'connect timeout', 'connection refused',
                'ssl handshake failed', 'certificate verify failed'
            ]):
                if attempt < max_retries - 1:
                    delay = base_delay * (2 ** attempt)  # 60s, 120s, 240s
                    logger.warning(f"YooKassa connection error (attempt {attempt + 1}/{max_retries}): {e}")
                    logger.info(f"Retrying in {delay} seconds...")
                    
                    # Notify user about retry attempt
                    if user_id:
                        await send_payment_retry_notification(user_id, attempt + 1, max_retries, int(delay))
                    
                    await asyncio.sleep(delay)
                    continue
                else:
                    logger.error(f"YooKassa request failed after {max_retries} attempts: {e}")
                    
                    # Notify user about final failure
                    if user_id:
                        try:
                            from main import bot
                            message = (
                                "❌ <b>Ошибка обработки платежа</b>\n\n"
                                "Не удалось подключиться к платежной системе после нескольких попыток.\n\n"
                                "🔄 Попробуйте повторить платеж через несколько минут\n"
                                "📞 Или обратитесь в поддержку, если проблема повторится"
                            )
                            await bot.send_message(user_id, message, parse_mode="HTML")
                        except Exception as notification_error:
                            logger.error(f"Failed to send failure notification: {notification_error}")
            else:
                # For non-connection errors, don't retry
                logger.error(f"YooKassa request failed with non-retriable error: {e}")
                raise e
    
    # If we get here, all retries failed
    raise last_exception


async def create_payment_for_plan(
    user_id: int, plan_id: str, amount: float,
    description: str, subscription_id: Optional[int] = None
) -> Dict[str, str]:
    """
    Create a payment for a subscription plan
    Returns a payment URL and payment ID
    """
    try:
        # Prepare payment data
        payment_data = {
            "amount": {
                "value": str(amount),
                "currency": "RUB"
            },
            "confirmation": {
                "type": "redirect",
                "return_url": YOOKASSA_RETURN_URL
            },
            "capture": True,
            "description": description,
            "metadata": {
                "user_id": str(user_id),
                "plan_id": plan_id,
                "subscription_id": str(subscription_id) if subscription_id else None
            }
        }
        
        # Create payment using YooKassa API with retry mechanism
        yookassa_payment = await retry_yookassa_request(
            YooKassaPayment.create,
            payment_data,
            user_id=user_id
        )
        
        payment_id = yookassa_payment.id
        confirmation_url = yookassa_payment.confirmation.confirmation_url
        
        # Store payment in database
        # Ensure subscription_id is valid integer or None
        validated_subscription_id = subscription_id if subscription_id and isinstance(subscription_id, int) else None
        
        payment = Payment(
            user_id=user_id,
            subscription_id=validated_subscription_id,
            amount=amount,
            currency="RUB",
            payment_id=payment_id,
            status=PaymentStatus.PENDING,
            plan_id=plan_id
        )
        
        await create_payment_postgres(payment)
        
        return {
            "payment_id": payment_id,
            "payment_url": confirmation_url
        }
    
    except Exception as e:
        logger.error(f"Error creating payment: {e}")
        raise


async def check_payment_status(payment_id: str, user_id: int = None) -> str:
    """
    Check payment status in YooKassa
    Returns one of: "pending", "succeeded", "canceled", "waiting_for_capture"
    """
    try:
        # For testing with test shop ID, simulate payment success after 30 seconds
        if YOOKASSA_SHOP_ID == "1087679":  # Test shop ID
            payment = await get_payment_by_payment_id_postgres(payment_id)
            if payment:
                time_since_creation = (datetime.now() - payment.created_at).total_seconds()
                if time_since_creation > 30:  # Simulate payment success after 30 seconds
                    logger.info(f"Test payment {payment_id} automatically marked as succeeded")
                    await update_payment_status_postgres(
                        payment_id,
                        "SUCCEEDED"
                    )
                    await _process_successful_payment(payment_id, {
                        'user_id': str(payment.user_id),
                        'plan_id': payment.plan_id,
                        'subscription_id': str(payment.subscription_id) if payment.subscription_id else None
                    })
                    return "succeeded"
                else:
                    return "pending"
        
        # Get payment details from YooKassa API with retry mechanism
        yookassa_payment = await retry_yookassa_request(
            YooKassaPayment.find_one,
            payment_id,
            user_id=user_id
        )
        
        if not yookassa_payment:
            logger.error(f"Payment {payment_id} not found")
            return "not_found"
        
        status = yookassa_payment.status
        logger.info(f"Payment {payment_id} status: {status}")
        
        # Update status in database
        if status == "succeeded":
            await update_payment_status_postgres(
                payment_id,
                "SUCCEEDED"
            )
            
            # Process successful payment
            await _process_successful_payment(payment_id, yookassa_payment.metadata)
        
        elif status == "canceled":
            await update_payment_status_postgres(
                payment_id,
                "CANCELED"
            )
        
        return status
    
    except Exception as e:
        logger.error(f"Error checking payment status: {e}")
        return "error"


async def _process_successful_payment(payment_id: str, metadata: Dict[str, str]) -> None:
    """Process a successful payment - extend or create subscription"""
    try:
        # Extract metadata
        user_id = int(metadata.get("user_id", 0))
        plan_id = metadata.get("plan_id")
        subscription_id = metadata.get("subscription_id")
        
        if subscription_id and subscription_id != "None":
            # Activate new subscription created during payment process
            subscription_id = int(subscription_id)
            subscription = await get_subscription_by_id_postgres(subscription_id)
            
            if not subscription:
                logger.error(f"Subscription {subscription_id} not found")
                return
            
            # Activate subscription
            await update_subscription_postgres(subscription_id, {
                'status': 'ACTIVE'
            })
            
            # Get VPN connection key from panel
            if subscription.vpn_email:
                try:
                    from bot.services.working_panel_api import get_client_link
                    vpn_key = await get_client_link(subscription.vpn_email)
                    
                    if vpn_key:
                        await update_subscription_postgres(subscription_id, {
                            'vpn_key': vpn_key
                        })
                        
                        # Log successful activation (notification will be sent by bot handler)
                        logger.info(f"Subscription {subscription.id} activated for user {user_id}")
                        logger.info(f"VPN key generated and sent to user {user_id}")
                    else:
                        logger.error(f"Could not get VPN key for {subscription.vpn_email}")
                except Exception as e:
                    logger.error(f"Error getting VPN key: {e}")
        
        else:
            # Legacy handling for existing subscriptions (renewals)
            subscription = await get_user_subscription_postgres(user_id)
            
            if subscription:
                # Get plan details
                plan = SUBSCRIPTION_PLANS.get(plan_id)
                if not plan:
                    logger.error(f"Plan {plan_id} not found")
                    return
                
                # Calculate new expiry date
                current_expiry = subscription.expires_at
                if current_expiry and current_expiry > datetime.now():
                    # Extend from current expiry
                    new_expiry = current_expiry + timedelta(days=plan['duration_days'])
                else:
                    # Start from now if expired
                    new_expiry = datetime.now() + timedelta(days=plan['duration_days'])
                
                # Update subscription
                await update_subscription_postgres(subscription.id, {
                    'expires_at': new_expiry,
                    'status': 'ACTIVE',
                    'traffic_limit': plan['traffic_limit']
                })
                
                logger.info(f"Subscription renewed for user {user_id} until {new_expiry}")
        
    except Exception as e:
        logger.error(f"Error processing successful payment: {e}")


async def handle_webhook(webhook_data: dict) -> bool:
    """Handle YooKassa webhook notification"""
    try:
        webhook_notification = WebhookNotification(webhook_data)
        payment = webhook_notification.object
        
        if payment.status == "succeeded":
            await update_payment_status_postgres(
                payment.id,
                PaymentStatus.SUCCEEDED
            )
            
            # Process successful payment
            await _process_successful_payment(payment.id, payment.metadata)
            return True
        
        elif payment.status == "canceled":
            await update_payment_status_postgres(
                payment.id,
                PaymentStatus.CANCELED
            )
            return True
        
        return False
    
    except Exception as e:
        logger.error(f"Error handling webhook: {e}")
        return False