"""
Оптимизированный API для панели 3X-UI
Используется для получения клиентских ссылок
"""

import logging
import aiohttp
from config import PANEL_API_URL, PANEL_USERNAME, PANEL_PASSWORD

logger = logging.getLogger(__name__)

class OptimizedPanelAPI:
    def __init__(self):
        self.base_url = PANEL_API_URL.rstrip('/')
        self.username = PANEL_USERNAME
        self.password = PANEL_PASSWORD
        self.session = None
        self.cookies = None

    async def _get_session(self):
        """Получить сессию с cookie"""
        if not self.session:
            self.session = aiohttp.ClientSession()
        
        if not self.cookies:
            await self._login()
        
        return self.session

    async def _login(self):
        """Авторизация в панели"""
        try:
            login_data = {
                "username": self.username,
                "password": self.password
            }
            
            async with self.session.post(f"{self.base_url}/login", data=login_data) as response:
                if response.status == 200:
                    self.cookies = response.cookies
                    logger.info("Успешная авторизация в панели")
                    return True
                else:
                    logger.error(f"Ошибка авторизации: {response.status}")
                    return False
        except Exception as e:
            logger.error(f"Ошибка при авторизации: {e}")
            return False

    async def get_client_link(self, email: str, protocol: str = "vless") -> str:
        """Получить ссылку клиента по email"""
        try:
            session = await self._get_session()
            
            # Получаем список инбаундов
            async with session.get(f"{self.base_url}/panel/api/inbounds/list", cookies=self.cookies) as response:
                if response.status == 200:
                    data = await response.json()
                    
                    if data.get("success"):
                        inbounds = data.get("obj", [])
                        
                        # Ищем клиента по email
                        for inbound in inbounds:
                            if inbound.get("settings"):
                                import json
                                settings = json.loads(inbound["settings"])
                                clients = settings.get("clients", [])
                                
                                for client in clients:
                                    if client.get("email") == email:
                                        # Формируем ссылку
                                        client_id = client.get("id")
                                        if protocol.lower() == "vless":
                                            port = inbound.get("port")
                                            return f"vless://{client_id}@{PANEL_API_URL.split('//')[1].split(':')[0]}:{port}?type=tcp&security=none#{email}"
                                        
                        logger.warning(f"Клиент с email {email} не найден")
                        return None
                    else:
                        logger.error("API вернул ошибку при получении инбаундов")
                        return None
                else:
                    logger.error(f"Ошибка запроса инбаундов: {response.status}")
                    return None
                    
        except Exception as e:
            logger.error(f"Ошибка при получении ссылки клиента: {e}")
            return None

    async def close(self):
        """Закрыть сессию"""
        if self.session:
            await self.session.close()

# Глобальный экземпляр
optimized_panel_api = OptimizedPanelAPI()

async def get_client_link(email: str, protocol: str = "vless") -> str:
    """Функция для получения ссылки клиента"""
    return await optimized_panel_api.get_client_link(email, protocol)