"""
Модуль для проверки существования клиентов в панели перед созданием
Реализует алгоритм защиты от дублирования клиентов
"""

import logging
import json
from typing import Optional, Dict, Any
from bot.services.working_panel_api import WorkingPanelAPI

logger = logging.getLogger(__name__)


async def check_client_exists_in_panel(email: str) -> Optional[Dict[str, Any]]:
    """
    Проверяет существование клиента в панели через getClientTraffics API
    
    Args:
        email: Email клиента для проверки
        
    Returns:
        Dict с данными клиента если существует, None если не найден
        
    Пример ответа при существовании клиента:
    {
        "success": true,
        "msg": "",
        "obj": {
            "id": 108,
            "inboundId": 1,
            "enable": true,
            "email": "278702313@VVShilin",
            "up": 0,
            "down": 0,
            "expiryTime": 1751208600000,
            "total": 161061273600,
            "reset": 1
        }
    }
    """
    try:
        # Инициализируем API панели с правильными учетными данными
        from config_keys import get_panel_credentials
        panel_url, panel_username, panel_password = get_panel_credentials()
        
        api = WorkingPanelAPI()
        api.base_url = panel_url
        api.username = panel_username
        api.password = panel_password
        
        # Авторизуемся
        if not await api.login():
            logger.error("Не удалось авторизоваться в панели для проверки клиента")
            return None
        
        # Получаем сессию для запросов
        session = await api._get_session()
        
        # Формируем URL для проверки клиента через getClientTraffics
        # Пример: https://31.172.75.92:47773/Sh1YLmgVQZ7cgcS/panel/api/inbounds/getClientTraffics/278702313@VVShilin
        # base_url уже содержит /panel/, поэтому добавляем только api/inbounds/getClientTraffics/
        if api.base_url.endswith('/'):
            check_url = f"{api.base_url}api/inbounds/getClientTraffics/{email}"
        else:
            check_url = f"{api.base_url}/api/inbounds/getClientTraffics/{email}"
        
        logger.info(f"Проверяем существование клиента: {email}")
        logger.info(f"URL запроса: {check_url}")
        
        async with session.get(check_url) as response:
            if response.status == 200:
                response_text = await response.text()
                logger.info(f"Ответ панели: {response_text}")
                
                try:
                    data = json.loads(response_text)
                    
                    if data.get("success") and data.get("obj"):
                        # Клиент существует в панели
                        client_info = data["obj"]
                        logger.info(f"✅ Клиент {email} найден в панели с ID: {client_info.get('id')}")
                        
                        return {
                            "exists": True,
                            "client_id": client_info.get("id"),
                            "email": client_info.get("email"),
                            "enabled": client_info.get("enable", True),
                            "traffic_limit": client_info.get("total", 0),
                            "traffic_used": (client_info.get("up", 0) + client_info.get("down", 0)),
                            "expiry_time": client_info.get("expiryTime", 0),
                            "inbound_id": client_info.get("inboundId", 1)
                        }
                    else:
                        # Клиент не найден
                        logger.info(f"🔍 Клиент {email} не найден в панели")
                        return None
                        
                except json.JSONDecodeError as e:
                    logger.error(f"Ошибка парсинга ответа панели: {e}")
                    return None
            else:
                logger.warning(f"Ошибка запроса к панели: HTTP {response.status}")
                response_text = await response.text()
                logger.warning(f"Ответ: {response_text[:200]}")
                return None
                
    except Exception as e:
        logger.error(f"Ошибка при проверке существования клиента {email}: {e}")
        return None


async def create_client_safely(email: str, traffic_gb: int, days: int, user_id: int = 0) -> Optional[str]:
    """
    Безопасное создание клиента с проверкой дублирования
    Сначала проверяет существование через getClientTraffics, 
    создает только если клиент не найден
    
    Args:
        email: Email клиента
        traffic_gb: Лимит трафика в ГБ
        days: Срок действия в днях
        user_id: Telegram ID пользователя
        
    Returns:
        str: ID созданного или существующего клиента, None при ошибке
    """
    try:
        logger.info(f"🔄 Начинаем безопасное создание клиента: {email}")
        
        # ЭТАП 1: Проверяем существование клиента
        existing_client = await check_client_exists_in_panel(email)
        
        if existing_client:
            logger.info(f"✅ Клиент {email} уже существует в панели, используем существующий")
            logger.info(f"📋 Данные клиента: ID={existing_client['client_id']}, "
                       f"enabled={existing_client['enabled']}, "
                       f"traffic_limit={existing_client['traffic_limit']} bytes")
            
            # Возвращаем ID существующего клиента
            return str(existing_client['client_id'])
        
        # ЭТАП 2: Клиент не найден, создаем новый
        logger.info(f"🆕 Клиент {email} не найден, создаем нового")
        
        # Используем рабочий API для создания с правильными учетными данными
        from config_keys import get_panel_credentials
        panel_url, panel_username, panel_password = get_panel_credentials()
        
        api = WorkingPanelAPI()
        api.base_url = panel_url
        api.username = panel_username
        api.password = panel_password
        
        client_id = await api.create_client(
            email=email,
            traffic_gb=traffic_gb,
            days=days,
            user_id=user_id
        )
        
        if client_id:
            logger.info(f"✅ Новый клиент создан: {email}, ID: {client_id}")
            
            # ЭТАП 3: Проверяем создание через getClientTraffics
            verification = await check_client_exists_in_panel(email)
            if verification:
                logger.info(f"✅ Создание подтверждено через getClientTraffics API")
                return str(verification['client_id'])
            else:
                logger.warning(f"⚠️ Клиент создан, но не подтвержден через getClientTraffics")
                return client_id
        else:
            # Проверяем, возможно клиент уже существует (ошибка дублирования)
            logger.info(f"🔄 Создание не удалось, повторно проверяем существование клиента")
            existing_check = await check_client_exists_in_panel(email)
            if existing_check:
                logger.info(f"✅ Клиент {email} найден при повторной проверке, используем существующий")
                return str(existing_check['client_id'])
            else:
                logger.error(f"❌ Не удалось создать клиента {email}")
                return None
            
    except Exception as e:
        logger.error(f"❌ Ошибка при безопасном создании клиента {email}: {e}")
        return None


async def get_existing_client_id(email: str) -> Optional[str]:
    """
    Получает ID существующего клиента в панели по email
    Упрощенная версия check_client_exists_in_panel только для получения ID
    
    Args:
        email: Email клиента для поиска
        
    Returns:
        str: ID клиента или None если не найден
    """
    try:
        existing_client = await check_client_exists_in_panel(email)
        if existing_client:
            return str(existing_client['client_id'])
        return None
    except Exception as e:
        logger.error(f"Ошибка при получении ID клиента {email}: {e}")
        return None


async def verify_client_creation(email: str, expected_traffic_gb: int) -> bool:
    """
    Проверяет успешность создания клиента через getClientTraffics API
    
    Args:
        email: Email клиента для проверки
        expected_traffic_gb: Ожидаемый лимит трафика в ГБ
        
    Returns:
        bool: True если клиент создан корректно
    """
    try:
        client_info = await check_client_exists_in_panel(email)
        
        if not client_info:
            logger.error(f"❌ Клиент {email} не найден при проверке")
            return False
        
        # Проверяем основные параметры
        traffic_limit_bytes = client_info.get('traffic_limit', 0)
        expected_bytes = expected_traffic_gb * 1024 * 1024 * 1024
        
        if abs(traffic_limit_bytes - expected_bytes) > 1024:  # Допустимая погрешность 1KB
            logger.warning(f"⚠️ Лимит трафика не совпадает: ожидали {expected_bytes}, получили {traffic_limit_bytes}")
        
        is_enabled = client_info.get('enabled', False)
        if not is_enabled:
            logger.warning(f"⚠️ Клиент {email} создан, но отключен")
        
        logger.info(f"✅ Проверка клиента {email} пройдена успешно")
        return True
        
    except Exception as e:
        logger.error(f"❌ Ошибка при проверке создания клиента {email}: {e}")
        return False