"""
Сервис для проверки существования клиента в панели 3X-UI
"""
import logging
from typing import Optional, Dict, Any
from bot.services.working_panel_api import WorkingPanelAPI

logger = logging.getLogger(__name__)


class ClientChecker:
    """Класс для проверки существования клиента в панели"""
    
    def __init__(self):
        self.panel_api = WorkingPanelAPI()
    
    async def check_client_exists(self, email: str) -> Optional[Dict[str, Any]]:
        """
        Проверяет существование клиента в панели по email
        
        Args:
            email: Email клиента для поиска
            
        Returns:
            Dict с данными клиента если найден, None если не найден
        """
        try:
            # Авторизуемся в панели
            if not await self.panel_api.login():
                logger.error("Не удалось авторизоваться в панели для проверки клиента")
                return None
            
            # Используем прямой запрос по email (более эффективно)
            client_info = await self.panel_api.get_client_by_email(email)
            if client_info:
                logger.info(f"Найден существующий клиент с email {email}")
                return client_info
            
            logger.info(f"Клиент с email {email} не найден в панели")
            return None
            
        except Exception as e:
            logger.error(f"Ошибка при проверке существования клиента {email}: {e}")
            return None
    
    async def get_client_config_url(self, client_data: Dict[str, Any]) -> Optional[str]:
        """
        Получает URL конфигурации для существующего клиента
        
        Args:
            client_data: Данные клиента из панели
            
        Returns:
            URL конфигурации или None
        """
        try:
            client_id = client_data.get('id')
            if not client_id:
                logger.error("ID клиента не найден в данных")
                return None
            
            # Получаем конфигурацию клиента
            config_url = await self.panel_api.get_client_config_url(client_id)
            if config_url:
                logger.info(f"Получен URL конфигурации для клиента {client_id}")
                return config_url
            else:
                logger.error(f"Не удалось получить URL конфигурации для клиента {client_id}")
                return None
                
        except Exception as e:
            logger.error(f"Ошибка при получении URL конфигурации: {e}")
            return None


async def get_client_config_by_subscription(subscription) -> Optional[Dict[str, Any]]:
    """
    Получает конфигурацию VPN ключа для подписки
    
    Args:
        subscription: Объект подписки
        
    Returns:
        Dict с данными конфигурации или None
    """
    try:
        from bot.services.working_panel_api import WorkingPanelAPI
        from config_keys import get_panel_credentials
        
        # Инициализируем API панели
        panel_url, panel_username, panel_password = get_panel_credentials()
        panel_api = WorkingPanelAPI()
        panel_api.base_url = panel_url
        panel_api.username = panel_username
        panel_api.password = panel_password
        
        # Авторизуемся
        if not await panel_api.login():
            logger.error("Не удалось авторизоваться в панели")
            return {'status': 'error', 'error': 'Ошибка авторизации в панели'}
        
        # Ищем клиента по email или UUID с использованием оптимизированных методов
        target_client = None
        search_email = subscription.vpn_email
        search_uuid = subscription.vpn_user_id
        
        # Сначала пробуем поиск по email (если есть)
        if search_email:
            target_client = await panel_api.get_client_by_email(search_email)
        
        # Если не найден по email, пробуем поиск по UUID (если есть)
        if not target_client and search_uuid:
            target_client = await panel_api.get_client_by_uuid(search_uuid)
        
        if not target_client:
            logger.error(f"Клиент не найден для подписки {subscription.id}")
            return {'status': 'error', 'error': 'Клиент не найден в панели'}
        
        # Получаем реальный VPN ключ через API панели
        client_id = target_client.get('id')
        email = target_client.get('email', subscription.vpn_email)
        
        # Генерируем VPN ключ через API панели
        vpn_key = await panel_api.get_client_vpn_key(client_id, email)
        
        if not vpn_key:
            logger.error(f"Не удалось сгенерировать VPN ключ для клиента {client_id}")
            return {'status': 'error', 'error': 'Не удалось сгенерировать VPN ключ'}
        
        vless_url = vpn_key
        
        return {
            'status': 'success',
            'vpn_key': vless_url,
            'config_url': vless_url,
            'client_id': client_id,
            'email': email
        }
        
    except Exception as e:
        logger.error(f"Ошибка при получении конфигурации: {e}")
        return {'status': 'error', 'error': str(e)}