"""
Улучшенный планировщик уведомлений с оптимизированной логикой
"""
import logging
import asyncio
from datetime import datetime, timedelta
from typing import Optional
from zoneinfo import ZoneInfo

from aiogram import Bot
from apscheduler.schedulers.asyncio import AsyncIOScheduler
from apscheduler.triggers.interval import IntervalTrigger
from apscheduler.triggers.cron import CronTrigger

from bot.services.notifications import (
    check_and_notify_expiring_subscriptions,
    check_and_notify_low_traffic,
    is_notification_time
)

logger = logging.getLogger(__name__)

# Московская временная зона
MOSCOW_TZ = ZoneInfo('Europe/Moscow')

def get_moscow_time():
    """Получить текущее время в московской временной зоне"""
    return datetime.now(MOSCOW_TZ)


class EnhancedNotificationScheduler:
    """Улучшенный планировщик для системы уведомлений"""
    
    def __init__(self, bot: Bot):
        self.bot = bot
        self.scheduler = AsyncIOScheduler(
            timezone='Europe/Moscow',
            job_defaults={
                'coalesce': False,
                'max_instances': 1,
                'misfire_grace_time': 300  # 5 минут
            }
        )
        self.is_running = False
        
    def setup_jobs(self):
        """Настройка заданий планировщика"""
        
        # Проверка тестовых подписок каждый час в :05 минут (в рабочее время)
        self.scheduler.add_job(
            self._check_trial_subscriptions_hourly,
            trigger=CronTrigger(minute=5, timezone=MOSCOW_TZ, hour='*'),
            id="trial_hourly_check", 
            name="Проверка тестовых подписок каждый час",
            replace_existing=True
        )
        
        # Проверка обычных подписок каждый день в 10:00 по московскому времени (7:00 UTC)
        self.scheduler.add_job(
            self._check_regular_subscriptions_daily,
            trigger=CronTrigger(hour=7, minute=0),  # 10:00 МСК = 7:00 UTC
            id="regular_daily_check",
            name="Ежедневная проверка подписок в 10:00 МСК",
            replace_existing=True
        )
        
        # Проверка трафика каждые 3 часа
        self.scheduler.add_job(
            self._check_traffic_usage,
            trigger=IntervalTrigger(hours=3),
            id="traffic_check",
            name="Проверка использования трафика каждые 3 часа",
            replace_existing=True
        )
        
        # Легкая синхронизация с панелью каждые 15 минут для лучшей производительности
        self.scheduler.add_job(
            self._sync_with_panel_light,
            trigger=IntervalTrigger(minutes=15),
            id="light_sync",
            name="Легкая синхронизация с панелью каждые 15 минут",
            replace_existing=True
        )
        
        logger.info("Настроены задания планировщика уведомлений")
    
    async def _check_trial_subscriptions_hourly(self):
        """Проверка тестовых подписок каждый час"""
        try:
            current_time = get_moscow_time()
            logger.info(f"[{current_time.strftime('%H:%M:%S MSK')}] ЕЖЕЧАСНАЯ ПРОВЕРКА: Запуск проверки тестовых подписок...")
            
            # Проверяем только в рабочее время
            if not await is_notification_time():
                logger.info(f"[{current_time.strftime('%H:%M:%S MSK')}] ЕЖЕЧАСНАЯ ПРОВЕРКА: Проверка тестовых подписок отложена - вне рабочего времени")
                return
                
            from bot.database.postgres_operations import get_expiring_subscriptions_postgres as get_expiring_subscriptions
            
            logger.info(f"[{current_time.strftime('%H:%M:%S MSK')}] Выполняется ежечасная проверка тестовых подписок...")
            
            # Получаем тестовые подписки, истекающие в ближайшие дни (0, 1, 2, 3 дня)
            trial_count = 0
            total_checked = 0
            
            for days in [0, 1, 2, 3]:
                trial_expiring = await get_expiring_subscriptions(days)
                total_checked += len(trial_expiring)
                
                for user, subscription in trial_expiring:
                    # Проверяем, является ли подписка тестовой (plan_id = "trial")
                    if subscription.plan_id == "trial":
                        # Проверяем, не отправлялось ли уведомление недавно
                        from bot.services.notifications import was_notification_recently_sent
                        if await was_notification_recently_sent(user.id, subscription.id, "expiry", days, hours=4):
                            logger.info(f"Уведомление пользователю {user.id} о подписке {subscription.id} (через {days} дней) уже отправлялось недавно, пропускаем")
                            continue
                            
                        trial_count += 1
                        logger.info(f"Отправка уведомления пользователю {user.id} о тестовой подписке (истекает через {days} дней)")
                        # Отправляем уведомление
                        from bot.services.notifications import notify_trial_expiring
                        await notify_trial_expiring(self.bot, user, subscription)
                        await asyncio.sleep(0.5)  # Задержка между уведомлениями
            
            if trial_count > 0:
                logger.info(f"[{current_time.strftime('%H:%M:%S MSK')}] Отправлено {trial_count} уведомлений о тестовых подписках из {total_checked} проверенных")
            else:
                logger.info("Нет тестовых подписок для уведомления")
                
        except Exception as e:
            logger.error(f"Ошибка при проверке тестовых подписок: {e}")
    
    async def _check_regular_subscriptions_daily(self):
        """Ежедневная проверка обычных подписок в 10:00"""
        try:
            logger.info("Выполняется ежедневная проверка подписок...")
            
            # Проверяем только в рабочее время
            if await is_notification_time():
                await check_and_notify_expiring_subscriptions(self.bot)
            else:
                logger.info("Проверка подписок отложена - вне рабочего времени")
                
        except Exception as e:
            logger.error(f"Ошибка при ежедневной проверке подписок: {e}")
    
    async def _check_traffic_usage(self):
        """Проверка использования трафика и синхронизация с панелью"""
        try:
            # Проверяем, включена ли задача проверки трафика
            if not await self._is_traffic_check_enabled():
                logger.debug("Проверка трафика отключена в настройках")
                return
                
            logger.debug("Выполняется проверка использования трафика...")
            
            # Синхронизируем трафик с панели
            await self._sync_traffic_from_panel()
            
        except Exception as e:
            logger.error(f"Ошибка при проверке трафика: {e}")
    
    async def _sync_traffic_from_panel(self):
        """Синхронизация трафика из панели 3X-UI в PostgreSQL"""
        try:
            from bot.services.working_panel_api import WorkingPanelAPI
            from config_keys import get_panel_credentials
            from sqlalchemy import create_engine, text
            import os
            
            # Подключаемся к PostgreSQL
            engine = create_engine(os.environ['DATABASE_URL'])
            
            with engine.connect() as conn:
                # Получаем активные подписки
                result = conn.execute(text("""
                    SELECT id, user_id, vpn_email, traffic_limit, traffic_used
                    FROM subscriptions 
                    WHERE status = 'ACTIVE' 
                    ORDER BY id
                """))
                
                subscriptions = result.fetchall()
                
                if not subscriptions:
                    logger.debug("Нет активных подписок для синхронизации трафика")
                    return
                
                logger.debug(f"Синхронизация трафика для {len(subscriptions)} подписок")
                
                # Инициализируем панель
                panel_url, panel_username, panel_password = get_panel_credentials()
                panel_api = WorkingPanelAPI(panel_url, panel_username, panel_password)
                
                if not await panel_api.login():
                    logger.error("Не удалось авторизоваться в панели для синхронизации трафика")
                    return
                
                updated_count = 0
                
                # Обрабатываем каждую подписку
                for sub in subscriptions:
                    sub_id = None
                    try:
                        sub_id, user_id, vpn_email, traffic_limit, current_traffic = sub
                        
                        # Получаем актуальный трафик из панели
                        traffic_data = await panel_api.get_traffic_by_email(vpn_email)
                        
                        if traffic_data:
                            panel_traffic = traffic_data.get('total', 0)
                            
                            # Обновляем только при существенной разности (больше 1 МБ)
                            if abs(panel_traffic - current_traffic) > 1048576:
                                conn.execute(text("""
                                    UPDATE subscriptions 
                                    SET traffic_used = :traffic_used
                                    WHERE id = :sub_id
                                """), {"traffic_used": panel_traffic, "sub_id": sub_id})
                                
                                logger.debug(f"Обновлен трафик для {vpn_email}: {current_traffic/1073741824:.3f} -> {panel_traffic/1073741824:.3f} ГБ")
                                updated_count += 1
                        
                    except Exception as e:
                        logger.warning(f"Ошибка синхронизации трафика для подписки {sub_id or 'неизвестная'}: {e}")
                
                if updated_count > 0:
                    conn.commit()
                    logger.info(f"Синхронизация трафика завершена: обновлено {updated_count} подписок")
                else:
                    logger.debug("Синхронизация трафика: обновлений не требуется")
                
        except Exception as e:
            logger.error(f"Ошибка синхронизации трафика: {e}")
    
    async def _check_and_notify_low_traffic(self, bot):
        """Проверка низкого трафика и отправка уведомлений"""
        try:
            from bot.services.working_panel_api import WorkingPanelAPI
            from config_keys import get_panel_credentials
            from sqlalchemy import create_engine, text
            import os
            
            # Используем PostgreSQL для получения подписок
            engine = create_engine(os.environ['DATABASE_URL'])
            
            with engine.connect() as conn:
                # Получаем активные подписки с низким остатком трафика
                result = conn.execute(text("""
                    SELECT id, user_id, vpn_email, traffic_limit, traffic_used, plan_id
                    FROM subscriptions 
                    WHERE status = 'ACTIVE' 
                    AND expires_at > NOW()
                    AND traffic_used > (traffic_limit * 0.8)
                    ORDER BY (traffic_used::float / traffic_limit) DESC
                """))
                
                low_traffic_subscriptions = result.fetchall()
                
                if not low_traffic_subscriptions:
                    logger.debug("Нет подписок с низким остатком трафика")
                    return
                
                logger.info(f"Найдено {len(low_traffic_subscriptions)} подписок с низким трафиком")
                
                for sub in low_traffic_subscriptions:
                    try:
                        usage_percent = (sub.traffic_used / sub.traffic_limit) * 100
                        remaining_gb = (sub.traffic_limit - sub.traffic_used) / (1024**3)
                        
                        # Отправляем предупреждение если осталось менее 20% трафика
                        if usage_percent > 80:
                            message = (
                                f"⚠️ Внимание! Трафик заканчивается\n\n"
                                f"Использовано: {usage_percent:.1f}%\n"
                                f"Осталось: {remaining_gb:.2f} ГБ\n\n"
                                f"Рекомендуем пополнить тариф или купить дополнительный трафик."
                            )
                            
                            await bot.send_message(sub.user_id, message)
                            logger.info(f"Отправлено предупреждение о трафике пользователю {sub.user_id}")
                            
                    except Exception as e:
                        logger.error(f"Ошибка отправки уведомления о трафике для подписки {sub.id}: {e}")
                        
        except Exception as e:
            logger.error(f"Ошибка проверки низкого трафика: {e}")
    
    async def _is_traffic_check_enabled(self):
        """Проверяет, включена ли задача проверки трафика"""
        try:
            from sqlalchemy import create_engine, text
            import os
            
            engine = create_engine(os.environ['DATABASE_URL'])
            
            with engine.connect() as conn:
                result = conn.execute(text(
                    "SELECT value FROM app_settings WHERE key = 'traffic_check_enabled'"
                ))
                row = result.fetchone()
                
                if row:
                    return row[0].lower() in ['true', '1', 'yes', 'on']
                else:
                    # По умолчанию включено
                    return True
        except Exception as e:
            logger.error(f"Ошибка при проверке настройки трафика: {e}")
            # По умолчанию включено при ошибке
            return True
    
    async def _sync_with_panel_light(self):
        """Легкая синхронизация с панелью"""
        try:
            from bot.services.light_sync import sync_traffic_lightweight
            
            logger.debug("Выполняется легкая синхронизация с панелью...")
            
            result = await sync_traffic_lightweight()
            
            if result["status"] == "success":
                if result["updated"] > 0:
                    logger.info(f"Синхронизация: обновлено {result['updated']} подписок")
            else:
                logger.warning(f"Проблема синхронизации: {result.get('message', 'Неизвестная ошибка')}")
                
        except Exception as e:
            logger.error(f"Ошибка при синхронизации с панелью: {e}")
    
    async def start(self):
        """Запуск планировщика"""
        if not self.is_running:
            self.setup_jobs()
            self.scheduler.start()
            self.is_running = True
            
            # Добавляем задачу мониторинга планировщика каждые 10 минут
            self.scheduler.add_job(
                self._scheduler_health_check,
                trigger=IntervalTrigger(minutes=10),
                id="scheduler_health_check",
                name="Проверка работоспособности планировщика",
                replace_existing=True
            )
            
            # Выполняем немедленную первичную проверку
            await self._initial_check()
            
            logger.info("Улучшенный планировщик уведомлений запущен с мониторингом")
            logger.info(f"Активные задачи планировщика: {[job.id for job in self.scheduler.get_jobs()]}")
    
    async def stop(self):
        """Остановка планировщика"""
        if self.is_running:
            self.scheduler.shutdown()
            self.is_running = False
            logger.info("Планировщик уведомлений остановлен")
    
    async def _initial_check(self):
        """Первичная проверка при запуске"""
        try:
            logger.info("Выполняется первичная проверка системы уведомлений...")
            
            # Проверяем трафик
            await self._check_traffic_usage()
            
            # Проверяем подписки только в рабочее время
            if await is_notification_time():
                await self._check_regular_subscriptions_daily()
                await self._check_trial_subscriptions_hourly()
            else:
                logger.info("Первичная проверка подписок отложена до рабочего времени")
            
            logger.info("Первичная проверка завершена")
            
        except Exception as e:
            logger.error(f"Ошибка при первичной проверке: {e}")
    
    async def _scheduler_health_check(self):
        """Проверка работоспособности планировщика"""
        try:
            current_time = get_moscow_time()
            logger.info(f"[{current_time.strftime('%H:%M:%S MSK')}] Проверка работоспособности планировщика...")
            
            # Проверяем активные задачи
            jobs = self.scheduler.get_jobs()
            active_jobs = [job.id for job in jobs]
            
            expected_jobs = [
                "trial_hourly_check", 
                "regular_daily_check", 
                "traffic_check", 
                "light_sync",
                "scheduler_health_check"
            ]
            
            missing_jobs = [job_id for job_id in expected_jobs if job_id not in active_jobs]
            
            if missing_jobs:
                logger.warning(f"Обнаружены отсутствующие задачи планировщика: {missing_jobs}")
                logger.info("Перезапуск отсутствующих задач...")
                self.setup_jobs()
            else:
                logger.info(f"Все задачи планировщика активны: {active_jobs}")
                
            # Проверяем статус планировщика
            if not self.scheduler.running:
                logger.error("КРИТИЧЕСКАЯ ОШИБКА: Планировщик не запущен!")
                logger.info("Попытка перезапуска планировщика...")
                self.scheduler.start()
                
        except Exception as e:
            logger.error(f"Ошибка при проверке работоспособности планировщика: {e}")
    
    def get_status(self) -> dict:
        """Получение статуса планировщика"""
        jobs = []
        if self.is_running:
            for job in self.scheduler.get_jobs():
                next_run = job.next_run_time
                jobs.append({
                    "id": job.id,
                    "name": job.name,
                    "next_run": next_run.isoformat() if next_run else None
                })
        
        return {
            "running": self.is_running,
            "jobs_count": len(jobs),
            "jobs": jobs
        }


# Глобальный экземпляр планировщика
enhanced_scheduler = None


async def setup_enhanced_notifications(bot: Bot):
    """
    Настройка улучшенной системы уведомлений
    
    Args:
        bot: Экземпляр бота
    """
    global enhanced_scheduler
    
    enhanced_scheduler = EnhancedNotificationScheduler(bot)
    await enhanced_scheduler.start()
    
    return enhanced_scheduler


def get_notification_status() -> dict:
    """Получение статуса системы уведомлений"""
    if enhanced_scheduler:
        return enhanced_scheduler.get_status()
    return {"running": False, "jobs_count": 0, "jobs": []}