"""
Refactored subscription handler with improved code structure and maintainability.
All functionality preserved while removing dead code and improving organization.
"""

from datetime import datetime, timedelta
import logging
from aiogram import Router, F, types
from aiogram.filters import StateFilter, Command
from aiogram.fsm.context import FSMContext
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.utils.keyboard import InlineKeyboardBuilder

from bot.utils.states import SubscriptionStates
from bot.database.operations import (
    get_user, get_user_subscription, create_subscription, update_subscription, 
    get_subscription_by_id, create_or_update_user, has_user_used_trial
)
from bot.database.postgres_operations import (
    get_user_postgres, get_user_subscription_postgres, create_subscription_postgres,
    update_subscription_postgres, get_subscription_by_id_postgres, 
    create_or_update_user_postgres, has_user_used_trial_postgres
)
from bot.database.models import User, Subscription, UserStatus
from bot.utils.keyboard import get_main_menu_keyboard, get_plans_keyboard, get_back_to_menu_inline
from bot.services.panel_service import create_vpn_account, get_vpn_key, get_vpn_link
from bot.services.payment import create_payment_for_plan
from bot.services.panel_session_manager import PanelSessionManager
from bot.services.client_checker import ClientChecker
from config import SUBSCRIPTION_PLANS

router = Router()
logger = logging.getLogger(__name__)


# Utility functions
async def get_user_id_from_message(message: Message) -> int:
    """Extract user ID from message object with proper validation"""
    if hasattr(message, 'from_user') and message.from_user:
        return message.from_user.id
    return message.chat.id


async def check_active_subscription(user_id: int) -> tuple[bool, Subscription | None]:
    """Check if user has active subscription and return status with subscription"""
    try:
        # Проверяем в PostgreSQL сначала
        subscription = await get_user_subscription_postgres(user_id)
        if not subscription:
            # Fallback к SQLite
            subscription = await get_user_subscription_postgres(user_id)
            
        if subscription and subscription.is_active:
            return True, subscription
        return False, subscription
    except Exception as e:
        logger.error(f"Error checking subscription for user {user_id}: {e}")
        return False, None


def format_active_subscription_message(subscription: Subscription) -> str:
    """Format message for users with active subscriptions"""
    # Используем функцию форматирования для корректного отображения
    from bot.utils.formatting import format_traffic_info
    
    total_gb, used_gb, remaining_gb = format_traffic_info(
        subscription.traffic_limit, 
        subscription.traffic_used
    )
    
    return (
        "⚠️ У вас уже есть активная подписка!\n\n"
        f"Ваша подписка действует еще {subscription.days_remaining} дней.\n"
        f"Осталось трафика: {remaining_gb} ГБ из {total_gb} ГБ.\n\n"
        "Если вы хотите продлить подписку, нажмите кнопку «Продлить Подписку»."
    )


def create_plan_confirmation_text(plan: dict) -> str:
    """Create confirmation text for selected plan"""
    return (
        f"🔍 <b>Выбран тариф: {plan['name']}</b>\n\n"
        f"📦 Трафик: {plan['traffic_gb']} ГБ\n"
        f"⏱️ Длительность: {plan['duration_days']} дней\n"
        f"💰 Стоимость: {plan['price']} руб.\n\n"
        f"Для продолжения оформления подписки нажмите кнопку «Оплатить»."
    )


# Handler implementations
@router.message(F.text == "Выбрать Тариф")
@router.message(Command("subscription"))
async def new_subscription(message: Message, state: FSMContext):
    """Handle new subscription request"""
    try:
        user_id = await get_user_id_from_message(message)
        has_active, subscription = await check_active_subscription(user_id)
        
        if has_active and subscription:
            await message.answer(
                format_active_subscription_message(subscription),
                reply_markup=get_main_menu_keyboard()
            )
            return
        
        # Show paid plans selection
        trial_kb = InlineKeyboardBuilder()
        trial_kb.button(text="🛒 Выбрать тариф", callback_data="select_paid_plan")
        trial_kb.adjust(1)
        
        await message.answer(
            "🔑 <b>Добро пожаловать в РазДваВПН!</b>\n\n"
            "Выберите подходящий тариф для подключения к нашему VPN сервису.",
            reply_markup=trial_kb.as_markup(),
            parse_mode="HTML"
        )
        
    except Exception as e:
        logger.error(f"Error in new_subscription: {e}")
        await message.answer("Произошла ошибка при получении информации о подписке.")


@router.callback_query(F.data == "select_paid_plan")
async def show_paid_plans(callback: CallbackQuery, state: FSMContext):
    """Show paid plan options"""
    try:
        await callback.message.edit_text(
            "🔑 <b>Выберите тарифный план:</b>",
            reply_markup=get_plans_keyboard(),
            parse_mode="HTML"
        )
    except Exception:
        # Fallback if edit fails (e.g., message with photo)
        await callback.message.answer(
            "🔑 <b>Выберите тарифный план:</b>",
            reply_markup=get_plans_keyboard(),
            parse_mode="HTML"
        )
    
    await state.set_state(SubscriptionStates.selecting_plan)
    await callback.answer()


@router.callback_query(StateFilter(SubscriptionStates.selecting_plan))
async def process_plan_selection(callback: CallbackQuery, state: FSMContext):
    """Process the selected plan and show confirmation"""
    plan_id = callback.data
    
    if plan_id not in SUBSCRIPTION_PLANS:
        try:
            await callback.message.edit_text(
                "❌ Неверный тарифный план. Пожалуйста, выберите один из доступных.",
                reply_markup=get_plans_keyboard(),
                parse_mode="HTML"
            )
        except Exception:
            await callback.message.answer(
                "❌ Неверный тарифный план. Пожалуйста, выберите один из доступных.",
                reply_markup=get_plans_keyboard(),
                parse_mode="HTML"
            )
        return
    
    plan = SUBSCRIPTION_PLANS[plan_id]
    await state.update_data(selected_plan=plan_id)
    
    # Show confirmation with plan details
    confirmation_text = create_plan_confirmation_text(plan)
    
    keyboard = InlineKeyboardBuilder()
    keyboard.button(text="💳 Оплатить", callback_data=f"pay_{plan_id}")
    keyboard.button(text="🔙 Назад", callback_data="back_to_plans")
    keyboard.adjust(1)
    
    await callback.message.edit_text(
        confirmation_text,
        reply_markup=keyboard.as_markup(),
        parse_mode="HTML"
    )
    await callback.answer()
    await state.set_state(SubscriptionStates.confirming_payment)


@router.callback_query(F.data == "back_to_plans")
async def back_to_plans(callback: CallbackQuery, state: FSMContext):
    """Return to plan selection"""
    await show_paid_plans(callback, state)


@router.callback_query(F.data.startswith("plan_"))
async def handle_direct_plan_selection(callback: CallbackQuery, state: FSMContext):
    """Handle direct plan selection without FSM state"""
    plan_id = callback.data.replace("plan_", "")
    
    if plan_id not in SUBSCRIPTION_PLANS:
        await callback.message.edit_text(
            "❌ Ошибка: неверный план. Попробуйте снова.",
            reply_markup=get_back_to_menu_inline()
        )
        return
    
    plan = SUBSCRIPTION_PLANS[plan_id]
    await state.update_data(selected_plan=plan_id)
    
    # Show confirmation with plan details
    confirmation_text = create_plan_confirmation_text(plan)
    
    keyboard = InlineKeyboardBuilder()
    keyboard.button(text="💳 Оплатить", callback_data=f"pay_{plan_id}")
    keyboard.button(text="🔙 Назад к тарифам", callback_data="start_subscription")
    keyboard.adjust(1)
    
    await callback.message.edit_text(
        confirmation_text,
        reply_markup=keyboard.as_markup(),
        parse_mode="HTML"
    )
    await callback.answer()
    await state.set_state(SubscriptionStates.confirming_payment)


@router.callback_query(StateFilter(SubscriptionStates.confirming_payment), F.data.startswith("pay_"))
async def process_payment(callback: CallbackQuery, state: FSMContext):
    """Process payment for the selected plan"""
    plan_id = callback.data.split("_")[1]
    
    if plan_id not in SUBSCRIPTION_PLANS:
        await callback.message.edit_text(
            "❌ Ошибка: неверный план. Попробуйте снова.",
            reply_markup=get_back_to_menu_inline()
        )
        return
    
    plan = SUBSCRIPTION_PLANS[plan_id]
    user_id = callback.from_user.id
    
    # Validate user exists
    user = await get_user_postgres(user_id)
    if not user or not user.vpn_email:
        await callback.message.edit_text(
            "❌ Ошибка: Пользователь не найден или отсутствует email.\n"
            "Пожалуйста, выполните команду /start заново.",
            reply_markup=get_back_to_menu_inline()
        )
        return
    
    try:
        # Check for existing client in panel
        client_checker = ClientChecker()
        existing_client = await client_checker.check_client_exists(user.vpn_email)
        
        vpn_account = None
        if existing_client:
            logger.info(f"Found existing client with email {user.vpn_email}")
            vpn_account = {
                "client_id": existing_client.get("id"),
                "id": existing_client.get("id"),
                "email": user.vpn_email
            }
        else:
            logger.info(f"Creating new client with email {user.vpn_email}")
            vpn_account = await create_vpn_account(
                user_id=user_id,
                email=user.vpn_email,
                protocol="VLESS",
                traffic_limit_gb=plan["traffic_gb"],
                days=plan["duration_days"]
            )
        
        if not vpn_account or not vpn_account.get("client_id"):
            logger.error(f"VPN account creation failed. Account: {vpn_account}")
            raise Exception("Не удалось создать VPN-клиента в панели")
        
        # Create subscription in database (inactive until payment)
        expires_at = datetime.now() + timedelta(days=plan["duration_days"])
        subscription_data = {
            "user_id": user_id,
            "plan_id": plan_id,
            "created_at": datetime.now(),
            "expires_at": expires_at,
            "traffic_limit": plan["traffic_gb"],
            "traffic_used": 0,
            "vpn_user_id": vpn_account["client_id"],
            "vpn_key": None,
            "status": "PENDING",
            "vpn_email": user.vpn_email
        }
        
        created_subscription = await create_subscription_postgres(subscription_data)
        
        if not created_subscription:
            logger.error("Failed to create subscription in PostgreSQL")
            raise Exception("Не удалось создать подписку в базе данных")
        
        # Update sync_id with real UUID if needed
        if vpn_account["client_id"] and vpn_account["client_id"].startswith("sync_id_"):
            from bot.services.sync_id_updater import update_single_subscription_sync_id
            await update_single_subscription_sync_id(created_subscription.id, user.vpn_email)
        
        # Create payment
        payment_info = await create_payment_for_plan(
            user_id=user_id,
            plan_id=plan_id,
            amount=plan['price'],
            description=f"Подписка VPN: {plan['name']} на {plan['duration_days']} дней",
            subscription_id=created_subscription.id
        )
        
        # Store payment information
        await state.update_data(
            payment_id=payment_info["payment_id"],
            payment_url=payment_info["payment_url"],
            subscription_id=created_subscription.id
        )
        
        # Create payment keyboard
        keyboard = InlineKeyboardBuilder()
        keyboard.button(text="💳 Перейти к оплате", url=payment_info["payment_url"])
        keyboard.button(text="🏠 В главное меню", callback_data="to_main_menu")
        keyboard.adjust(1)
        
        await callback.message.edit_text(
            f"💰 <b>Оплата подписки {plan['name']}</b>\n\n"
            f"Сумма к оплате: {plan['price']} руб.\n"
            f"Пользователь создан в системе.\n"
            f"VPN-клиент подготовлен в панели.\n\n"
            f"Нажмите на кнопку ниже, чтобы перейти к оплате.\n\n"
            f"💡 Подписка активируется автоматически после успешной оплаты.",
            reply_markup=keyboard.as_markup(),
            parse_mode="HTML"
        )
        
    except Exception as e:
        logger.error(f"Error creating payment: {e}")
        await callback.message.edit_text(
            "❌ Произошла ошибка при подготовке подписки. Пожалуйста, попробуйте позже.",
            reply_markup=get_back_to_menu_inline()
        )
    
    await callback.answer()


@router.callback_query(F.data.startswith("get_vpn_key_"))
async def get_vpn_key_handler(callback: CallbackQuery):
    """Handle VPN key retrieval"""
    try:
        subscription_id = int(callback.data.split("_")[3])
        subscription = await get_subscription_by_id_postgres(subscription_id)
        
        if not subscription:
            await callback.message.edit_text(
                "❌ Подписка не найдена. Пожалуйста, обратитесь в поддержку.",
                reply_markup=InlineKeyboardBuilder()
                    .button(text="🏠 В главное меню", callback_data="to_main_menu")
                    .as_markup()
            )
            await callback.answer()
            return
        
        # Show loading message
        await callback.message.edit_text(
            "⏳ Получаем ваш ключ VPN, пожалуйста, подождите...",
            reply_markup=None
        )
        
        # Check if subscription already has a valid VPN key
        if subscription.vpn_key and subscription.vpn_key.startswith("vless://"):
            logger.info(f"User {subscription.user_id} already has VPN key, returning existing key with updated subscription info")
            # Return existing key - no need to recreate it
            vpn_link = subscription.vpn_key
            vpn_type = "VLESS"
        else:
            # Use the stored VPN email and user ID from subscription
            email = subscription.vpn_email
            client_id = subscription.vpn_user_id
            
            logger.info(f"Getting VPN key for user {subscription.user_id}, email: {email}, client_id: {client_id}")
            
            # Get VPN key using direct panel API approach
            from bot.services.panel_session_manager import PanelSessionManager
            import json
            
            try:
                manager = PanelSessionManager()
                
                # Get inbound data to extract client information
                response = await manager.make_authenticated_request("GET", "inbounds/get/1")
                
                if response and response.get('success'):
                    obj = response.get('obj', {})
                    
                    # Parse settings
                    stream_settings = json.loads(obj.get('streamSettings', '{}')) if isinstance(obj.get('streamSettings'), str) else obj.get('streamSettings', {})
                    settings = json.loads(obj.get('settings', '{}')) if isinstance(obj.get('settings'), str) else obj.get('settings', {})
                    
                    reality_settings = stream_settings.get('realitySettings', {})
                    clients = settings.get('clients', [])
                    
                    # Find client by stored client_id (most reliable)
                    target_client = None
                    for client in clients:
                        if client.get('id') == client_id:
                            target_client = client
                            logger.info(f"Found target client by client_id: {client.get('email')}")
                            break
                    
                    # Fallback: search by email if client_id doesn't match
                    if not target_client and email:
                        for client in clients:
                            if client.get('email') == email:
                                target_client = client
                                logger.info(f"Found target client by email: {client.get('email')}")
                                break
                    
                    if target_client:
                        # Extract key and create VLESS link using working panel format
                        key = target_client['id']
                        
                        # Get correct port from inbound
                        port = obj.get("port", 443)
                        server_ip = "31.172.75.92"
                        
                        # СТРОГИЕ ПРАВИЛА: Используем фиксированные параметры Reality для всех VLESS ключей
                        # pbk всегда один и тот же для всех пользователей
                        public_key = "J7O8WbpqHE4mrSwOZZSb0zWc5Wx5ptCJzyfnVAi0pRw"
                        sni = "yahoo.com"
                        short_id = "cec02ca306"
                        
                        # Form VLESS link с фиксированными параметрами согласно строгим правилам
                        link = (
                            f"vless://{key}@{server_ip}:{port}"
                            f"?type=tcp&security=reality&pbk={public_key}"
                            f"&fp=chrome&sni={sni}&sid={short_id}"
                            f"&spx=%2F&flow=xtls-rprx-vision"
                            f"#RazDvaVPN%20Vless-{email.replace('@', '%40')}"
                        )
                        
                        # Update subscription with key
                        await update_subscription_postgres(subscription.id, {
                            'vpn_key': link,  # Store full link as vpn_key
                            'vpn_user_id': key  # Store client ID separately
                        })
                        
                        logger.info(f"Successfully extracted VPN key for client {email}: {key}")
                        vpn_link = link
                        vpn_type = "VLESS"
                        
                    else:
                        raise Exception("Клиент не найден в панели")
                else:
                    raise Exception("Не удалось получить данные из панели")
                    
            except Exception as panel_error:
                logger.error(f"Panel API error: {panel_error}")
                # Fallback to optimized panel API
                from bot.services.optimized_panel_api import get_client_link
                
                link = await get_client_link(email)
                
                if link:
                    # Update subscription with key
                    await update_subscription_postgres(subscription.id, {
                        'vpn_key': link  # Store full link
                    })
                    
                    vpn_link = link
                    vpn_type = "VLESS"
                    logger.info(f"Fallback method succeeded for {email}")
                else:
                    raise Exception("Не удалось получить VPN ключ")
        
        # Show the VPN key to user (either existing or newly retrieved)
        success_kb = InlineKeyboardBuilder()
        success_kb.button(text="🏠 В главное меню", callback_data="to_main_menu")
        success_kb.adjust(1)
        
        await callback.message.edit_text(
            f"✅ <b>Ваш ключ VPN получен!</b>\n\n"
            f"🔑 <b>Ключ {vpn_type}:</b> <code>{vpn_link}</code>\n\n"
            f"📋 <b>Скопируйте ключ и вставьте в приложение Hiddify или V2RayNG из буфера обмена</b>\n\n"
            f"⚠️ Сохраните ключ! Он также доступен в разделе «Личный Кабинет».",
            reply_markup=success_kb.as_markup(),
            parse_mode="HTML"
        )
        
    except Exception as e:
        logger.error(f"Error getting VPN key: {e}")
        
        # More specific error handling
        error_message = "❌ Произошла ошибка при получении ключа."
        if "SSL" in str(e) or "certificate" in str(e):
            error_message += "\n\n🔧 Проблема с SSL-сертификатом панели. Обратитесь к администратору."
        elif "timeout" in str(e).lower():
            error_message += "\n\n⏱️ Превышено время ожидания. Панель может быть временно недоступна."
        elif "connection" in str(e).lower():
            error_message += "\n\n🌐 Проблема с подключением к панели управления."
        else:
            error_message += "\n\n💬 Обратитесь к администратору если проблема повторяется."
        
        retry_kb = InlineKeyboardBuilder()
        retry_kb.button(text="🔄 Попробовать снова", callback_data=f"get_vpn_key_{subscription_id}")
        retry_kb.button(text="🏠 В главное меню", callback_data="to_main_menu")
        retry_kb.adjust(1)
        
        await callback.message.edit_text(
            error_message,
            reply_markup=retry_kb.as_markup()
        )
    
    await callback.answer()





@router.callback_query(F.data == "to_main_menu")
async def to_main_menu_handler(callback: CallbackQuery, state: FSMContext):
    """Return to main menu"""
    await state.clear()
    if callback.message:
        await callback.message.answer(
            "🏠 <b>Главное меню</b>\n\nВыберите действие:",
            reply_markup=get_main_menu_keyboard(),
            parse_mode="HTML"
        )
    await callback.answer()


# Trial period handler (currently active for activation)
@router.callback_query(F.data == "activate_test_period")
async def activate_test_period(callback: CallbackQuery, state: FSMContext):
    """Activate trial period subscription"""
    user_id = callback.from_user.id
    
    # Check for existing active subscription
    has_active, subscription = await check_active_subscription(user_id)
    if has_active and subscription:
        # Always send new message for consistency
        await callback.message.answer(
            format_active_subscription_message(subscription),
            reply_markup=get_main_menu_keyboard(),
            parse_mode="HTML"
        )
        await callback.answer()
        return
    
    # Check if user has already used a trial subscription
    has_used_trial = await has_user_used_trial_postgres(user_id)
    if has_used_trial:
        # User has already used trial, show notification about paid plans
        kb = InlineKeyboardBuilder()
        kb.button(text="💳 Выбрать платный тариф", callback_data="show_paid_plans")
        kb.button(text="⬅️ В главное меню", callback_data="back_to_menu")
        kb.adjust(1)
        
        await callback.message.answer(
            "❌ <b>Тестовый период уже использован</b>\n\n"
            "Вы уже воспользовались бесплатным тестовым периодом. "
            "Для продолжения использования VPN выберите один из платных тарифов.\n\n"
            "💡 <b>Преимущества платных тарифов:</b>\n"
            "• Больше трафика\n"
            "• Более длительный срок действия\n"
            "• Приоритетная поддержка\n"
            "• Стабильное соединение",
            reply_markup=kb.as_markup(),
            parse_mode="HTML"
        )
        await callback.answer()
        return
    
    try:
        # Get or create user
        user = await get_user_postgres(user_id)
        if not user:
            await callback.answer("❌ Пользователь не найден. Выполните /start", show_alert=True)
            return
        
        # Always send new message to avoid edit failures with media messages
        await callback.message.answer("⏳ Создаю ваш тестовый аккаунт, пожалуйста, подождите...")
        
        # Create VPN account for trial
        trial_plan = SUBSCRIPTION_PLANS["trial"]
        vpn_account = await create_vpn_account(
            user_id=user_id,
            email=user.vpn_email,
            protocol="VLESS",
            traffic_limit_gb=trial_plan["traffic_gb"],
            days=trial_plan["duration_days"]
        )
        
        if vpn_account and vpn_account.get("client_id"):
            # Create trial subscription
            expires_at = datetime.now() + timedelta(days=trial_plan["duration_days"])
            subscription_data = {
                "user_id": user_id,
                "plan_id": "trial",
                "created_at": datetime.now(),
                "expires_at": expires_at,
                "traffic_limit": trial_plan["traffic_gb"],
                "traffic_used": 0,
                "vpn_user_id": vpn_account["client_id"],
                "vpn_key": None,
                "status": "ACTIVE",
                "vpn_email": user.vpn_email
            }
            
            created_sub = await create_subscription_postgres(subscription_data)
            
            if not created_sub:
                logger.error("Failed to create subscription in PostgreSQL")
                raise Exception("Не удалось создать подписку в базе данных")
            
            # Update sync_id with real UUID if needed (for trial subscriptions too)
            if vpn_account["client_id"] and vpn_account["client_id"].startswith("sync_id_"):
                from bot.services.sync_id_updater import update_single_subscription_sync_id
                await update_single_subscription_sync_id(created_sub.id, user.vpn_email)
            
            # Create success keyboard
            success_kb = InlineKeyboardBuilder()
            success_kb.button(text="🔑 Получить ключ VPN", callback_data=f"get_vpn_key_{created_sub.id}")
            success_kb.adjust(1)
            
            # Always send new message to avoid edit failures with media messages
            await callback.message.answer(
                f"✅ <b>Ваш тестовый аккаунт успешно создан!</b>\n\n"
                f"<b>Тестовый период:</b> {trial_plan['duration_days']} дня\n"
                f"<b>Трафик:</b> {trial_plan['traffic_gb']} ГБ\n"
                f"<b>Срок действия:</b> до {expires_at.strftime('%d.%m.%Y')}\n\n"
                f"Нажмите кнопку ниже, чтобы получить ключ для настройки VPN.",
                reply_markup=success_kb.as_markup(),
                parse_mode="HTML"
            )
        else:
            # Always send new message to avoid edit failures with media messages
            await callback.message.answer(
                "❌ Произошла ошибка при создании тестового аккаунта. "
                "Пожалуйста, попробуйте позже или обратитесь к администратору.",
                reply_markup=get_main_menu_keyboard()
            )
    
    except Exception as e:
        logger.error(f"Error creating trial subscription: {e}")
        # Always send new message to avoid edit failures with media messages
        await callback.message.answer(
            "❌ Произошла ошибка при создании тестового аккаунта. "
            "Пожалуйста, попробуйте позже или обратитесь к администратору.",
            reply_markup=get_main_menu_keyboard()
        )
    
    await callback.answer()


def register_handlers(dp):
    """Register all subscription handlers"""
    dp.include_router(router)