import logging
from aiogram import Router, F, types
from aiogram.filters import Command
from aiogram.fsm.context import FSMContext
from aiogram.types import Message, CallbackQuery
from aiogram.utils.keyboard import ReplyKeyboardBuilder, InlineKeyboardBuilder

from bot.database.operations import get_user, get_user_subscription
from bot.utils.keyboard import get_main_menu_keyboard, get_personal_account_keyboard
from bot.utils.formatting import format_traffic_info
from bot.services.panel_service import get_traffic_stats
from config import SUBSCRIPTION_PLANS

router = Router()
logger = logging.getLogger(__name__)


@router.message(F.text == "Личный Кабинет")
@router.message(Command("account"))
async def personal_account(message: Message):
    """Handle personal account menu"""
    await message.answer(
        "👤 <b>Личный кабинет</b>\n\n"
        "Выберите раздел:",
        reply_markup=get_personal_account_keyboard()
    )


@router.message(F.text == "Текущая Подписка")
async def current_subscription(message: Message):
    """Show current subscription details"""
    from bot.utils.formatting import format_bytes_to_gb
    from bot.database.postgres_operations import get_user_subscription_postgres
    
    user_id = message.from_user.id
    
    # Получаем данные из PostgreSQL для актуальной информации о трафике
    subscription = await get_user_subscription_postgres(user_id)
    
    # Если не найдено в PostgreSQL, пробуем SQLite (для совместимости)
    if not subscription:
        subscription = await get_user_subscription(user_id)
    
    if not subscription:
        await message.answer(
            "❌ У вас нет активной подписки.\n\n"
            "Для оформления подписки нажмите «Выбрать Тариф» в главном меню.",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # Используем данные из базы данных (более надежно)
    logger.debug(f"Отображение подписки {subscription.id} с трафиком из базы: {subscription.traffic_used / (1024**3):.3f} ГБ")
    
    # Get plan details
    plan = SUBSCRIPTION_PLANS.get(subscription.plan_id, {"name": "Неизвестный тариф"})
    
    # traffic_limit уже в ГБ, не требует дополнительной конвертации
    traffic_limit_gb = subscription.traffic_limit
    total_gb, used_gb, remaining_gb = format_traffic_info(
        traffic_limit_gb, 
        subscription.traffic_used
    )
    
    # Format subscription info
    subscription_info = (
        f"📋 <b>Информация о подписке</b>\n\n"
        f"🆔 ID подписки: {subscription.id}\n"
        f"📦 Тариф: {plan.get('name', 'Неизвестный тариф')}\n"
        f"⏱️ Действует до: {subscription.expires_at.strftime('%d.%m.%Y')}\n"
        f"⌛ Осталось дней: {subscription.days_remaining}\n\n"
        f"📊 <b>Трафик:</b>\n"
        f"• Всего: {total_gb} ГБ\n"
        f"• Использовано: {used_gb} ГБ\n"
        f"• Осталось: {remaining_gb} ГБ\n\n"
        f"🔑 <b>Ключ VPN:</b> <code>{subscription.vpn_key}</code>\n\n"
    )
    
    # Build keyboard
    keyboard = InlineKeyboardBuilder()
    keyboard.button(
        text="🔑 Получить/Обновить VPN Ключ",
        callback_data=f"get_vpn_key_{subscription.id}"
    )
    keyboard.adjust(1)
    
    await message.answer(
        subscription_info,
        reply_markup=keyboard.as_markup()
    )


@router.message(F.text == "Мои Платежи")
async def payment_history(message: Message):
    """Show payment history for today using PostgreSQL"""
    user_id = message.from_user.id
    
    # Используем PostgreSQL вместо SQLite
    payments = await get_payment_history_for_period(user_id, 1)
    
    if not payments:
        keyboard = InlineKeyboardBuilder()
        keyboard.button(text="📅 За месяц", callback_data="payments_month")
        keyboard.button(text="📊 За квартал", callback_data="payments_quarter")
        keyboard.button(text="📈 За год", callback_data="payments_year")
        keyboard.button(text="🔙 Назад", callback_data="back_to_personal")
        keyboard.adjust(2, 1, 1)
        
        await message.answer(
            "📊 <b>Мои платежи за сегодня</b>\n\n"
            "❌ У вас нет платежей за сегодня.\n\n"
            "Попробуйте выбрать другой период для просмотра:",
            reply_markup=keyboard.as_markup()
        )
        return
    
    # Формируем сообщение с платежами, используя функцию format_payment_history_message
    payment_text = format_payment_history_message(payments, "сегодня", 1)
    
    # Добавляем inline клавиатуру для выбора периода
    keyboard = InlineKeyboardBuilder()
    keyboard.button(text="📅 Сегодня", callback_data="payments_today")
    keyboard.button(text="📅 За месяц", callback_data="payments_month")
    keyboard.button(text="📊 За квартал", callback_data="payments_quarter")
    keyboard.button(text="📈 За год", callback_data="payments_year")
    keyboard.button(text="🔙 Назад", callback_data="back_to_personal")
    keyboard.adjust(2, 2, 1)
    
    await message.answer(
        payment_text,
        reply_markup=keyboard.as_markup()
    )


@router.message(F.text == "Помощь")
async def help_section(message: Message):
    """Show help information"""
    help_text = (
        "❓ <b>Помощь</b>\n\n"
        "<b>Как использовать VPN:</b>\n"
        "1. Оформите подписку через бота\n"
        "2. Получите ключ VPN\n"
        "3. Установите приложение для своего устройства\n"
        "4. Введите полученный ключ в приложение\n\n"
        
        "<b>Советы по использованию:</b>\n"
        "• Не делитесь ключом с посторонними лицами\n"
        "• Для экономии трафика отключайте VPN, когда он не нужен\n"
        "• При проблемах с подключением перезапустите приложение\n\n"
        
        "<b>Приложения для подключения:</b>\n"
        "• <a href='https://apps.apple.com/us/app/v2box/id6446212981'>iOS</a>\n"
        "• <a href='https://play.google.com/store/apps/details?id=com.v2ray.v2box'>Android</a>\n"
        "• <a href='https://github.com/2dust/v2rayN/releases'>Windows</a>\n"
        "• <a href='https://github.com/yanue/V2rayU/releases'>macOS</a>\n\n"
        
        "По всем вопросам обращайтесь к администратору: @admin_username"
    )
    
    await message.answer(
        help_text,
        reply_markup=get_personal_account_keyboard()
    )


@router.message(F.text == "Назад в меню")
async def back_to_main_menu(message: Message):
    """Return to main menu"""
    await message.answer(
        "Выберите действие из главного меню:",
        reply_markup=get_main_menu_keyboard()
    )


@router.callback_query(F.data == "setup_instructions")
async def show_setup_instructions(callback: CallbackQuery):
    """Show VPN setup instructions"""
    instructions = (
        "📱 <b>Инструкция по настройке VPN</b>\n\n"
        
        "<b>Для iOS:</b>\n"
        "1. Установите приложение Hiddify Next (рекомендуется): <a href='https://apps.apple.com/app/hiddify-next/id6596777532'>App Store</a>\n"
        "   Альтернатива: <a href='https://apps.apple.com/us/app/v2box/id6446212981'>V2Box</a>\n"
        "2. Откройте приложение\n"
        "3. Нажмите на + для добавления нового подключения\n"
        "4. Выберите 'Import from clipboard'\n"
        "5. Скопируйте ваш ключ VPN и вставьте\n"
        "6. Нажмите на добавленное подключение для активации\n\n"
        
        "<b>Для Android:</b>\n"
        "1. Установите приложение Hiddify (рекомендуется): <a href='https://play.google.com/store/apps/details?id=app.hiddify.com'>Google Play</a>\n"
        "   Альтернатива: <a href='https://play.google.com/store/apps/details?id=com.v2ray.v2box'>V2Box</a>\n"
        "2. Откройте приложение\n"
        "3. Нажмите на значок + внизу экрана\n"
        "4. Выберите 'Import from clipboard'\n"
        "5. Скопируйте ваш ключ VPN и вставьте\n"
        "6. Нажмите на переключатель для подключения\n\n"
        
        "<b>Для Windows:</b>\n"
        "1. Скачайте <a href='https://github.com/2dust/v2rayN/releases'>v2rayN</a>\n"
        "2. Распакуйте и запустите программу\n"
        "3. Кликните правой кнопкой мыши в области серверов\n"
        "4. Выберите 'Import from clipboard'\n"
        "5. Скопируйте ваш ключ VPN и вставьте\n\n"
        
        "<b>Для macOS:</b>\n"
        "1. Скачайте <a href='https://github.com/yanue/V2rayU/releases'>V2rayU</a>\n"
        "2. Установите и запустите программу\n"
        "3. Нажмите на меню V2rayU в строке меню\n"
        "4. Выберите 'Import from clipboard'\n"
        "5. Скопируйте ваш ключ VPN и вставьте\n\n"
        
        "Если у вас возникли проблемы с настройкой, обратитесь в нашу службу поддержки."
    )
    
    keyboard = InlineKeyboardBuilder()
    keyboard.button(text="🔙 Назад", callback_data="back_to_subscription")
    
    await callback.message.edit_text(
        instructions,
        reply_markup=keyboard.as_markup()
    )
    await callback.answer()


@router.callback_query(F.data == "extend_from_info")
async def extend_from_info(callback: CallbackQuery):
    """Shortcut to extend subscription from info page"""
    # Check if user has an active subscription
    from bot.database.postgres_operations import get_user_subscription_postgres
    
    subscription = await get_user_subscription_postgres(callback.from_user.id)
    if not subscription:
        subscription = await get_user_subscription(callback.from_user.id)
    
    if not subscription:
        await callback.message.edit_text(
            "❌ У вас нет активной подписки, которую можно продлить.\n\n"
            "Пожалуйста, оформите новую подписку с помощью кнопки «Выбрать Тариф».",
            reply_markup=get_main_menu_keyboard()
        )
        await callback.answer()
        return
    
    # Check if user has trial subscription
    if subscription.plan_id == "trial":
        kb = InlineKeyboardBuilder()
        kb.button(text="💳 Выбрать платный тариф", callback_data="show_paid_plans")
        kb.button(text="⬅️ В главное меню", callback_data="back_to_menu")
        kb.adjust(1)
        
        await callback.message.edit_text(
            "ℹ️ <b>У вас оформлена тестовая подписка</b>\n\n"
            "Тестовый период можно получить только один раз. "
            "Для продолжения пользования сервисом выберите платный тариф.\n\n"
            "💡 Платные тарифы предоставляют больше трафика и более длительный срок действия.",
            reply_markup=kb.as_markup(),
            parse_mode="HTML"
        )
        await callback.answer()
        return
    
    # Show current plan and options
    current_plan = SUBSCRIPTION_PLANS.get(subscription.plan_id, {"name": "Неизвестный тариф"})
    
    # Create keyboard with plan options
    keyboard = InlineKeyboardBuilder()
    
    # Option to extend current plan
    keyboard.button(
        text=f"🔄 Продлить текущий тариф ({current_plan['name']})",
        callback_data=f"extend_{subscription.plan_id}"
    )
    
    # Option to change plan
    keyboard.button(
        text="📈 Выбрать другой тариф",
        callback_data="change_plan"
    )
    
    keyboard.adjust(1)
    
    await callback.message.edit_text(
        f"📋 <b>Продление подписки</b>\n\n"
        f"Ваша текущая подписка: <b>{current_plan['name']}</b>\n"
        f"Срок действия: до {subscription.expires_at.strftime('%d.%m.%Y') if subscription.expires_at else 'не указан'}\n\n"
        f"Выберите способ продления:",
        reply_markup=keyboard.as_markup()
    )
    
    await callback.answer()


@router.callback_query(F.data == "back_to_subscription")
async def back_to_subscription(callback: CallbackQuery):
    """Return to subscription information"""
    from bot.utils.formatting import format_bytes_to_gb
    
    from bot.database.postgres_operations import get_user_subscription_postgres
    
    user_id = callback.from_user.id
    
    # Получаем актуальные данные из PostgreSQL
    subscription = await get_user_subscription_postgres(user_id)
    if not subscription:
        subscription = await get_user_subscription(user_id)
    
    if not subscription:
        await callback.message.edit_text(
            "❌ У вас нет активной подписки.\n\n"
            "Для оформления подписки нажмите «Выбрать Тариф» в главном меню.",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # Get current traffic usage from the panel
    try:
        traffic_stats = await get_traffic_stats(subscription.vpn_user_id)
        if traffic_stats and "traffic_used" in traffic_stats:
            subscription.traffic_used = traffic_stats["traffic_used"]
    except Exception as e:
        logger.error(f"Error getting traffic stats: {e}")
        # Continue with existing data if API call fails
    
    # Get plan details
    plan = SUBSCRIPTION_PLANS.get(subscription.plan_id, {"name": "Неизвестный тариф"})
    
    # traffic_limit уже в ГБ, не требует дополнительной конвертации
    traffic_limit_gb = subscription.traffic_limit
    total_gb, used_gb, remaining_gb = format_traffic_info(
        traffic_limit_gb, 
        subscription.traffic_used
    )
    
    # Format subscription info
    subscription_info = (
        f"📋 <b>Информация о подписке</b>\n\n"
        f"🆔 ID подписки: {subscription.id}\n"
        f"📦 Тариф: {plan.get('name', 'Неизвестный тариф')}\n"
        f"⏱️ Действует до: {subscription.expires_at.strftime('%d.%m.%Y')}\n"
        f"⌛ Осталось дней: {subscription.days_remaining}\n\n"
        f"📊 <b>Трафик:</b>\n"
        f"• Всего: {total_gb} ГБ\n"
        f"• Использовано: {used_gb} ГБ\n"
        f"• Осталось: {remaining_gb} ГБ\n\n"
        f"🔑 <b>Ключ VPN:</b> <code>{subscription.vpn_key}</code>\n\n"
    )
    
    # Build keyboard
    keyboard = InlineKeyboardBuilder()
    keyboard.button(
        text="🔑 Получить/Обновить VPN Ключ",
        callback_data=f"get_vpn_key_{subscription.id}"
    )
    keyboard.button(
        text="📋 Инструкция по настройке",
        callback_data="setup_instructions"
    )
    keyboard.button(
        text="💰 Продлить подписку",
        callback_data="extend_from_info"
    )
    keyboard.adjust(1)
    
    await callback.message.edit_text(
        subscription_info,
        reply_markup=keyboard.as_markup()
    )
    await callback.answer()


@router.callback_query(F.data.startswith("extend_"))
async def handle_extend_subscription(callback: CallbackQuery, state: FSMContext):
    """Handle subscription extension for specific plan"""
    from bot.services.payment import create_payment_for_plan
    
    plan_id = callback.data.split("_")[1]
    plan = SUBSCRIPTION_PLANS.get(plan_id)
    
    if not plan:
        await callback.message.edit_text(
            "❌ Произошла ошибка. Тариф не найден.",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # Get current subscription
    subscription = await get_user_subscription(callback.from_user.id)
    if not subscription:
        await callback.message.edit_text(
            "❌ У вас нет активной подписки, которую можно продлить.",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    try:
        # Create payment for extension
        payment_info = await create_payment_for_plan(
            user_id=callback.from_user.id,
            plan_id=plan_id,
            amount=plan['price'],
            description=f"Продление подписки VPN: {plan['name']} на {plan['duration_days']} дней",
            subscription_id=subscription.id
        )
        
        # Build keyboard with payment link
        keyboard = InlineKeyboardBuilder()
        keyboard.button(
            text="💳 Перейти к оплате",
            url=payment_info["payment_url"]
        )
        keyboard.button(
            text="🔙 Назад",
            callback_data="extend_from_info"
        )
        keyboard.button(
            text="🏠 В главное меню",
            callback_data="to_main_menu"
        )
        keyboard.adjust(1)
        
        await callback.message.edit_text(
            f"💰 <b>Продление подписки {plan['name']}</b>\n\n"
            f"Сумма к оплате: {plan['price']} руб.\n"
            f"Срок продления: {plan['duration_days']} дней\n"
            f"Трафик: {plan['traffic_gb']} ГБ\n\n"
            f"Нажмите на кнопку ниже, чтобы перейти к оплате.\n\n"
            f"💡 Подписка будет продлена автоматически после успешной оплаты.",
            reply_markup=keyboard.as_markup()
        )
        
    except Exception as e:
        logger.error(f"Error creating payment for extension: {e}")
        await callback.message.edit_text(
            "❌ Произошла ошибка при создании платежа. Пожалуйста, попробуйте позже.",
            reply_markup=get_main_menu_keyboard()
        )
    
    await callback.answer()


@router.callback_query(F.data == "change_plan")
async def handle_change_plan(callback: CallbackQuery, state: FSMContext):
    """Handle plan change request"""
    from bot.handlers.subscription import show_paid_plans
    
    # Redirect to plan selection
    await show_paid_plans(callback, state)


async def get_payment_history_for_period(user_id: int, days: int):
    """Get payment history for specified period using PostgreSQL"""
    from datetime import datetime, timedelta
    from bot.database.postgres_operations import get_postgres_connection
    from sqlalchemy import text
    
    # Получаем дату начала периода
    period_end = datetime.now()
    period_start = period_end - timedelta(days=days)
    
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT 
                    p.id,
                    p.amount,
                    p.currency,
                    p.payment_id,
                    p.created_at,
                    p.completed_at,
                    p.status,
                    p.plan_id,
                    CASE p.plan_id 
                        WHEN 'trial' THEN 'Пробный период'
                        WHEN 'basic' THEN 'Базовый'
                        WHEN 'weekly' THEN 'Недельный' 
                        WHEN 'premium' THEN 'Премиум'
                        WHEN 'pro' THEN 'Профессиональный'
                        ELSE p.plan_id
                    END as plan_name
                FROM payments p
                WHERE p.user_id = :user_id 
                AND p.created_at >= :period_start 
                AND p.created_at <= :period_end
                ORDER BY p.created_at DESC
            """), {"user_id": user_id, "period_start": period_start, "period_end": period_end})
            
            payments = result.fetchall()
            logger.info(f"Найдено платежей для пользователя {user_id} за {days} дней: {len(payments)}")
            return payments
            
    except Exception as e:
        logger.error(f"Ошибка при получении истории платежей за {days} дней из PostgreSQL: {e}")
        return []


def format_payment_history_message(payments, period_name, days):
    """Format payment history message"""
    from datetime import datetime
    
    if not payments:
        return f"📊 <b>Мои платежи за {period_name}</b>\n\n❌ Платежей за указанный период не найдено."
    
    # Формируем сообщение с платежами
    payment_text = f"📊 <b>Мои платежи за {period_name}</b>\n\n"
    payment_text += f"📅 Период: последние {days} дней\n"
    payment_text += f"📈 Найдено платежей: {len(payments)}\n\n"
    
    successful_count = 0
    failed_count = 0
    total_successful_amount = 0
    
    for payment in payments:
        payment_id_db, amount, currency, payment_id_yookassa, created_at, completed_at, status, plan_id, plan_name = payment
        
        # Парсим дату создания - в PostgreSQL это уже datetime объект
        if isinstance(created_at, str):
            created_datetime = datetime.fromisoformat(created_at.replace('T', ' ').replace('Z', ''))
        else:
            created_datetime = created_at
        
        created_date = created_datetime.strftime('%d.%m')
        created_time = created_datetime.strftime('%H:%M')
        
        # Определяем статус и эмодзи
        if status == "SUCCEEDED":
            status_emoji = "✅"
            status_text = "Успешно"
            successful_count += 1
            total_successful_amount += amount
        elif status == "CANCELED":
            status_emoji = "❌"
            status_text = "Отменен"
            failed_count += 1
        elif status == "PENDING":
            status_emoji = "⏳"
            status_text = "В обработке"
        else:
            status_emoji = "❓"
            status_text = status
        
        # Добавляем информацию о платеже
        payment_text += f"{status_emoji} <b>Платеж #{payment_id_db}</b>\n"
        payment_text += f"💰 {amount:.0f} {currency} • 📦 {plan_name or plan_id}\n"
        payment_text += f"📅 {created_date} {created_time} • {status_text}\n\n"
    
    # Добавляем статистику
    payment_text += f"📈 <b>Статистика за {period_name}:</b>\n"
    payment_text += f"✅ Успешных: {successful_count} • ❌ Неуспешных: {failed_count}\n"
    
    if successful_count > 0:
        payment_text += f"💵 Сумма успешных: {total_successful_amount:.0f} RUB"
    
    return payment_text


@router.callback_query(F.data == "payments_today")
async def show_payments_today(callback: CallbackQuery):
    """Show payments for today"""
    user_id = callback.from_user.id
    payments = await get_payment_history_for_period(user_id, 1)
    
    payment_text = format_payment_history_message(payments, "сегодня", 1)
    
    # Добавляем inline клавиатуру для выбора периода (только сегодня и месяц)
    keyboard = InlineKeyboardBuilder()
    keyboard.button(text="📅 Сегодня", callback_data="payments_today")
    keyboard.button(text="📅 За месяц", callback_data="payments_month")
    keyboard.button(text="🔙 Назад", callback_data="back_to_personal_menu")
    keyboard.adjust(2, 1)
    
    await callback.message.edit_text(
        payment_text,
        reply_markup=keyboard.as_markup()
    )
    await callback.answer()


@router.callback_query(F.data == "payments_month")
async def show_payments_month(callback: CallbackQuery):
    """Show payments for last month"""
    user_id = callback.from_user.id
    payments = await get_payment_history_for_period(user_id, 30)
    
    payment_text = format_payment_history_message(payments, "месяц", 30)
    
    # Добавляем inline клавиатуру для выбора периода (только сегодня и месяц)
    keyboard = InlineKeyboardBuilder()
    keyboard.button(text="📅 Сегодня", callback_data="payments_today")
    keyboard.button(text="📅 За месяц", callback_data="payments_month")
    keyboard.button(text="🔙 Назад", callback_data="back_to_personal_menu")
    keyboard.adjust(2, 1)
    
    await callback.message.edit_text(
        payment_text,
        reply_markup=keyboard.as_markup()
    )
    await callback.answer()





@router.callback_query(F.data == "back_to_personal_menu")  
async def back_to_personal_menu(callback: CallbackQuery):
    """Return to personal account menu"""
    try:
        # Удаляем inline клавиатуру из сообщения
        await callback.message.delete()
        
        # Отправляем новое сообщение с обычной клавиатурой
        await callback.message.answer(
            "👤 <b>Личный кабинет</b>\n\n"
            "Выберите раздел:",
            reply_markup=get_personal_account_keyboard()
        )
        await callback.answer()
    except Exception as e:
        logger.error(f"Ошибка в обработчике back_to_personal_menu: {e}")
        # Резервный вариант - просто отправляем сообщение
        await callback.message.answer(
            "👤 <b>Личный кабинет</b>\n\n"
            "Выберите раздел:",
            reply_markup=get_personal_account_keyboard()
        )
        await callback.answer()


def register_handlers(dp):
    """Register all personal account handlers"""
    dp.include_router(router)
