from aiogram import Router, F, types
from aiogram.filters import Command
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.utils.keyboard import ReplyKeyboardBuilder
from aiogram.types import FSInputFile
import asyncio
import logging
import os

from bot.database.postgres_operations import get_user_postgres as get_user, create_or_update_user_postgres as create_or_update_user, get_user_subscription_postgres as get_user_subscription
from bot.database.models import User, UserStatus
from bot.utils.keyboard import get_main_menu_keyboard
from bot.utils.email_generator import generate_vpn_email
from bot.database.sqlite_tariffs import get_subscription_plans

router = Router()
logger = logging.getLogger(__name__)

# Константы для путей к ресурсам
PLASHKA_IMAGE_PATH = "bot/assets/plashka.jpg"
HIDDIFY_LOGO_PATH = "bot/assets/hiddifylogo1280.png"


async def safe_send_photo(message_or_callback, photo_path: str, caption: str, reply_markup=None, max_retries: int = 3):
    """Безопасная отправка изображения с повторными попытками при ошибках соединения"""
    
    # Определяем объект для отправки сообщения
    if hasattr(message_or_callback, 'message'):
        sender = message_or_callback.message
    else:
        sender = message_or_callback
    
    for attempt in range(max_retries):
        try:
            if os.path.exists(photo_path):
                await sender.answer_photo(
                    FSInputFile(photo_path),
                    caption=caption,
                    reply_markup=reply_markup,
                    parse_mode="HTML"
                )
                return True
            else:
                # Если изображение не найдено, отправляем только текст
                await sender.answer(caption, reply_markup=reply_markup, parse_mode="HTML")
                return True
                
        except Exception as e:
            logger.error(f"Попытка {attempt + 1}/{max_retries} - Ошибка при отправке изображения {photo_path}: {e}")
            
            if attempt == max_retries - 1:  # Последняя попытка
                try:
                    # Fallback - отправляем только текст
                    await sender.answer(caption, reply_markup=reply_markup, parse_mode="HTML")
                    logger.info(f"Fallback: отправлен текст вместо изображения {photo_path}")
                    return True
                except Exception as fallback_error:
                    logger.error(f"Критическая ошибка при отправке текста: {fallback_error}")
                    return False
            else:
                # Небольшая пауза перед повторной попыткой
                await asyncio.sleep(1)
    
    return False


def generate_tariff_info_text():
    """Генерирует текст с информацией о тарифах на основе SQLite базы данных"""
    try:
        SUBSCRIPTION_PLANS = get_subscription_plans()
    except Exception:
        # Fallback к статическим тарифам если SQLite недоступен
        from config import SUBSCRIPTION_PLANS as STATIC_PLANS
        SUBSCRIPTION_PLANS = STATIC_PLANS
    
    text = "<b>Доступные тарифные планы:</b>\n\n"

    # Добавляем тестовый тариф отдельно
    if "trial" in SUBSCRIPTION_PLANS:
        trial = SUBSCRIPTION_PLANS["trial"]
        text += (
            f"<b>Тестовый «Старт»</b> - ⏱️ на {trial['duration_days']} дня, "
            f"💰- {trial['price']} руб.\n"
            f"📦 {trial['traffic_gb']} ГБ трафика\n"
            f"<i>*Разовый бесплатный тестовый период</i>\n"
            f"==================================\n"
            f"<b>Тарифы на платные подписки:</b>\n\n")

    # Добавляем платные тарифы
    for plan_id, plan in SUBSCRIPTION_PLANS.items():
        if plan_id == "trial":
            continue

        # Приблизительное время видео (1 ГБ ≈ 1.1 часа видео)
        video_hours = int(plan["traffic_gb"] * 1.1)

        # Определяем количество звезд по плану
        stars = ""
        if plan_id == "lux":
            stars = "⭐️⭐️⭐️⭐️"
        elif plan_id == "premium":
            stars = "⭐️⭐️⭐️"
        elif plan_id == "standard":
            stars = "⭐️⭐️"
        elif plan_id in ["basic", "weekly"]:
            stars = "⭐️"

        # Центрируем звездочки
        if stars:
            text += f"                        {stars}\n"

        # Основная информация о тарифе
        text += (
            f"<b>{plan['name']}</b> - ⏱️ {plan['duration_days']} дней, "
            f"💰 {plan['price']} руб.\n"
        )
        
        # Добавляем информацию о скидке только если она больше 0
        if "discount_percent" in plan and "price_per_30_days" in plan:
            discount = plan['discount_percent']
            if discount > 0:  # Показываем скидку только если она есть
                if plan_id == "lux":
                    text += f"  🔥 <b>Скидка !! {discount}%</b> (по {plan['price_per_30_days']} руб за 30 дней)\n"
                else:
                    text += f"  💸 Скидка {discount}% (по {plan['price_per_30_days']} руб за 30 дней)\n"
            
        text += (
            f"           {plan['traffic_gb']} ГБ трафика (~{video_hours} ч. видео)\n\n"
        )

    text += (
        "<i>*Значения часов трафика видео с Youtube даны примерные</i>\n\n"
        "👇 Для оплаты Тарифа, Продления, Изменения используйте Кнопки в ГЛАВНОМ МЕНЮ."
    )

    return text


@router.message(Command("start"))
async def cmd_start(message: Message):
    """
    Handle /start command - entry point for users
    """
    if not message.from_user:
        return
    
    # Create or update user in the database
    user = await get_user(message.from_user.id)

    if not user:
        # Генерируем email для нового пользователя
        vpn_email = generate_vpn_email(message.from_user.id,
                                       message.from_user.username or "user")

        user = User(
            id=message.from_user.id,
            username=message.from_user.username,
            first_name=message.from_user.first_name,
            last_name=message.from_user.last_name,
            status="NEW",
            vpn_email=vpn_email,
        )
        await create_or_update_user(user)
    elif not user.vpn_email:
        # Если у существующего пользователя нет email, создаем его
        user.vpn_email = generate_vpn_email(message.from_user.id,
                                            message.from_user.username or "user")
        await create_or_update_user(user)

    # Check if user already has a subscription
    subscription = await get_user_subscription(message.from_user.id)

    # Отправляем плашку с логотипом в самом начале
    from aiogram.types import FSInputFile

    try:
        if os.path.exists(PLASHKA_IMAGE_PATH):
            await message.answer_photo(FSInputFile(PLASHKA_IMAGE_PATH))
    except Exception as e:
        # Логируем ошибку, но продолжаем работу
        print(f"Ошибка при отправке изображения: {e}")

    # Welcome message с отступом
    welcome_text = (
        f"    👋 <b>Добро пожаловать в РазДваВПН!</b>\n\n"
        f"    С помощью этого бота вы сможете просто и легко\n\n"
        f"    Получить Ключ для ВПН приложения Hiddify или V2rayNG\n\n"
        f"    Получить ссылки для установки Приложений на телефон\n"
        f"    🚀 Для Smart TV | iPhone | Windows | Macos | Android и других устройств.\n"
        f"    ✓ Высокая скорость на всех устройствах\n"
        f"    ✓ Безопасное соединение и анонимность\n"
        f"    ✓ Доступ к любым сайтам и сервисам\n"
        f"    -Не блокирует сайты RU и банковские приложения !\n\n"
        f"    Легко активировать пробный тестовый период:\n"
        f"    • 🔑 Приобрести новую подписку\n"
        f"    • ⌛ Продлить существующую подписку\n"
        f"    • ℹ️ Узнать о доступных тарифах\n"
        f"    • 👤 Получить информацию о Доп.Сервисах\n\n"
        f"    Всего три шага\n"
        f"    1 Установите Приложение, если его нет\n"
        f"    2. Оформить Тестовый период и Получить Ключ\n"
        f"    3 . Скопировать целиком Ключ добавить новый Профиль и вставить Ключ. Нажать кнопку Пуск\n\n")

    # Если у пользователя еще нет подписки, новый флоу для установки приложения
    if not subscription:
        welcome_text += (
            f"    🎁 <b>Специальное предложение!</b>\n"
            f"    Попробуйте наш сервис бесплатно в течение 3 дней с лимитом 10 ГБ трафика. "
        )

        # Отправляем приветственное сообщение
        await message.answer(welcome_text, parse_mode="HTML")

        # Рекомендация установить приложение Hiddify с отступом
        recommendation_text = (
            "    📱 <b>Для использования VPN установите бесплатное приложение Hiddify или V2RayNG они доступны в Google Play или App Store</b>\n\n"
            "    Для комфортного использования VPN рекомендуем установить приложение Hiddify.\n\n"
            "    до активации тестового периода или оплаты подписки.\n\n"
            "    После установки приложения получите ключ, скопируйте его и добавьте в профиль. прим. КЛЮЧ! подходит для обоих приложений "
            'с помощью кнопки <b>"+ Добавить новый профиль"</b>.')

        # Создаем 3 основные кнопки
        main_kb = InlineKeyboardMarkup(inline_keyboard=[
            [
                InlineKeyboardButton(
                    text="📋 Получить ссылки для установки приложения VPN",
                    callback_data="get_install_info",
                )
            ],
            [
                InlineKeyboardButton(
                    text="🚀 Попробовать Тестовый Период",
                    callback_data="activate_test_period",
                )
            ],
            [
                InlineKeyboardButton(
                    text="💳 Выбрать Тариф для оплаты",
                    callback_data="select_paid_plan"
                )
            ]
        ])

        # Сначала отправляем основную клавиатуру с кнопками меню
        await message.answer("Используйте кнопки меню ниже:", 
                             reply_markup=get_main_menu_keyboard())
        
        # Затем отправляем рекомендации с inline кнопками
        await message.answer(recommendation_text,
                             reply_markup=main_kb,
                             parse_mode="HTML")
    else:
        welcome_text += "    Выберите действие в меню ниже:"
        await message.answer(welcome_text,
                             reply_markup=get_main_menu_keyboard(),
                             parse_mode="HTML")


@router.callback_query(F.data == "get_install_info")
async def handle_get_install_info(callback: CallbackQuery):
    """Show Hiddify app installation information"""
    # Отправляем изображение приложения Hiddify
    from aiogram.types import FSInputFile

    caption = (
        "🚀 <b>Приложение Hiddify для РазДваВПН</b>\n\n"
        "• Простой и лаконичный интерфейс\n"
        "• Кроссплатформенность: Android, iOS, Windows, Mac, Linux\n"
        
        "<b>Скачать приложение:</b>\n"
        "📱 Android: <a href='https://play.google.com/store/apps/details?id=app.hiddify.com'>Google Play</a> или <a href='https://github.com/hiddify/hiddify-next/releases/latest'>GitHub</a>\n"
        "🍎 iOS: <a href='https://apps.apple.com/us/app/hiddify-next/id6476212173'>App Store</a>\n"
        "💻 Windows: <a href='https://github.com/hiddify/hiddify-next/releases/latest/download/Hiddify-Windows-Setup-x64.exe'>Скачать</a>\n"
        "🖥 macOS: <a href='https://github.com/hiddify/hiddify-next/releases/latest'>GitHub</a>\n"
        "🐧 Linux: <a href='https://github.com/hiddify/hiddify-next/releases/latest/download/Hiddify-Linux-x64.AppImage'>Скачать</a>\n\n"
        "После установки выберите действие:")

    # Создаем 3 основные кнопки после показа ссылок
    action_kb = InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(
                text="🚀 Попробовать Тестовый Период",
                callback_data="activate_test_period",
            )
        ],
        [
            InlineKeyboardButton(
                text="💳 Выбрать Тариф для оплаты",
                callback_data="select_paid_plan"
            )
        ]
    ])

    # Отправляем изображение с подписью и кнопкой
    if not callback.message:
        await callback.answer()
        return
        
    # Сначала отправляем основную клавиатуру с кнопками меню (если её ещё нет)
    await callback.message.answer("Основное меню:", 
                                  reply_markup=get_main_menu_keyboard())
    
    # Используем безопасную функцию отправки изображения
    await safe_send_photo(callback, HIDDIFY_LOGO_PATH, caption, action_kb)

    await callback.answer()





@router.message(F.text == "Главное меню")
async def return_to_main_menu(message: Message):
    """Handle returning to the main menu"""
    await message.answer("Выберите действие из главного меню:",
                         reply_markup=get_main_menu_keyboard())


@router.message(F.text == "Информация о Тарифах")
async def show_tariff_info(message: Message):
    """Display information about available tariff plans"""
    tariff_text = generate_tariff_info_text()
    await message.answer(tariff_text,
                         reply_markup=get_main_menu_keyboard(),
                         parse_mode="HTML")


def register_handlers(dp):
    dp.include_router(router)
