"""
Обработчики для дополнительных сервисов
"""
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.utils.keyboard import InlineKeyboardBuilder

from bot.database.sqlite_tariffs import get_additional_services
from bot.utils.keyboard import get_main_menu_keyboard
import logging

logger = logging.getLogger(__name__)
router = Router()

@router.message(F.text == "Доп. Сервисы")
async def show_additional_services(message: Message):
    """Показать список дополнительных сервисов отдельными блоками"""
    try:
        services = get_additional_services()
        
        if not services:
            await message.answer(
                "🔧 Дополнительные сервисы временно недоступны.\n"
                "Пожалуйста, попробуйте позже.",
                reply_markup=get_main_menu_keyboard()
            )
            return
        
        # Сортируем услуги по приоритету (sort_order)
        services_sorted = sorted(services, key=lambda x: x.get('sort_order', 0))
        
        # Отправляем заголовок
        await message.answer("🛠 <b>Дополнительные сервисы</b>")
        
        # Отправляем каждую услугу отдельным блоком
        for service in services_sorted:
            # Формируем текст для каждой услуги
            service_text = f"📋 <b>{service['title']}</b>\n\n"
            service_text += f"{service['description']}"
            
            # Создаем клавиатуру для каждой услуги
            kb = InlineKeyboardBuilder()
            
            if service.get('link'):
                # Кнопка с прямой ссылкой
                kb.button(
                    text=f"🔗 {service.get('button_text', 'Перейти')}", 
                    url=service['link']
                )
            else:
                # Информационная кнопка для услуг без ссылки
                kb.button(
                    text="ℹ️ Скоро доступно", 
                    callback_data=f"service_info_{service['id']}"
                )
            
            # Отправляем блок с услугой
            await message.answer(service_text, reply_markup=kb.as_markup())
        
        # В конце отправляем кнопку возврата в главное меню
        back_kb = InlineKeyboardBuilder()
        back_kb.button(text="⬅️ В главное меню", callback_data="back_to_main_menu")
        
        await message.answer(
            "👆 Выберите интересующий вас сервис",
            reply_markup=back_kb.as_markup()
        )
        
    except Exception as e:
        logger.error(f"Ошибка при показе доп. сервисов: {e}")
        await message.answer(
            "❌ Произошла ошибка при загрузке сервисов.",
            reply_markup=get_main_menu_keyboard()
        )

@router.callback_query(F.data.startswith("service_info_"))
async def handle_service_info(callback: CallbackQuery):
    """Показ информации о сервисе без ссылки"""
    try:
        service_id = int(callback.data.split("_")[2])
        services = get_additional_services()
        
        # Находим выбранный сервис
        selected_service = None
        for service in services:
            if service['id'] == service_id:
                selected_service = service
                break
        
        if not selected_service:
            await callback.answer("Сервис не найден", show_alert=True)
            return
        
        # Показываем информацию о сервисе
        text = f"🛠 <b>{selected_service['title']}</b>\n\n"
        text += f"{selected_service['description']}\n\n"
        text += "📋 Данный сервис находится в разработке и скоро будет доступен."
        
        await callback.answer(text, show_alert=True)
        
    except Exception as e:
        logger.error(f"Ошибка при показе информации о сервисе: {e}")
        await callback.answer("Произошла ошибка", show_alert=True)

@router.callback_query(F.data == "back_to_services")
async def back_to_services(callback: CallbackQuery):
    """Возврат к списку сервисов"""
    # Создаем новое сообщение с текстом "Доп. Сервисы" для обработки
    fake_message = type('obj', (object,), {
        'answer': callback.message.answer,
        'text': "Доп. Сервисы"
    })()
    
    # Перенаправляем на показ сервисов
    await show_additional_services(fake_message)
    await callback.answer()

@router.callback_query(F.data == "back_to_main_menu")
async def back_to_main_menu(callback: CallbackQuery):
    """Возврат в главное меню"""
    await callback.message.answer(
        "🏠 Вы вернулись в главное меню",
        reply_markup=get_main_menu_keyboard()
    )
    await callback.message.delete()
    await callback.answer()