"""
Адаптер для работы с тарифами из SQLite вместо config.py
"""
from database.sqlite_config import config_db
from typing import Dict, List, Optional
import logging

logger = logging.getLogger(__name__)

def get_subscription_plans() -> Dict:
    """Получить все тарифы в формате совместимом с SUBSCRIPTION_PLANS"""
    try:
        tariffs = config_db.get_all_tariffs()
        plans = {}
        
        for tariff in tariffs:
            plans[tariff['id']] = {
                'name': tariff['name'],
                'price': tariff['price'],
                'traffic_gb': tariff['traffic_gb'],
                'duration_days': tariff['duration_days'],
                'description': tariff['description'] or '',
                'discount_percent': tariff.get('discount_percent', 0),
                'price_per_30_days': tariff.get('price_per_30_days', 0)
            }
        
        return plans
    except Exception as e:
        logger.error(f"Ошибка получения тарифов из SQLite: {e}")
        # Fallback к статическим тарифам если SQLite недоступен
        return {
            "trial": {
                "name": "Старт",
                "price": 0,
                "traffic_gb": 1,
                "duration_days": 3,
                "description": "Тестовый тариф: 1 ГБ на 3 дня"
            },
            "basic": {
                "name": "Базовый",
                "price": 149,
                "traffic_gb": 150,
                "duration_days": 30,
                "description": "Базовый тариф: на 30 дней -150 ГБ трафика"
            }
        }

def get_additional_services() -> List[Dict]:
    """Получить список дополнительных услуг"""
    try:
        return config_db.get_all_services()
    except Exception as e:
        logger.error(f"Ошибка получения доп. услуг: {e}")
        return []

def update_tariff_price(tariff_id: str, new_price: int) -> bool:
    """Обновить цену тарифа"""
    try:
        tariff = config_db.get_tariff(tariff_id)
        if tariff:
            tariff['price'] = new_price
            return config_db.create_or_update_tariff(tariff)
        return False
    except Exception as e:
        logger.error(f"Ошибка обновления цены тарифа: {e}")
        return False