"""
PostgreSQL операции для бота VPN
Замена SQLite операций для работы с PostgreSQL
"""
import os
import logging
from datetime import datetime, timedelta
from typing import List, Optional, Dict, Any, Tuple
from contextlib import contextmanager
from sqlalchemy import create_engine, text

from bot.database.models import User, Subscription, Payment, UserStatus, PaymentStatus

logger = logging.getLogger(__name__)

# Используем PostgreSQL подключение
DATABASE_URL = os.environ['DATABASE_URL']

@contextmanager
def get_postgres_connection():
    """Получение соединения с PostgreSQL"""
    engine = create_engine(DATABASE_URL)
    conn = engine.connect()
    try:
        yield conn
    finally:
        conn.close()

async def get_user_subscription_postgres(user_id: int) -> Optional[Subscription]:
    """Получить активную подписку пользователя из PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT id, user_id, plan_id, created_at, expires_at, 
                       traffic_limit, traffic_used, vpn_user_id, vpn_key, 
                       status, vpn_email
                FROM subscriptions 
                WHERE user_id = :user_id 
                AND expires_at > NOW()
                AND (status IS NULL OR status != 'EXPIRED')
                ORDER BY expires_at DESC 
                LIMIT 1
            """), {"user_id": user_id})
            
            row = result.fetchone()
            
            if row:
                return Subscription(
                    id=row.id,
                    user_id=row.user_id,
                    plan_id=row.plan_id,
                    created_at=row.created_at,
                    expires_at=row.expires_at,
                    traffic_limit=float(row.traffic_limit) if row.traffic_limit else 0.0,
                    traffic_used=float(row.traffic_used) if row.traffic_used else 0.0,
                    vpn_user_id=row.vpn_user_id,
                    vpn_key=row.vpn_key,
                    status=row.status,
                    vpn_email=row.vpn_email
                )
        
        return None
        
    except Exception as e:
        logger.error(f"Ошибка получения подписки пользователя {user_id}: {e}")
        return None

async def get_user_postgres(user_id: int) -> Optional[User]:
    """Получить пользователя из PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT id, username, telegram_username, first_name, last_name, created_at, 
                       status, phone_number, phone, vpn_email, email
                FROM users 
                WHERE id = :user_id
            """), {"user_id": user_id})
            
            row = result.fetchone()
            
            if row:
                return User(
                    id=row.id,
                    username=row.username or row.telegram_username,  # Приоритет username, fallback telegram_username
                    first_name=row.first_name,
                    last_name=row.last_name,
                    created_at=row.created_at,
                    status=getattr(UserStatus, row.status, UserStatus.NEW) if row.status else UserStatus.NEW,
                    phone_number=row.phone_number or row.phone,  # Приоритет phone_number, fallback phone
                    vpn_email=row.vpn_email or row.email  # Приоритет vpn_email, fallback email
                )
        
        return None
        
    except Exception as e:
        logger.error(f"Ошибка получения пользователя {user_id}: {e}")
        return None

async def get_user_payments_postgres(user_id: int, limit: int = 10) -> List[Payment]:
    """Получить платежи пользователя из PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT id, user_id, subscription_id, plan_id, amount, currency,
                       payment_id_yookassa, status, created_at, plan_name
                FROM payments 
                WHERE user_id = :user_id
                ORDER BY created_at DESC
                LIMIT :limit
            """), {"user_id": user_id, "limit": limit})
            
            payments = []
            for row in result.fetchall():
                payment = Payment(
                    id=row.id,
                    user_id=row.user_id,
                    subscription_id=row.subscription_id,
                    plan_id=row.plan_id,
                    amount=float(row.amount),
                    currency=row.currency,
                    payment_id_yookassa=row.payment_id_yookassa,
                    status=PaymentStatus[row.status] if row.status else PaymentStatus.PENDING,
                    created_at=row.created_at,
                    plan_name=row.plan_name
                )
                payments.append(payment)
            
            return payments
        
    except Exception as e:
        logger.error(f"Ошибка получения платежей пользователя {user_id}: {e}")
        return []

async def create_or_update_user_postgres(user: User) -> Optional[User]:
    """Создать или обновить пользователя в PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            user_data = {
                "id": user.id,
                "username": user.username,
                "telegram_username": user.username,  # Заполняем оба поля одинаково
                "first_name": user.first_name,
                "last_name": user.last_name,
                "phone_number": user.phone_number,
                "phone": user.phone_number,  # Заполляем оба поля phone
                "created_at": user.created_at or datetime.now(),
                "status": user.status.name if hasattr(user.status, 'name') else str(user.status),
                "vpn_email": user.vpn_email,
                "email": user.vpn_email  # Синхронизируем email с vpn_email
            }
            
            result = conn.execute(text("""
                INSERT INTO users (id, username, telegram_username, first_name, last_name, 
                                 phone_number, phone, created_at, status, vpn_email, email)
                VALUES (:id, :username, :telegram_username, :first_name, :last_name, 
                        :phone_number, :phone, :created_at, :status, :vpn_email, :email)
                ON CONFLICT (id) DO UPDATE SET
                    username = EXCLUDED.username,
                    telegram_username = EXCLUDED.telegram_username,
                    first_name = EXCLUDED.first_name,
                    last_name = EXCLUDED.last_name,
                    phone_number = COALESCE(EXCLUDED.phone_number, users.phone_number),
                    phone = COALESCE(EXCLUDED.phone, users.phone),
                    status = COALESCE(EXCLUDED.status, users.status),
                    vpn_email = COALESCE(EXCLUDED.vpn_email, users.vpn_email),
                    email = COALESCE(EXCLUDED.email, users.email)
                RETURNING *
            """), user_data)
            
            conn.commit()
            row = result.fetchone()
            
            if row:
                return User(
                    id=row.id,
                    username=row.username,
                    first_name=row.first_name,
                    last_name=row.last_name,
                    created_at=row.created_at,
                    status=getattr(UserStatus, row.status, UserStatus.NEW) if row.status else UserStatus.NEW,
                    phone_number=row.phone_number,
                    vpn_email=row.vpn_email
                )
        
        return None
        
    except Exception as e:
        logger.error(f"Ошибка создания/обновления пользователя: {e}")
        return None

async def create_subscription_postgres(subscription_data: dict) -> Optional[Subscription]:
    """Создать подписку в PostgreSQL"""
    try:
        logger.info(f"Создание подписки с данными: {subscription_data}")
        
        with get_postgres_connection() as conn:
            # Правильный синтаксис SQLAlchemy для передачи параметров
            result = conn.execute(text("""
                INSERT INTO subscriptions (user_id, plan_id, created_at, expires_at, 
                                         traffic_limit, traffic_used, vpn_user_id, 
                                         vpn_key, status, vpn_email)
                VALUES (:user_id, :plan_id, :created_at, :expires_at, 
                        :traffic_limit, :traffic_used, :vpn_user_id, 
                        :vpn_key, :status, :vpn_email)
                RETURNING *
            """), subscription_data)
            
            conn.commit()
            row = result.fetchone()
            
            if row:
                logger.info(f"Подписка успешно создана с ID: {row.id}")
                return Subscription(
                    id=row.id,
                    user_id=row.user_id,
                    plan_id=row.plan_id,
                    created_at=row.created_at,
                    expires_at=row.expires_at,
                    traffic_limit=float(row.traffic_limit) if row.traffic_limit else 0.0,
                    traffic_used=float(row.traffic_used) if row.traffic_used else 0.0,
                    vpn_user_id=row.vpn_user_id,
                    vpn_key=row.vpn_key,
                    status=row.status,
                    vpn_email=row.vpn_email
                )
        
        logger.error("Не удалось получить данные созданной подписки")
        return None
        
    except Exception as e:
        logger.error(f"Ошибка создания подписки: {e}")
        logger.error(f"Данные подписки: {subscription_data}")
        import traceback
        logger.error(f"Traceback: {traceback.format_exc()}")
        return None

async def update_subscription_postgres(subscription_id: int, updates: dict) -> bool:
    """Обновить подписку в PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            # Формируем SET часть запроса
            set_parts = []
            params = {"subscription_id": subscription_id}
            
            for key, value in updates.items():
                set_parts.append(f"{key} = :{key}")
                params[key] = value
            
            if not set_parts:
                return False
            
            set_clause = ", ".join(set_parts)
            
            result = conn.execute(text(f"""
                UPDATE subscriptions 
                SET {set_clause}
                WHERE id = :subscription_id
            """), params)
            
            conn.commit()
            return result.rowcount > 0
        
    except Exception as e:
        logger.error(f"Ошибка обновления подписки {subscription_id}: {e}")
        return False

async def has_user_used_trial_postgres(user_id: int) -> bool:
    """Проверить, использовал ли пользователь тестовый период"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT COUNT(*) as count
                FROM subscriptions 
                WHERE user_id = :user_id AND plan_id = 'trial'
            """), {"user_id": user_id})
            
            row = result.fetchone()
            return row.count > 0 if row else False
        
    except Exception as e:
        logger.error(f"Ошибка проверки тестового периода для пользователя {user_id}: {e}")
        return False

async def get_expiring_subscriptions_postgres(days_before: int) -> List[Tuple[User, Subscription]]:
    """Получить подписки, истекающие через указанное количество дней из PostgreSQL"""
    try:
        from datetime import datetime, timedelta
        
        target_date = datetime.now() + timedelta(days=days_before)
        start_of_day = target_date.replace(hour=0, minute=0, second=0, microsecond=0)
        end_of_day = target_date.replace(hour=23, minute=59, second=59, microsecond=999999)
        
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT u.id, u.username, u.first_name, u.last_name, u.vpn_email, u.user_status,
                       s.id, s.user_id, s.plan_id, s.created_at, s.expires_at, 
                       s.traffic_limit, s.traffic_used, s.vpn_user_id, s.vpn_key, 
                       s.status, s.vpn_email as subscription_email
                FROM users u
                JOIN subscriptions s ON u.id = s.user_id
                WHERE s.status = 'ACTIVE'
                AND s.expires_at BETWEEN :start_date AND :end_date
                ORDER BY s.expires_at ASC
            """), {
                "start_date": start_of_day,
                "end_date": end_of_day
            })
            
            results = []
            for row in result.fetchall():
                user = User(
                    id=row[0],
                    username=row[1],
                    first_name=row[2],
                    last_name=row[3],
                    vpn_email=row[4],
                    user_status=row[5]
                )
                
                subscription = Subscription(
                    id=row[6],
                    user_id=row[7],
                    plan_id=row[8],
                    created_at=row[9],
                    expires_at=row[10],
                    traffic_limit=float(row[11]) if row[11] else 0.0,
                    traffic_used=float(row[12]) if row[12] else 0.0,
                    vpn_user_id=row[13],
                    vpn_key=row[14],
                    status=row[15],
                    vpn_email=row[16]
                )
                
                results.append((user, subscription))
            
            return results
        
    except Exception as e:
        logger.error(f"Ошибка получения истекающих подписок через {days_before} дней: {e}")
        return []

async def get_subscription_by_id_postgres(subscription_id: int) -> Optional[Subscription]:
    """Получить подписку по ID из PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT id, user_id, plan_id, created_at, expires_at, 
                       traffic_limit, traffic_used, vpn_user_id, vpn_key, 
                       status, vpn_email
                FROM subscriptions 
                WHERE id = :subscription_id
            """), {"subscription_id": subscription_id})
            
            row = result.fetchone()
            
            if row:
                return Subscription(
                    id=row.id,
                    user_id=row.user_id,
                    plan_id=row.plan_id,
                    created_at=row.created_at,
                    expires_at=row.expires_at,
                    traffic_limit=float(row.traffic_limit) if row.traffic_limit else 0.0,
                    traffic_used=float(row.traffic_used) if row.traffic_used else 0.0,
                    vpn_user_id=row.vpn_user_id,
                    vpn_key=row.vpn_key,
                    status=row.status,
                    vpn_email=row.vpn_email
                )
        
        return None
        
    except Exception as e:
        logger.error(f"Ошибка получения подписки {subscription_id}: {e}")
        return None


async def get_all_active_subscriptions_postgres() -> List[Subscription]:
    """Получить все активные подписки из PostgreSQL"""
    try:
        import os
        from sqlalchemy import create_engine, text
        
        engine = create_engine(os.environ.get('DATABASE_URL'))
        
        with engine.connect() as conn:
            result = conn.execute(text("""
                SELECT id, user_id, plan_id, created_at, expires_at, 
                       traffic_limit, traffic_used, vpn_user_id, vpn_key, 
                       status, vpn_email
                FROM subscriptions 
                WHERE status = 'ACTIVE' AND expires_at > NOW()
                ORDER BY expires_at ASC
            """))
            
            subscriptions = []
            for row in result.fetchall():
                subscription = Subscription(
                    id=row[0],
                    user_id=row[1],
                    plan_id=row[2],
                    created_at=row[3],
                    expires_at=row[4],
                    traffic_limit=int(row[5]) if row[5] else 0,
                    traffic_used=int(row[6]) if row[6] else 0,
                    vpn_user_id=row[7],
                    vpn_key=row[8],
                    status=row[9],
                    vpn_email=row[10]
                )
                subscriptions.append(subscription)
            
            return subscriptions
    
    except Exception as e:
        logger.error(f"Ошибка получения активных подписок: {e}")
        return []


async def update_traffic_usage_postgres(user_id: int, traffic_used: int) -> bool:
    """Обновить использованный трафик пользователя в PostgreSQL"""
    try:
        conn = get_postgres_connection()
        with conn.cursor() as cursor:
            cursor.execute("""
                UPDATE subscriptions 
                SET traffic_used = %s 
                WHERE user_id = %s AND status = 'ACTIVE'
            """, (traffic_used, user_id))
            
            conn.commit()
            return cursor.rowcount > 0
    
    except Exception as e:
        logger.error(f"Ошибка обновления трафика для пользователя {user_id}: {e}")
        return False
    finally:
        if 'conn' in locals():
            conn.close()


async def get_low_traffic_subscriptions_postgres(threshold_percent: float) -> List[Tuple[User, Subscription]]:
    """Получить подписки с низким трафиком из PostgreSQL"""
    try:
        conn = get_postgres_connection()
        with conn.cursor() as cursor:
            cursor.execute("""
                SELECT u.id, u.username, u.first_name, u.last_name, u.vpn_email, u.user_status,
                       s.id, s.user_id, s.plan_id, s.created_at, s.expires_at, 
                       s.traffic_limit, s.traffic_used, s.vpn_user_id, s.vpn_key, 
                       s.status, s.vpn_email as subscription_email
                FROM users u
                JOIN subscriptions s ON u.id = s.user_id
                WHERE s.status = 'ACTIVE' 
                AND s.traffic_limit > 0
                AND (CAST(s.traffic_used AS FLOAT) / CAST(s.traffic_limit AS FLOAT) * 100) >= %s
                ORDER BY s.expires_at ASC
            """, (threshold_percent,))
            
            results = []
            for row in cursor.fetchall():
                user = User(
                    id=row[0],
                    username=row[1],
                    first_name=row[2],
                    last_name=row[3],
                    vpn_email=row[4],
                    user_status=row[5]
                )
                
                subscription = Subscription(
                    id=row[6],
                    user_id=row[7],
                    plan_id=row[8],
                    created_at=row[9],
                    expires_at=row[10],
                    traffic_limit=int(row[11]) if row[11] else 0,
                    traffic_used=int(row[12]) if row[12] else 0,
                    vpn_user_id=row[13],
                    vpn_key=row[14],
                    status=row[15],
                    vpn_email=row[16]
                )
                
                results.append((user, subscription))
            
            return results
    
    except Exception as e:
        logger.error(f"Ошибка получения подписок с низким трафиком: {e}")
        return []
    finally:
        if 'conn' in locals():
            conn.close()


async def create_payment_postgres(payment: Payment) -> bool:
    """Создать платеж в PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                INSERT INTO payments (user_id, subscription_id, amount, currency, payment_id, status, plan_id, created_at)
                VALUES (:user_id, :subscription_id, :amount, :currency, :payment_id, :status, :plan_id, :created_at)
                RETURNING id
            """), {
                "user_id": payment.user_id,
                "subscription_id": payment.subscription_id,
                "amount": payment.amount,
                "currency": payment.currency,
                "payment_id": payment.payment_id,
                "status": str(payment.status),
                "plan_id": payment.plan_id,
                "created_at": payment.created_at or datetime.now()
            })
            
            conn.commit()
            return True
            
    except Exception as e:
        logger.error(f"Ошибка создания платежа: {e}")
        return False


async def get_payment_by_payment_id_postgres(payment_id: str) -> Optional[Payment]:
    """Получить платеж по payment_id из PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT id, user_id, subscription_id, amount, currency, payment_id, status, plan_id, created_at
                FROM payments 
                WHERE payment_id = :payment_id
            """), {"payment_id": payment_id})
            
            row = result.fetchone()
            
            if row:
                # Обрабатываем статус платежа (может быть строкой или числом)
                status_value = row.status
                if isinstance(status_value, str):
                    if status_value == '1':
                        payment_status = PaymentStatus.PENDING
                    elif status_value == '2':
                        payment_status = PaymentStatus.SUCCEEDED  
                    elif status_value == '3':
                        payment_status = PaymentStatus.FAILED
                    elif status_value.upper() == 'PENDING':
                        payment_status = PaymentStatus.PENDING
                    elif status_value.upper() == 'SUCCEEDED':
                        payment_status = PaymentStatus.SUCCEEDED
                    elif status_value.upper() == 'FAILED':
                        payment_status = PaymentStatus.FAILED
                    else:
                        payment_status = PaymentStatus.PENDING
                else:
                    payment_status = PaymentStatus(status_value)
                
                return Payment(
                    id=row.id,
                    user_id=row.user_id,
                    subscription_id=row.subscription_id,
                    amount=row.amount,
                    currency=row.currency,
                    payment_id=row.payment_id,
                    status=payment_status,
                    plan_id=row.plan_id,
                    created_at=row.created_at
                )
            return None
            
    except Exception as e:
        logger.error(f"Ошибка получения платежа: {e}")
        return None


async def update_payment_status_postgres(payment_id: str, status: str) -> bool:
    """Обновить статус платежа в PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                UPDATE payments 
                SET status = :status
                WHERE payment_id = :payment_id
            """), {
                "status": status,
                "payment_id": payment_id
            })
            
            conn.commit()
            return result.rowcount > 0
            
    except Exception as e:
        logger.error(f"Ошибка обновления статуса платежа: {e}")
        return False


async def get_subscription_by_id_postgres(subscription_id: int) -> Optional[Subscription]:
    """Получить подписку по ID из PostgreSQL"""
    try:
        with get_postgres_connection() as conn:
            result = conn.execute(text("""
                SELECT id, user_id, plan_id, created_at, expires_at, 
                       traffic_limit, traffic_used, vpn_user_id, vpn_key, 
                       status, vpn_email
                FROM subscriptions 
                WHERE id = :subscription_id
            """), {"subscription_id": subscription_id})
            
            row = result.fetchone()
            
            if row:
                return Subscription(
                    id=row.id,
                    user_id=row.user_id,
                    plan_id=row.plan_id,
                    created_at=row.created_at,
                    expires_at=row.expires_at,
                    traffic_limit=int(row.traffic_limit),
                    traffic_used=int(row.traffic_used),
                    vpn_user_id=row.vpn_user_id,
                    vpn_key=row.vpn_key,
                    status=row.status,
                    vpn_email=row.vpn_email
                )
            return None
            
    except Exception as e:
        logger.error(f"Ошибка получения подписки по ID: {e}")
        return None


async def get_user_active_subscription_postgres(user_id: int) -> Optional[dict]:
    """Получить активную подписку пользователя из PostgreSQL"""
    try:
        from sqlalchemy import create_engine, text
        import os
        
        # Создаем подключение к PostgreSQL
        engine = create_engine(os.environ['DATABASE_URL'])
        
        with engine.connect() as conn:
            # Исправлено: ищем по expires_at > NOW() вместо статуса
            # Подходят статусы 'ACTIVE', 'PENDING' - главное чтобы не истекла
            query = text("""
                SELECT id, user_id, plan_id, created_at, expires_at, 
                       traffic_limit, traffic_used, vpn_user_id, vpn_key, 
                       status, vpn_email
                FROM subscriptions 
                WHERE user_id = :user_id 
                AND expires_at > NOW()
                AND status IN ('ACTIVE', 'PENDING')
                ORDER BY created_at DESC
                LIMIT 1
            """)
            
            result = conn.execute(query, {"user_id": user_id})
            row = result.fetchone()
            
            if row:
                return {
                    'id': row[0],
                    'user_id': row[1],
                    'plan_id': row[2],
                    'created_at': row[3],
                    'expires_at': row[4],
                    'traffic_limit': row[5],
                    'traffic_used': row[6],
                    'vpn_user_id': row[7],
                    'vpn_key': row[8],
                    'status': row[9],
                    'vpn_email': row[10]
                }
            return None
            
    except Exception as e:
        logger.error(f"Ошибка получения активной подписки пользователя {user_id}: {e}")
        return None


async def delete_payment_by_subscription_postgres(subscription_id: int) -> bool:
    """Удалить все платежи связанные с подпиской из PostgreSQL"""
    try:
        async with get_postgres_connection() as conn:
            query = "DELETE FROM payments WHERE subscription_id = %s"
            await conn.execute(query, subscription_id)
            logger.info(f"Удалены платежи для подписки {subscription_id}")
            return True
            
    except Exception as e:
        logger.error(f"Ошибка удаления платежей для подписки {subscription_id}: {e}")
        return False


async def delete_subscription_postgres(subscription_id: int) -> bool:
    """Удалить подписку из PostgreSQL"""
    try:
        async with get_postgres_connection() as conn:
            query = "DELETE FROM subscriptions WHERE id = %s"
            result = await conn.execute(query, subscription_id)
            logger.info(f"Удалена подписка {subscription_id}")
            return True
            
    except Exception as e:
        logger.error(f"Ошибка удаления подписки {subscription_id}: {e}")
        return False