from datetime import datetime, timedelta

# Константы статусов как простые строки
class UserStatus:
    NEW = "NEW"
    ACTIVE = "ACTIVE"
    SUSPENDED = "SUSPENDED"
    EXPIRED = "EXPIRED"

class PaymentStatus:
    PENDING = "PENDING"
    SUCCEEDED = "SUCCEEDED"
    CANCELED = "CANCELED"
    FAILED = "FAILED"


class User:
    def __init__(self, id, username=None, first_name=None, last_name=None, 
                 created_at=None, status=None, phone_number=None, vpn_email=None, user_status=None):
        self.id = id
        self.username = username
        self.first_name = first_name
        self.last_name = last_name
        self.created_at = created_at or datetime.now()
        self.status = status or UserStatus.NEW
        self.phone_number = phone_number
        self.vpn_email = vpn_email
        self.user_status = user_status

    @property
    def full_name(self):
        if self.first_name and self.last_name:
            return f"{self.first_name} {self.last_name}"
        elif self.first_name:
            return self.first_name
        elif self.username:
            return self.username
        else:
            return f"User {self.id}"
        

class Subscription:
    def __init__(self, id=None, user_id=None, plan_id=None, created_at=None, 
                 expires_at=None, traffic_limit=0, traffic_used=0, 
                 vpn_user_id=None, vpn_key=None, status=None, vpn_email=None):
        self.id = id
        self.user_id = user_id
        self.plan_id = plan_id
        self.created_at = created_at or datetime.now()
        self.expires_at = expires_at
        self.traffic_limit = float(traffic_limit) if traffic_limit is not None else 0.0
        self.traffic_used = float(traffic_used) if traffic_used is not None else 0.0
        self.vpn_user_id = vpn_user_id
        self.vpn_key = vpn_key
        self.status = status
        self.vpn_email = vpn_email  # Email клиента в панели для синхронизации
        
    @property
    def traffic_remaining(self):
        # Нормализуем единицы измерения
        traffic_limit_bytes = self.traffic_limit
        traffic_used_bytes = self.traffic_used
        
        # Если traffic_limit в ГБ (значение меньше 1000), конвертируем в байты
        if traffic_limit_bytes < 1000:
            traffic_limit_bytes = traffic_limit_bytes * (1024**3)
        
        # Если traffic_used в ГБ (значение меньше 1000), конвертируем в байты  
        if traffic_used_bytes < 1000:
            traffic_used_bytes = traffic_used_bytes * (1024**3)
            
        return max(0, traffic_limit_bytes - traffic_used_bytes)
    
    @property
    def traffic_percent_used(self):
        # Нормализуем единицы измерения
        traffic_limit_bytes = self.traffic_limit
        traffic_used_bytes = self.traffic_used
        
        # Если traffic_limit в ГБ (значение меньше 1000), конвертируем в байты
        if traffic_limit_bytes < 1000:
            traffic_limit_bytes = traffic_limit_bytes * (1024**3)
        
        # Если traffic_used в ГБ (значение меньше 1000), конвертируем в байты  
        if traffic_used_bytes < 1000:
            traffic_used_bytes = traffic_used_bytes * (1024**3)
            
        if traffic_limit_bytes == 0:
            return 100
        return (traffic_used_bytes / traffic_limit_bytes) * 100
    
    @property
    def is_active(self):
        return (self.expires_at > datetime.now() and 
                self.traffic_remaining > 0)
    
    @property
    def days_remaining(self):
        if not self.expires_at:
            return 0
        delta = self.expires_at - datetime.now()
        return max(0, delta.days)


class Payment:
    def __init__(self, id=None, user_id=None, subscription_id=None,
                 amount=0, currency="RUB", payment_id=None, 
                 created_at=None, completed_at=None, 
                 status=PaymentStatus.PENDING, plan_id=None):
        self.id = id
        self.user_id = user_id
        self.subscription_id = subscription_id
        self.amount = float(amount) if amount is not None else 0.0
        self.currency = currency
        self.payment_id = payment_id
        self.created_at = created_at or datetime.now()
        self.completed_at = completed_at
        self.status = status
        self.plan_id = plan_id
