import os
from datetime import datetime, timedelta
import threading
import time

from flask import Flask, render_template, request, redirect, url_for, jsonify, flash, make_response
from flask_sqlalchemy import SQLAlchemy
from sqlalchemy.orm import DeclarativeBase
from werkzeug.middleware.proxy_fix import ProxyFix


class Base(DeclarativeBase):
    pass


db = SQLAlchemy(model_class=Base)
# create the app
app = Flask(__name__)
app.secret_key = os.environ.get("SESSION_SECRET", "a_secret_key_for_development")
app.wsgi_app = ProxyFix(
    app.wsgi_app, x_proto=1, x_host=1
)  # needed for url_for to generate with https

# configure the database, relative to the app instance folder
from config import DATABASE_FILE

# Принудительно используем SQLite, игнорируя DATABASE_URL
app.config["SQLALCHEMY_DATABASE_URI"] = f"sqlite:///{DATABASE_FILE}"
app.config["SQLALCHEMY_ENGINE_OPTIONS"] = {
    "pool_recycle": 300,
    "pool_pre_ping": True,
}
# initialize the app with the extension, flask-sqlalchemy >= 3.0.x
db.init_app(app)

# Регистрируем blueprint для системы бекапов
from backup_web_interface import backup_bp
app.register_blueprint(backup_bp)

# Простая система кеширования
_cache = {}
_cache_lock = threading.Lock()

def get_cached_data(key, max_age_minutes=5):
    """Получить данные из кеша если они еще актуальны"""
    with _cache_lock:
        if key in _cache:
            data, timestamp = _cache[key]
            if datetime.now() - timestamp < timedelta(minutes=max_age_minutes):
                return data
    return None

def set_cached_data(key, data):
    """Сохранить данные в кеш"""
    with _cache_lock:
        _cache[key] = (data, datetime.now())
        # Очищаем старые записи (простая очистка)
        cutoff = datetime.now() - timedelta(minutes=30)
        keys_to_remove = [k for k, (_, ts) in _cache.items() if ts < cutoff]
        for k in keys_to_remove:
            del _cache[k]


@app.route("/")
def index():
    """Главная страница с информацией о VPN сервисе"""
    return render_template("index.html")


def run_async_safely(coro):
    """Безопасный запуск асинхронных функций в синхронном контексте Flask"""
    import asyncio
    try:
        # Попробуем использовать существующий event loop
        loop = asyncio.get_event_loop()
        if loop.is_running():
            # Если loop уже запущен, создаем задачу
            import concurrent.futures
            with concurrent.futures.ThreadPoolExecutor() as executor:
                future = executor.submit(asyncio.run, coro)
                return future.result(timeout=30)  # 30 секунд таймаут
        else:
            return loop.run_until_complete(coro)
    except RuntimeError:
        # Если нет активного loop, создаем новый
        return asyncio.run(coro)
    except Exception as e:
        print(f"Ошибка в run_async_safely: {e}")
        return None


@app.route("/status")
def status():
    """Страница со статусом сервисов и статистикой системы"""
    from web_stats import get_web_statistics

    # Получаем реальную статистику из базы данных
    try:
        stats = run_async_safely(get_web_statistics())
    except Exception as e:
        # В случае ошибки используем базовую статистику
        from datetime import datetime

        stats = {
            "total_users": 0,
            "active_subscriptions": 0,
            "trial_subscriptions": 0,
            "expiring_soon": 0,
            "bot_version": "1.2.0",
            "last_restart": datetime.now().strftime("%d.%m.%Y %H:%M"),
            "uptime": "0:00:00",
            "last_sync": datetime.now().strftime("%d.%m.%Y %H:%M"),
            "sync_success_rate": 0,
            "error": str(e),
        }

    return render_template("status.html", **stats)


@app.route("/api/status")
def api_status():
    """API эндпоинт для проверки статуса приложения"""
    return jsonify({"status": "ok", "service": "VPN Bot API", "version": "1.0.0"})


@app.route("/api/users")
def api_users():
    """API эндпоинт для получения данных пользователей без перезагрузки страницы"""
    try:
        import sqlite3
        from datetime import datetime

        conn = sqlite3.connect(DATABASE_FILE)
        conn.row_factory = sqlite3.Row
        cursor = conn.cursor()

        query = """
        SELECT 
            u.id as user_id,
            u.username,
            u.first_name,
            u.last_name,
            u.created_at as user_created_at,
            u.status,
            u.vpn_email,
            COUNT(s.id) as subscriptions_count,
            SUM(CASE WHEN s.status = 'ACTIVE' THEN 1 ELSE 0 END) as active_subscriptions,
            MAX(s.expires_at) as latest_expiry,
            SUM(CASE WHEN s.status = 'ACTIVE' THEN s.traffic_used ELSE 0 END) as total_traffic_used,
            SUM(CASE WHEN s.status = 'ACTIVE' THEN s.traffic_limit ELSE 0 END) as total_traffic_limit
        FROM users u
        LEFT JOIN subscriptions s ON u.id = s.user_id
        GROUP BY u.id, u.username, u.first_name, u.last_name, u.created_at, u.status, u.vpn_email
        ORDER BY u.created_at DESC
        LIMIT 50
        """

        cursor.execute(query)
        users_data = cursor.fetchall()
        conn.close()

        users_list = []
        for user in users_data:
            user_dict = dict(user)
            user_dict["subscriptions_count"] = user_dict["subscriptions_count"] or 0
            user_dict["active_subscriptions"] = user_dict["active_subscriptions"] or 0
            user_dict["total_traffic_used"] = user_dict["total_traffic_used"] or 0
            user_dict["total_traffic_limit"] = user_dict["total_traffic_limit"] or 0

            # Вычисляем процент использования трафика
            # traffic_limit в ГБ, traffic_used в байтах - приводим к одинаковым единицам
            if user_dict["total_traffic_limit"] > 0:
                traffic_limit_bytes = user_dict["total_traffic_limit"] * 1024 * 1024 * 1024
                user_dict["traffic_percentage"] = round(
                    (user_dict["total_traffic_used"] / traffic_limit_bytes) * 100, 1
                )
            else:
                user_dict["traffic_percentage"] = 0

            users_list.append(user_dict)

        return jsonify(
            {
                "status": "success",
                "users": users_list,
                "count": len(users_list),
                "last_update": datetime.now().strftime("%d.%m.%Y %H:%M:%S"),
            }
        )

    except Exception as e:
        return jsonify({"status": "error", "message": str(e)}), 500


@app.route("/users")
def users():
    """Страница со списком всех пользователей и их данными с пагинацией"""
    import sqlite3
    from datetime import datetime

    # Получаем параметры пагинации
    page = request.args.get('page', 1, type=int)
    per_page = request.args.get('per_page', 50, type=int)
    
    # Ограничиваем количество элементов на странице
    per_page = min(per_page, 100)  # Максимум 100 пользователей на страницу
    
    # Вычисляем OFFSET
    offset = (page - 1) * per_page

    try:
        # Проверяем кеш с учетом пагинации
        cache_key = f"users_data_page_{page}_per_{per_page}"
        cached_result = get_cached_data(cache_key, max_age_minutes=10)
        if cached_result:
            print(f"Используем данные из кеша для страницы {page} (10 минут)")
            return cached_result

        print(f"Загружаем данные страницы {page} из базы (без принудительной синхронизации трафика)")

        # Подключаемся к базе данных
        conn = sqlite3.connect(DATABASE_FILE)
        conn.row_factory = sqlite3.Row  # Для получения данных как словарь
        cursor = conn.cursor()

        # Сначала получаем общее количество пользователей для пагинации
        count_query = "SELECT COUNT(*) as total FROM users"
        cursor.execute(count_query)
        total_users = cursor.fetchone()["total"]
        
        # Вычисляем общее количество страниц
        total_pages = (total_users + per_page - 1) // per_page

        # Получаем пользователей с пагинацией
        query = """
        SELECT 
            u.id as user_id,
            u.username,
            u.first_name,
            u.last_name,
            u.created_at as user_created_at,
            u.status,
            u.vpn_email,
            COUNT(DISTINCT s.id) as subscriptions_count,
            SUM(CASE WHEN s.status = 'ACTIVE' THEN 1 ELSE 0 END) as active_subscriptions,
            MAX(s.expires_at) as latest_expiry,
            COALESCE((SELECT SUM(amount) FROM payments WHERE user_id = u.id AND status = 'SUCCEEDED'), 0) as total_paid,
            SUM(CASE WHEN s.status = 'ACTIVE' THEN s.traffic_used ELSE 0 END) as total_traffic_used,
            SUM(CASE WHEN s.status = 'ACTIVE' THEN s.traffic_limit ELSE 0 END) as total_traffic_limit
        FROM users u
        LEFT JOIN subscriptions s ON u.id = s.user_id
        GROUP BY u.id, u.username, u.first_name, u.last_name, u.created_at, u.status, u.vpn_email
        ORDER BY u.created_at DESC
        LIMIT ? OFFSET ?
        """

        cursor.execute(query, (per_page, offset))
        users_data = cursor.fetchall()

        # Инициализируем структуры данных
        subscriptions_by_user = {}
        payments_by_user = {}
        
        # Только если есть пользователи, выполняем запросы
        if users_data:
            user_ids = [user["user_id"] for user in users_data]
            placeholders = ','.join('?' * len(user_ids))
            
            # Получаем все подписки одним запросом
            subscriptions_query = f"""
            SELECT 
                user_id, id, plan_id, status, traffic_limit, traffic_used, 
                created_at, expires_at, vpn_user_id, vpn_key, vpn_email
            FROM subscriptions 
            WHERE user_id IN ({placeholders})
            ORDER BY user_id, created_at DESC
            """
            cursor.execute(subscriptions_query, user_ids)
            all_subscriptions = cursor.fetchall()

            # Получаем все платежи одним запросом
            payments_query = f"""
            SELECT 
                user_id, id, amount, status, created_at, currency, payment_id,
                ROW_NUMBER() OVER (PARTITION BY user_id ORDER BY created_at DESC) as rn
            FROM payments 
            WHERE user_id IN ({placeholders}) AND status = 'SUCCEEDED'
            """
            cursor.execute(payments_query, user_ids)
            all_payments_raw = cursor.fetchall()
            
            # Фильтруем только первые 5 платежей для каждого пользователя
            all_payments = [p for p in all_payments_raw if p["rn"] <= 5]

            # Группируем данные по пользователям
            for sub in all_subscriptions:
                user_id = sub["user_id"]
                if user_id not in subscriptions_by_user:
                    subscriptions_by_user[user_id] = []
                subscriptions_by_user[user_id].append(dict(sub))
                
            for payment in all_payments:
                user_id = payment["user_id"]
                if user_id not in payments_by_user:
                    payments_by_user[user_id] = []
                payments_by_user[user_id].append(dict(payment))

        # Обрабатываем каждого пользователя
        users_with_subscriptions = []
        for user in users_data:
            user_id = user["user_id"]
            subscriptions = subscriptions_by_user.get(user_id, [])
            payments = payments_by_user.get(user_id, [])

            # Обрабатываем None значения
            user_dict = dict(user)
            user_dict["total_paid"] = user_dict["total_paid"] or 0
            user_dict["subscriptions_count"] = user_dict["subscriptions_count"] or 0
            user_dict["active_subscriptions"] = user_dict["active_subscriptions"] or 0
            user_dict["total_traffic_used"] = user_dict["total_traffic_used"] or 0
            user_dict["total_traffic_limit"] = user_dict["total_traffic_limit"] or 0

            # Преобразуем трафик в ГБ и добавляем процент
            if user_dict["total_traffic_limit"] and user_dict["total_traffic_limit"] > 0:
                traffic_used_gb = (user_dict["total_traffic_used"] or 0) / (1024**3)

                # Проверяем, если лимит в байтах (большое число), конвертируем в ГБ
                if user_dict["total_traffic_limit"] > 1000000000:  # Если больше 1 ГБ в байтах
                    traffic_limit_gb = user_dict["total_traffic_limit"] / (1024**3)
                else:
                    # Уже в ГБ
                    traffic_limit_gb = user_dict["total_traffic_limit"]

                traffic_percentage = (
                    (traffic_used_gb / traffic_limit_gb) * 100 if traffic_limit_gb > 0 else 0
                )
            else:
                traffic_used_gb = 0
                traffic_limit_gb = 0
                traffic_percentage = 0

            # Получаем информацию о последней подписке
            latest_subscription = subscriptions[0] if subscriptions else None
            latest_subscription_id = (
                latest_subscription["id"] if latest_subscription else "Нет подписки"
            )

            # Создаем упрощенную структуру для шаблона
            user_simple = {
                "user_id": user_dict["user_id"],
                "username": user_dict["username"],
                "first_name": user_dict["first_name"],
                "last_name": user_dict["last_name"],
                "vpn_email": user_dict["vpn_email"],
                "status": user_dict["status"],
                "user_created_at": user_dict["user_created_at"],
                "subscriptions_count": user_dict["subscriptions_count"],
                "active_subscriptions": user_dict["active_subscriptions"],
                "latest_expiry": user_dict["latest_expiry"],
                "latest_subscription_id": latest_subscription_id,
                "subscriptions": [dict(s) for s in subscriptions],  # Все подписки
                "traffic_used_gb": round(traffic_used_gb, 3),
                "traffic_limit_gb": traffic_limit_gb,
                "traffic_percentage": round(traffic_percentage, 1),
                "total_paid": user_dict["total_paid"],
                "payments_count": len(payments),
                "recent_payments": [dict(p) for p in payments[:3]],  # Последние 3 платежа
            }

            users_with_subscriptions.append(user_simple)

        conn.close()

        # Создаем ответ с заголовками для отключения кеширования
        response = make_response(
            render_template(
                "users_restored.html",
                users_data=users_with_subscriptions,
                current_time=datetime.now().strftime("%d.%m.%Y %H:%M:%S"),
                # Данные пагинации
                current_page=page,
                total_pages=total_pages,
                total_users=total_users,
                per_page=per_page,
                has_prev=page > 1,
                has_next=page < total_pages,
                prev_page=page - 1 if page > 1 else None,
                next_page=page + 1 if page < total_pages else None,
            )
        )

        # Отключаем кеширование для актуальных данных
        response.headers["Cache-Control"] = "no-cache, no-store, must-revalidate"
        response.headers["Pragma"] = "no-cache"
        response.headers["Expires"] = "0"

        # Сохраняем результат в кеш на 10 минут
        set_cached_data(cache_key, response)

        return response

    except Exception as e:
        flash(f"Ошибка при загрузке данных пользователей: {str(e)}", "error")
        return render_template("users_restored.html", users_data=[], error=str(e), 
                             current_page=1, total_pages=1, total_users=0, per_page=50,
                             has_prev=False, has_next=False, prev_page=None, next_page=None)


@app.route("/notifications")
def notifications():
    """Страница управления уведомлениями"""
    from config import NOTIFICATION_SETTINGS, DATABASE_FILE
    import sqlite3
    from datetime import datetime, timedelta

    # Получаем настройки из базы данных, если есть, иначе из конфигурации
    settings_dict = {}

    try:
        conn = sqlite3.connect(DATABASE_FILE)
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM notification_settings")
        db_settings = cursor.fetchall()
        conn.close()

        # Преобразуем настройки из базы данных
        import json

        for row in db_settings:
            plan_id = row[1]
            try:
                days_before = json.loads(row[3]) if row[3] else []
            except:
                days_before = []

            settings_dict[plan_id] = {
                "plan_name": row[2],
                "days_before": days_before,
                "send_time_start": row[4],
                "send_time_end": row[5],
                "check_interval_hours": row[6],
                "enabled": bool(row[7]),
            }
    except Exception as e:
        print(f"Ошибка чтения настроек из базы: {e}")

    # Дополняем недостающие настройки из конфигурации
    for plan_type, config in NOTIFICATION_SETTINGS.items():
        if plan_type not in settings_dict:
            settings_dict[plan_type] = {
                "plan_name": "Тестовые периоды" if plan_type == "trial" else "Оплаченные подписки",
                "days_before": config["days_before"],
                "send_time_start": config["send_time_start"],
                "send_time_end": config["send_time_end"],
                "check_interval_hours": config["check_interval_hours"],
                "enabled": True,
            }

    return render_template("notifications.html", settings=settings_dict)


@app.route("/notifications/settings/<plan_id>", methods=["GET", "POST"])
def notification_settings(plan_id):
    """Настройки уведомлений для конкретного тарифа"""
    from config import NOTIFICATION_SETTINGS, DATABASE_FILE
    import sqlite3

    if request.method == "POST":
        # Получаем данные из формы
        days_before = request.form.getlist("days_before")
        days_before = [int(d) for d in days_before if d.isdigit()]

        send_time_start = int(request.form.get("send_time_start", 9))
        send_time_end = int(request.form.get("send_time_end", 21))
        check_interval_hours = int(request.form.get("check_interval_hours", 1))
        enabled = "enabled" in request.form

        # Сохраняем настройки в SQLite базу
        try:
            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()

            # Удаляем старые настройки и вставляем новые
            cursor.execute("DELETE FROM notification_settings WHERE plan_id = ?", (plan_id,))
            cursor.execute(
                """
                INSERT INTO notification_settings 
                (plan_id, plan_name, days_before, send_time_start, send_time_end, check_interval_hours, enabled)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """,
                (
                    plan_id,
                    "Тестовые периоды" if plan_id == "trial" else "Оплаченные подписки",
                    str(days_before),
                    send_time_start,
                    send_time_end,
                    check_interval_hours,
                    enabled,
                ),
            )
            conn.commit()
            conn.close()

            flash(f"Настройки уведомлений для тарифа {plan_id} обновлены", "success")
        except Exception as e:
            flash(f"Ошибка сохранения настроек: {e}", "error")

        return redirect(url_for("notifications"))

    # Получаем настройки из базы данных или используем настройки по умолчанию
    setting = None
    try:
        conn = sqlite3.connect(DATABASE_FILE)
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM notification_settings WHERE plan_id = ?", (plan_id,))
        row = cursor.fetchone()

        if row:
            import json

            try:
                days_before = json.loads(row[3]) if row[3] else []
            except:
                days_before = []

            setting = {
                "plan_id": row[1],
                "plan_name": row[2],
                "days_before": days_before,
                "send_time_start": row[4],
                "send_time_end": row[5],
                "check_interval_hours": row[6],
                "enabled": bool(row[7]),
            }
        conn.close()
    except Exception as e:
        print(f"Ошибка получения настроек: {e}")

    # Если нет настроек в базе, используем настройки по умолчанию
    if not setting and plan_id in NOTIFICATION_SETTINGS:
        config = NOTIFICATION_SETTINGS[plan_id]
        setting = {
            "plan_id": plan_id,
            "plan_name": "Тестовые периоды" if plan_id == "trial" else "Оплаченные подписки",
            "days_before": config["days_before"],
            "send_time_start": config["send_time_start"],
            "send_time_end": config["send_time_end"],
            "check_interval_hours": config["check_interval_hours"],
            "enabled": True,
        }

    return render_template("notification_settings.html", plan_id=plan_id, setting=setting)


@app.route("/notifications/send-test", methods=["POST"])
def send_test_notification():
    """Отправка тестового уведомления"""
    user_id = request.form.get("user_id")

    if not user_id or not user_id.isdigit():
        flash("Неверный ID пользователя", "error")
        return redirect(url_for("notifications"))

    # Запускаем отправку тестового уведомления в фоне
    from threading import Thread

    def send_test():
        import asyncio

        run_async_safely(send_test_notification_async(int(user_id)))

    Thread(target=send_test).start()
    flash(f"Тестовое уведомление отправляется пользователю {user_id}", "info")
    return redirect(url_for("notifications"))


async def send_test_notification_async(user_id: int):
    """Асинхронная отправка тестового уведомления"""
    try:
        import sys
        import os

        sys.path.append(os.path.dirname(os.path.abspath(__file__)))

        from aiogram import Bot
        from config import BOT_TOKEN
        from datetime import datetime

        bot = Bot(token=BOT_TOKEN)

        test_message = (
            "🔔 <b>Тестовое уведомление</b>\n\n"
            "Это тестовое сообщение для проверки системы уведомлений.\n"
            f"Отправлено: {datetime.now().strftime('%d.%m.%Y %H:%M')}"
        )

        try:
            await bot.send_message(chat_id=user_id, text=test_message, parse_mode="HTML")

            # Логируем отправку в SQLite базу бота
            from config import DATABASE_FILE
            import sqlite3

            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()
            cursor.execute(
                """
                INSERT INTO notification_log 
                (user_id, subscription_id, notification_type, message_text, success, sent_at)
                VALUES (?, ?, ?, ?, ?, ?)
            """,
                (user_id, 0, "test", test_message, True, datetime.now().isoformat()),
            )
            conn.commit()
            conn.close()
        finally:
            # Всегда закрываем сессию бота
            await bot.session.close()

    except Exception as e:
        # Логируем ошибку в SQLite базу бота
        from config import DATABASE_FILE
        import sqlite3

        try:
            from datetime import datetime
            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()
            cursor.execute(
                """
                INSERT INTO notification_log 
                (user_id, subscription_id, notification_type, message_text, success, error_message, sent_at)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """,
                (
                    user_id,
                    0,
                    "test",
                    "Тестовое уведомление",
                    False,
                    str(e),
                    datetime.now().isoformat(),
                ),
            )
            conn.commit()
            conn.close()
        except:
            pass


@app.route("/scheduler")
def scheduler():
    """Страница мониторинга планировщика уведомлений"""
    import sqlite3
    from config import DATABASE_FILE
    from datetime import datetime, timedelta

    # Получаем состояние планировщика
    scheduler_status = get_scheduler_status()

    # Получаем предстоящие уведомления
    upcoming_notifications = get_upcoming_notifications()

    # Получаем статистику уведомлений за последние 24 часа
    conn = sqlite3.connect(DATABASE_FILE)
    cursor = conn.cursor()

    # Статистика за последние 24 часа
    yesterday = (datetime.now() - timedelta(days=1)).isoformat()
    cursor.execute(
        """
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN success = 1 THEN 1 ELSE 0 END) as successful,
            SUM(CASE WHEN success = 0 THEN 1 ELSE 0 END) as failed,
            notification_type
        FROM notification_log 
        WHERE sent_at > ?
        GROUP BY notification_type
    """,
        (yesterday,),
    )
    stats_24h = cursor.fetchall()

    # Последние логи
    cursor.execute(
        """
        SELECT user_id, notification_type, days_before_expiry, 
               success, error_message, sent_at 
        FROM notification_log 
        ORDER BY sent_at DESC 
        LIMIT 20
    """
    )
    recent_logs = cursor.fetchall()

    conn.close()

    return render_template(
        "scheduler.html",
        scheduler_status=scheduler_status,
        upcoming_notifications=upcoming_notifications,
        stats_24h=stats_24h,
        recent_logs=recent_logs,
    )


def get_scheduler_status():
    """Получить статус планировщика задач"""
    try:
        from datetime import datetime, timedelta
        import os
        import sqlite3
        from config import DATABASE_FILE

        status = {"is_running": False, "active_jobs": [], "last_check": None, "uptime": None}

        # Проверяем наличие недавних логов планировщика в базе данных
        try:
            conn = sqlite3.connect(DATABASE_FILE)
            cursor = conn.cursor()

            # Проверяем последние логи уведомлений (если есть активность за последние 15 минут)
            recent_time = (datetime.now() - timedelta(minutes=15)).isoformat()
            cursor.execute(
                """
                SELECT COUNT(*) FROM notification_log 
                WHERE sent_at > ? AND notification_type = 'scheduler_check'
            """,
                (recent_time,),
            )

            recent_scheduler_activity = cursor.fetchone()[0]
            conn.close()

            if recent_scheduler_activity > 0:
                status["is_running"] = True

        except Exception as db_error:
            print(f"Ошибка проверки базы данных: {db_error}")

        # Альтернативная проверка - ищем процесс бота
        if not status["is_running"]:
            try:
                # Проверяем наличие файла блокировки или процесса
                import subprocess

                result = subprocess.run(["pgrep", "-f", "main.py"], capture_output=True, text=True)
                if result.returncode == 0 and result.stdout.strip():
                    status["is_running"] = True
            except Exception as proc_error:
                print(f"Ошибка проверки процесса: {proc_error}")

        # Если планировщик работает, определяем активные задачи
        if status["is_running"]:
            status["active_jobs"] = [
                "light_sync",
                "scheduler_health_check",
                "trial_hourly_check",
                "traffic_check",
                "regular_daily_check",
            ]

        status["last_check"] = datetime.now()
        return status

    except Exception as e:
        from datetime import datetime

        return {
            "is_running": False,
            "active_jobs": [],
            "last_check": datetime.now(),
            "error": str(e),
        }


def get_upcoming_notifications():
    """Получить предстоящие уведомления"""
    import sqlite3
    import json
    from config import DATABASE_FILE
    from datetime import datetime, timedelta

    upcoming = []

    try:
        conn = sqlite3.connect(DATABASE_FILE)
        cursor = conn.cursor()

        # Получаем все активные подписки и настройки уведомлений для них
        # Используем правильную группировку: trial для тестовых, paid для остальных
        cursor.execute(
            """
            SELECT u.id, u.username, s.id, s.expires_at, s.plan_id,
                   ns.days_before, ns.send_time_start, ns.send_time_end, ns.check_interval_hours, ns.plan_name
            FROM users u
            JOIN subscriptions s ON u.id = s.user_id
            LEFT JOIN notification_settings ns ON 
                CASE 
                    WHEN s.plan_id = 'trial' THEN ns.plan_id = 'trial'
                    ELSE ns.plan_id = 'paid'
                END
            WHERE datetime(s.expires_at) > datetime('now') 
            AND UPPER(s.status) = 'ACTIVE'
            AND (ns.enabled = 1 OR ns.enabled IS NULL)
            ORDER BY s.expires_at ASC
            LIMIT 50
        """
        )

        subscriptions = cursor.fetchall()

        for sub in subscriptions:
            (
                user_id,
                username,
                sub_id,
                expires_at,
                plan_type,
                days_before_json,
                start_time,
                end_time,
                interval_hours,
                plan_name,
            ) = sub
            try:
                # Обрабатываем разные форматы даты
                if "T" in expires_at:
                    expires_date = datetime.fromisoformat(expires_at.replace("Z", "+00:00"))
                elif " " in expires_at and ":" in expires_at:
                    # Формат с датой и временем: "2025-06-14 11:46:51"
                    expires_date = datetime.strptime(expires_at, "%Y-%m-%d %H:%M:%S")
                else:
                    # Формат только с датой: "2025-06-14"
                    expires_date = datetime.strptime(expires_at, "%Y-%m-%d")

                days_until_expiry = (expires_date - datetime.now()).days

                # Получаем дни уведомлений из настроек или используем дефолтные
                if days_before_json:
                    try:
                        notification_days = json.loads(days_before_json)
                    except:
                        notification_days = [1, 3] if plan_type != "trial" else [0, 1, 2, 3]
                else:
                    notification_days = [1, 3] if plan_type != "trial" else [0, 1, 2, 3]

                # Определяем частоту уведомлений
                if plan_type == "trial":
                    frequency = f"каждый час с {start_time or 9}:00 до {end_time or 21}:00"
                else:
                    frequency = f"ежедневно в {start_time or 9}:00 МСК"

                # Показываем предстоящие уведомления для каждого настроенного дня
                for days_before in notification_days:
                    if days_until_expiry >= days_before:
                        # Определяем дату следующего уведомления
                        target_date = expires_date - timedelta(days=days_before)

                        # Для тестовых подписок - проверяем близость к дате уведомления
                        if plan_type == "trial":
                            if days_until_expiry == days_before:
                                next_notification = "В ближайший час"
                            else:
                                next_notification = (
                                    f"{target_date.strftime('%d.%m')} с {start_time or 9}:00"
                                )
                        else:
                            next_notification = (
                                f"{target_date.strftime('%d.%m')} в {start_time or 9}:00"
                            )

                        notification_text = (
                            f"За {days_before} дн." if days_before > 0 else "В день истечения"
                        )

                        upcoming.append(
                            {
                                "user_id": user_id,
                                "username": username or f"ID{user_id}",
                                "subscription_id": sub_id,
                                "expires_at": expires_date.strftime("%d.%m.%Y %H:%M"),
                                "days_until_expiry": days_until_expiry,
                                "plan_type": plan_type,
                                "plan_name": plan_name or plan_type.title(),
                                "notification_type": notification_text,
                                "next_notification": next_notification,
                                "frequency": frequency,
                                "days_before": days_before,
                            }
                        )
            except Exception as date_error:
                print(f"Ошибка обработки даты для подписки {sub_id}: {date_error}")
                continue

        conn.close()

        # Сортируем по дате ближайшего уведомления
        upcoming.sort(key=lambda x: (x["days_until_expiry"], x["days_before"]))

    except Exception as e:
        print(f"Ошибка получения предстоящих уведомлений: {e}")

    return upcoming


@app.route("/yookassa-webhook", methods=["POST"])
def yookassa_webhook():
    """
    Webhook endpoint для обработки уведомлений от YooKassa
    Реализует правильную обработку HTTP-кодов согласно документации YooKassa
    """
    import sqlite3
    import os
    import traceback
    import asyncio
    from datetime import datetime

    db_path = os.path.join("database", "vpnbot.db")

    try:
        # Получаем данные от YooKassa
        notification_data = request.get_json()

        # Логируем входящий webhook
        timestamp = datetime.now().isoformat()
        print(f"[{timestamp}] YooKassa Webhook получен: {notification_data}")

        if not notification_data:
            print(f"[{timestamp}] ОШИБКА: Пустые данные от YooKassa")
            # Согласно документации YooKassa - возвращаем 400 для некорректного запроса
            return jsonify({"error": "Invalid request: no data received"}), 400

        # Извлекаем данные о платеже
        event = notification_data.get("event")
        payment_data = notification_data.get("object", {})
        payment_id = payment_data.get("id")
        status = payment_data.get("status")
        metadata = payment_data.get("metadata", {})
        amount_data = payment_data.get("amount", {})

        print(f"[{timestamp}] Событие: {event}")
        print(f"[{timestamp}] Статус платежа: {status}")
        print(f"[{timestamp}] ID платежа: {payment_id}")
        print(f"[{timestamp}] Metadata: {metadata}")
        print(f"[{timestamp}] Сумма: {amount_data}")

        if not payment_id:
            print(f"[{timestamp}] ОШИБКА: Отсутствует ID платежа")
            return jsonify({"error": "Payment ID required"}), 400

        # Обрабатываем только события успешных платежей
        if event == "payment.succeeded" and status == "succeeded":
            print(f"[{timestamp}] Обработка успешного платежа {payment_id}")

            # Проверяем статус платежа в базе данных
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()

            cursor.execute(
                "SELECT id, user_id, subscription_id, plan_id, status FROM payments WHERE payment_id = ?",
                (payment_id,),
            )
            payment_record = cursor.fetchone()

            if not payment_record:
                print(f"[{timestamp}] ОШИБКА: Платеж {payment_id} не найден в базе данных")
                conn.close()
                # Возвращаем 200 чтобы YooKassa не повторяла запрос
                return (
                    jsonify(
                        {
                            "status": "payment_not_found",
                            "message": f"Payment {payment_id} not found in database",
                        }
                    ),
                    200,
                )

            payment_db_id, user_id, subscription_id, plan_id, current_status = payment_record

            # Проверяем, не был ли платеж уже обработан
            if current_status == "SUCCEEDED":
                print(f"[{timestamp}] ИНФОРМАЦИЯ: Платеж {payment_id} уже был обработан ранее")
                conn.close()
                return (
                    jsonify(
                        {
                            "status": "already_processed",
                            "message": f"Payment {payment_id} already processed",
                        }
                    ),
                    200,
                )

            print(
                f"[{timestamp}] Найден платеж в БД: user_id={user_id}, subscription_id={subscription_id}, plan_id={plan_id}"
            )

            # Обновляем статус платежа на SUCCEEDED
            cursor.execute(
                "UPDATE payments SET status = 'SUCCEEDED', completed_at = ? WHERE payment_id = ?",
                (datetime.now().isoformat(), payment_id),
            )
            conn.commit()
            conn.close()

            print(f"[{timestamp}] Статус платежа {payment_id} обновлен на SUCCEEDED")

            # Автоматически обрабатываем платеж
            import asyncio

            try:
                success = run_async_safely(
                    process_successful_payment(
                        user_id, subscription_id, payment_id, plan_id, amount_data.get("value", "0")
                    )
                )

                if success:
                    print(f"[{timestamp}] ✅ Платеж {payment_id} успешно обработан")
                    return (
                        jsonify(
                            {
                                "status": "success",
                                "message": f"Payment {payment_id} processed successfully",
                            }
                        ),
                        200,
                    )
                else:
                    print(f"[{timestamp}] ❌ Ошибка при обработке платежа {payment_id}")
                    # Возвращаем 500 чтобы YooKassa повторила запрос
                    return jsonify({"error": "Payment processing failed"}), 500

            except Exception as processing_error:
                print(f"[{timestamp}] ОШИБКА обработки платежа: {processing_error}")
                print(f"[{timestamp}] Трассировка: {traceback.format_exc()}")
                # Возвращаем 500 для повторной попытки от YooKassa
                return jsonify({"error": "Payment processing exception"}), 500

        elif event == "payment.canceled":
            print(f"[{timestamp}] Платеж {payment_id} отменен")
            # Обновляем статус в базе данных
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()
            cursor.execute(
                "UPDATE payments SET status = 'CANCELED', completed_at = ? WHERE payment_id = ?",
                (datetime.now().isoformat(), payment_id),
            )

            # Получаем данные пользователя для отправки уведомления
            cursor.execute(
                """
                SELECT p.user_id, p.amount, p.subscription_id, p.plan_id 
                FROM payments p 
                WHERE p.payment_id = ?
            """,
                (payment_id,),
            )
            payment_data = cursor.fetchone()

            conn.commit()
            conn.close()

            if payment_data:
                user_id, amount, subscription_id, plan_id = payment_data
                # Отправляем уведомление о неуспешном платеже в отдельном потоке
                import threading

                def send_notification():
                    try:
                        import asyncio

                        run_async_safely(
                            send_payment_failure_notification(
                                user_id,
                                payment_id,
                                "canceled",
                                "Платеж был отменен",
                                amount,
                                plan_id,
                            )
                        )
                    except Exception as e:
                        print(f"Ошибка отправки уведомления о неуспешном платеже: {e}")

                thread = threading.Thread(target=send_notification)
                thread.daemon = True
                thread.start()

            return jsonify({"status": "canceled_processed"}), 200

        elif event == "payment.failed":
            print(f"[{timestamp}] Платеж {payment_id} неуспешен")

            # Извлекаем причину неуспеха из cancellation_details
            cancellation_reason = "Неизвестная ошибка"
            if "cancellation_details" in payment_data:
                cancellation_details = payment_data["cancellation_details"]
                if cancellation_details.get("reason"):
                    cancellation_reason = get_payment_error_description(
                        cancellation_details["reason"]
                    )

            # Обновляем статус в базе данных
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()
            cursor.execute(
                "UPDATE payments SET status = 'FAILED', completed_at = ? WHERE payment_id = ?",
                (datetime.now().isoformat(), payment_id),
            )

            # Получаем данные пользователя для отправки уведомления
            cursor.execute(
                """
                SELECT p.user_id, p.amount, p.subscription_id, p.plan_id 
                FROM payments p 
                WHERE p.payment_id = ?
            """,
                (payment_id,),
            )
            payment_data = cursor.fetchone()

            conn.commit()
            conn.close()

            if payment_data:
                user_id, amount, subscription_id, plan_id = payment_data
                # Отправляем уведомление о неуспешном платеже в отдельном потоке
                import threading

                def send_notification():
                    try:
                        import asyncio

                        run_async_safely(
                            send_payment_failure_notification(
                                user_id, payment_id, "failed", cancellation_reason, amount, plan_id
                            )
                        )
                    except Exception as e:
                        print(f"Ошибка отправки уведомления о неуспешном платеже: {e}")

                thread = threading.Thread(target=send_notification)
                thread.daemon = True
                thread.start()

            return jsonify({"status": "failed_processed"}), 200

        elif event == "payment.waiting_for_capture":
            print(f"[{timestamp}] Платеж {payment_id} ожидает подтверждения")
            return jsonify({"status": "waiting_processed"}), 200

        else:
            print(f"[{timestamp}] Неизвестное событие или статус: {event}/{status}")
            # Возвращаем 200 для неизвестных событий чтобы не засорять логи YooKassa
            return jsonify({"status": "event_ignored", "event": event, "status": status}), 200

    except Exception as e:
        error_trace = traceback.format_exc()
        timestamp = datetime.now().isoformat()
        print(f"[{timestamp}] КРИТИЧЕСКАЯ ОШИБКА webhook YooKassa: {e}")
        print(f"[{timestamp}] Полная трассировка: {error_trace}")

        # Согласно документации YooKassa - возвращаем 500 для серверных ошибок
        # YooKassa будет повторять запрос при получении 5xx кодов
        return jsonify({"error": "Internal server error", "message": str(e)}), 500


async def process_successful_payment(
    user_id: int, subscription_id: int, payment_id: str, plan_id: str, amount: str
):
    """Обработка успешного платежа и автоматическое обновление подписки"""
    try:
        # Импортируем нужные модули
        import sys
        import os
        import sqlite3
        from datetime import datetime, timedelta
        from pytz import timezone

        # Добавляем путь к модулям бота
        sys.path.append(os.path.join(os.path.dirname(__file__)))

        from bot.database.operations import update_subscription
        from config import SUBSCRIPTION_PLANS
        from bot.utils.date_validation import validate_subscription_dates, log_date_calculation

        print(
            f"Обработка платежа {payment_id} для пользователя {user_id}, подписка {subscription_id}"
        )

        # Получаем информацию о тарифе
        plan_info = SUBSCRIPTION_PLANS.get(plan_id)
        if not plan_info:
            print(f"Тариф {plan_id} не найден в конфигурации")
            return

        # Получаем текущую подписку для накопления времени и лимитов
        db_path = os.path.join("database", "vpnbot.db")
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()

        cursor.execute(
            "SELECT expires_at, traffic_limit FROM subscriptions WHERE id = ?", (subscription_id,)
        )
        current_subscription = cursor.fetchone()
        conn.close()

        # Используем московское время (UTC+3) без timezone для consistency
        from pytz import timezone

        moscow_tz = timezone("Europe/Moscow")
        now_moscow = datetime.now(moscow_tz).replace(tzinfo=None)  # Убираем timezone info для простоты

        if current_subscription:
            current_expiry_str, current_traffic_limit = current_subscription

            # Парсим текущую дату окончания
            if current_expiry_str:
                try:
                    # Пробуем разные форматы даты и приводим к naive datetime
                    if "T" in current_expiry_str:
                        current_expiry = datetime.fromisoformat(
                            current_expiry_str.replace("Z", "").replace("+00:00", "")
                        )
                    else:
                        current_expiry = datetime.strptime(current_expiry_str, "%Y-%m-%d %H:%M:%S")
                    
                    # Убираем timezone info если есть
                    if current_expiry.tzinfo is not None:
                        current_expiry = current_expiry.replace(tzinfo=None)

                    print(f"DEBUG: Текущая дата истечения: {current_expiry}")
                    print(f"DEBUG: Текущая дата (МСК): {now_moscow}")

                    # ИСПРАВЛЕНИЕ: Для продления подписки ВСЕГДА добавляем дни к текущей дате
                    # вместо проверки активности подписки
                    new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])
                    
                    print(f"DEBUG: Новая дата истечения: {new_expiry_date}")
                    
                    # ВАЛИДАЦИЯ: Проверяем корректность расчета даты
                    is_valid = validate_subscription_dates(
                        payment_date=now_moscow,
                        plan_duration_days=plan_info["duration_days"],
                        calculated_expiry=new_expiry_date,
                        user_id=user_id,
                        plan_id=plan_id
                    )
                    
                    if not is_valid:
                        print(f"КРИТИЧЕСКАЯ ОШИБКА: Неверный расчет даты для пользователя {user_id}")
                        return False
                    
                    # АУДИТ: Логируем расчет для отслеживания
                    log_date_calculation(
                        user_id=user_id,
                        plan_id=plan_id,
                        payment_date=now_moscow,
                        calculated_expiry=new_expiry_date,
                        is_extension=bool(current_subscription)
                    )
                    
                except Exception as e:
                    print(f"DEBUG: Ошибка парсинга даты '{current_expiry_str}': {e}")
                    # Если ошибка парсинга, используем текущую дату
                    new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])
            else:
                new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])

            # Накапливаем лимит трафика
            new_traffic_limit = (current_traffic_limit or 0) + plan_info["traffic_gb"]
        else:
            # Новая подписка
            new_expiry_date = now_moscow + timedelta(days=plan_info["duration_days"])
            new_traffic_limit = plan_info["traffic_gb"]

        print(
            f"Обновляем подписку {subscription_id}: тариф={plan_id}, до={new_expiry_date.strftime('%d.%m.%Y')}, лимит={new_traffic_limit}ГБ (накоплено)"
        )

        # Обновляем подписку в базе данных с накопленными значениями
        try:
            # Подключаемся к базе данных
            db_path = os.path.join("database", "vpnbot.db")
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()

            # Выполняем обновление подписки с накопленными значениями и сбросом трафика
            cursor.execute(
                """
                UPDATE subscriptions 
                SET plan_id = ?, 
                    expires_at = ?, 
                    traffic_limit = ?, 
                    traffic_used = 0,
                    status = 'ACTIVE'
                WHERE id = ?
            """,
                (plan_id, new_expiry_date.isoformat(), new_traffic_limit, subscription_id),
            )

            # Проверяем, что обновление прошло успешно
            if cursor.rowcount > 0:
                print(f"Обновлена {cursor.rowcount} подписка, трафик сброшен на 0")
            else:
                print(f"ВНИМАНИЕ: Подписка {subscription_id} не найдена для обновления")

            conn.commit()
            conn.close()
            print(f"Подписка {subscription_id} успешно обновлена в базе данных")

        except Exception as e:
            print(f"Ошибка обновления подписки {subscription_id} в базе данных: {e}")
            return

        # Теперь обновляем данные в панели 3X-UI
        try:
            # Получаем данные подписки для поиска vpn_user_id
            db_path = os.path.join("database", "vpnbot.db")
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()

            cursor.execute(
                "SELECT vpn_user_id, vpn_email FROM subscriptions WHERE id = ?", (subscription_id,)
            )
            subscription_data = cursor.fetchone()
            conn.close()

            if subscription_data:
                vpn_user_id, vpn_email = subscription_data
                print(f"Найдены данные клиента в панели: UUID={vpn_user_id}, Email={vpn_email}")

                # Импортируем функцию обновления панели
                from bot.services.panel_sync import update_client_in_panel

                # Конвертируем накопленный лимит ГБ в байты для панели
                traffic_limit_bytes = new_traffic_limit * 1024 * 1024 * 1024

                # Конвертируем дату в миллисекунды для панели
                expiry_time_ms = int(new_expiry_date.timestamp() * 1000)

                # Обновляем клиента в панели (сбрасываем трафик при продлении)
                panel_update_success = await update_client_in_panel(
                    client_id=vpn_user_id or vpn_email,  # Используем UUID или email для поиска
                    traffic_limit=traffic_limit_bytes,
                    expiry_time=expiry_time_ms,
                    reset_traffic=True,  # Сбрасываем использованный трафик при продлении
                    user_id=user_id,  # ID пользователя в Telegram
                    subscription_id=subscription_id,  # ID подписки в базе данных
                )

                if panel_update_success:
                    print(
                        f"Клиент {vpn_email} успешно обновлен в панели: {plan_info['traffic_gb']}ГБ до {new_expiry_date.strftime('%d.%m.%Y')}"
                    )
                else:
                    print(f"Ошибка обновления клиента {vpn_email} в панели")
            else:
                print(f"Не найдены данные клиента в панели для подписки {subscription_id}")

        except Exception as e:
            print(f"Ошибка обновления клиента в панели: {e}")

        # Отправляем уведомление пользователю
        await send_payment_success_notification(user_id, payment_id, plan_info["name"], amount)

        print(f"Платеж {payment_id} успешно обработан, подписка {subscription_id} обновлена")
        return True

    except Exception as e:
        print(f"Ошибка обработки платежа {payment_id}: {e}")
        return False


async def send_payment_success_notification(
    user_id: int, payment_id: str, subscription_type: str, amount: str
):
    """Отправка уведомления об успешном платеже пользователю в Telegram"""
    try:
        import os
        from aiogram import Bot
        from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton

        # Получаем токен бота
        bot_token = os.environ.get("BOT_TOKEN")
        if not bot_token:
            print("Отсутствует BOT_TOKEN для отправки уведомления")
            return

        bot = Bot(token=bot_token)

        # Получаем данные подписки пользователя
        try:
            import sqlite3
            import os

            db_path = os.path.join("database", "vpnbot.db")
            conn = sqlite3.connect(db_path)
            cursor = conn.cursor()

            # Находим подписку пользователя
            cursor.execute(
                """
                SELECT id FROM subscriptions 
                WHERE user_id = ? AND status = 'ACTIVE'
                ORDER BY created_at DESC 
                LIMIT 1
            """,
                (user_id,),
            )

            subscription_result = cursor.fetchone()
            subscription_id = subscription_result[0] if subscription_result else None
            conn.close()

        except Exception as db_error:
            print(f"Ошибка получения подписки: {db_error}")
            subscription_id = None

        # Формируем сообщение
        message = f"""
🎉 Платеж успешно обработан!

💳 ID платежа: {payment_id}
💰 Сумма: {amount} ₽
📦 Тариф: {subscription_type}

✅ Ваша подписка автоматически обновлена!
• Новый срок действия установлен
• Лимит трафика сброшен
• Счетчик использования обнулен

🔑 Теперь вы можете получить ваш VPN ключ:
"""

        # Создаем инлайн-клавиатуру с кнопкой получения ключа
        keyboard = None
        if subscription_id:
            keyboard = InlineKeyboardMarkup(
                inline_keyboard=[
                    [
                        InlineKeyboardButton(
                            text="🔑 Получить VPN Ключ",
                            callback_data=f"get_vpn_key_{subscription_id}",
                        )
                    ]
                ]
            )

        # Отправляем сообщение пользователю
        try:
            await bot.send_message(
                chat_id=user_id, text=message, reply_markup=keyboard, parse_mode="HTML"
            )
            print(f"Уведомление об успешном платеже отправлено пользователю {user_id}")
        finally:
            # Закрываем сессию бота
            await bot.session.close()

    except Exception as e:
        print(f"Ошибка отправки уведомления пользователю {user_id}: {e}")


async def send_payment_failure_notification(
    user_id: int, payment_id: str, failure_type: str, error_reason: str, amount: str, plan_id: str
):
    """Отправка уведомления о неуспешном платеже пользователю в Telegram"""
    try:
        import os
        from aiogram import Bot
        from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton

        # Получаем токен бота
        bot_token = os.environ.get("BOT_TOKEN")
        if not bot_token:
            print("Отсутствует BOT_TOKEN для отправки уведомления")
            return

        bot = Bot(token=bot_token)

        # Определяем тип ошибки для пользователя
        if failure_type == "canceled":
            status_emoji = "❌"
            status_text = "отменен"
        elif failure_type == "failed":
            status_emoji = "⚠️"
            status_text = "не прошел"
        else:
            status_emoji = "❌"
            status_text = "неуспешен"

        # Получаем название тарифа
        from config import SUBSCRIPTION_PLANS

        plan_info = SUBSCRIPTION_PLANS.get(plan_id, {})
        plan_name = plan_info.get("name", plan_id)

        message = f"""{status_emoji} Платеж {status_text}

💳 ID платежа: {payment_id}
📦 Тариф: {plan_name}
💰 Сумма: {amount} ₽

🔍 Причина: {error_reason}

💡 Вы можете повторить попытку оплаты позже или выбрать другой способ платежа.

🔄 Для повторного платежа используйте в Главном меню кнопку "Продлить Подписку" """

        # Создаем инлайн-клавиатуру с кнопкой повтора оплаты
        keyboard = InlineKeyboardMarkup(
            inline_keyboard=[
                [InlineKeyboardButton(text="🔄 Повторить оплату", callback_data="retry_payment")]
            ]
        )

        # Отправляем сообщение пользователю
        try:
            await bot.send_message(
                chat_id=user_id, text=message, reply_markup=keyboard, parse_mode="HTML"
            )
            print(f"Уведомление о неуспешном платеже отправлено пользователю {user_id}")
        finally:
            # Закрываем сессию бота
            await bot.session.close()

    except Exception as e:
        print(f"Ошибка отправки уведомления о неуспешном платеже пользователю {user_id}: {e}")


def get_payment_error_description(error_code: str) -> str:
    """Возвращает понятное описание ошибки платежа по коду YooKassa"""
    error_descriptions = {
        # Ошибки карт
        "card_expired": "Истек срок действия карты",
        "insufficient_funds": "Недостаточно средств на карте",
        "invalid_card_number": "Неверный номер карты",
        "invalid_csc": "Неверный CVV/CVC код",
        "issuer_unavailable": "Банк-эмитент недоступен",
        "payment_method_limit_exceeded": "Превышен лимит платежей",
        "payment_method_restricted": "Платежи с данной карты ограничены",
        "rejected_by_issuer": "Банк отклонил операцию",
        "lost_card": "Карта заблокирована (утеряна)",
        "stolen_card": "Карта заблокирована (украдена)",
        "expired_card": "Истек срок действия карты",
        # Ошибки 3DS
        "three_d_secure_failed": "Ошибка подтверждения 3D-Secure",
        "three_d_secure_not_supported": "3D-Secure не поддерживается",
        # Технические ошибки
        "processing_error": "Техническая ошибка обработки",
        "general_decline": "Платеж отклонен банком",
        "invalid_processing_request": "Некорректный запрос",
        # YooMoney ошибки
        "account_blocked": "Кошелек заблокирован",
        "account_closed": "Кошелек закрыт",
        "fraud_suspected": "Подозрение на мошенничество",
        # Временные ошибки
        "expired_on_capture": "Превышено время подтверждения платежа",
        "expired_on_confirmation": "Превышено время ожидания подтверждения",
        "payment_timeout": "Превышено время ожидания",
        "timeout": "Истекло время ожидания ответа",
        # Sberbank ошибки
        "sberbank_online_rejected": "Sberbank Online отклонил платеж",
        # Общие ошибки
        "canceled_by_merchant": "Платеж отменен продавцом",
        "permission_revoked": "Отозвано разрешение на списание",
        "invalid_data": "Неверные данные платежа",
        "merchant_account_error": "Ошибка аккаунта продавца",
        "payment_method_unavailable": "Способ оплаты временно недоступен",
        "unsupported_mobile_operator": "Мобильный оператор не поддерживается",
        # Ошибки интернет-банкинга
        "alfabank_online_rejected": "Альфа-Банк отклонил платеж",
        "payment_method_not_available": "Способ оплаты недоступен",
        # Ошибки валидации
        "invalid_request": "Некорректный запрос",
        "forbidden": "Операция запрещена",
        "not_supported": "Операция не поддерживается",
    }

    return error_descriptions.get(error_code, f"Ошибка платежной системы ({error_code})")


@app.route("/api/refresh-users", methods=["POST"])
def api_refresh_users():
    """API эндпоинт для принудительного обновления данных пользователей с синхронизацией трафика"""
    try:
        print("Принудительное обновление: синхронизация трафика...")
        from traffic_sync_web import sync_traffic_for_web
        
        # Принудительная синхронизация трафика
        run_async_safely(sync_traffic_for_web())
        
        # Очищаем кеш пользователей для принудительного обновления
        cache_key = "users_data"
        with _cache_lock:
            if cache_key in _cache:
                del _cache[cache_key]
                print("Кеш пользователей очищен")
        
        return jsonify({
            "status": "success", 
            "message": "Данные обновлены, синхронизация трафика выполнена"
        })
        
    except Exception as e:
        print(f"Ошибка при обновлении данных: {e}")
        return jsonify({"status": "error", "message": str(e)}), 500


@app.route("/api/sync-traffic", methods=["POST"])
def api_sync_traffic():
    """API эндпоинт для синхронизации трафика с панелью 3X-UI"""
    try:
        from traffic_sync_web import sync_traffic_for_web
        import asyncio

        # Запускаем синхронизацию трафика
        result = run_async_safely(sync_traffic_for_web())

        return jsonify(
            {"success": True, "message": "Синхронизация трафика завершена", "data": result}
        )

    except Exception as e:
        return jsonify({"success": False, "error": f"Ошибка синхронизации: {str(e)}"}), 500


@app.route("/api/get-panel-id", methods=["POST"])
def api_get_panel_id():
    """API эндпоинт для получения ID клиента из панели 3X-UI"""
    try:
        import asyncio
        import sys
        import os

        # Добавляем путь к модулям бота
        sys.path.append(os.path.join(os.path.dirname(__file__), "bot"))

        from bot.services.working_panel_api import WorkingPanelAPI
        from config_keys import get_panel_credentials

        data = request.json
        if not data:
            return jsonify({"success": False, "error": "Не получены данные запроса"})
        
        vpn_email = data.get("vpn_email")

        if not vpn_email:
            return jsonify({"success": False, "error": "VPN email не указан"})

        # Получаем данные панели
        panel_url, panel_username, panel_password = get_panel_credentials()

        async def get_client_id():
            panel_api = WorkingPanelAPI(panel_url, panel_username, panel_password)
            try:
                await panel_api.login()
                clients = await panel_api.get_clients()

                if clients:
                    for client in clients:
                        if client.get("email") == vpn_email:
                            return client.get("id")
                return None
            finally:
                await panel_api.close()

        # Запускаем асинхронную функцию
        try:
            panel_id = run_async_safely(get_client_id())
            return jsonify({"success": True, "panel_id": panel_id})
        except Exception as e:
            return jsonify({"success": False, "error": str(e)})

    except Exception as e:
        return jsonify({"success": False, "error": str(e)}), 500


@app.route("/api/delete-user/<int:user_id>", methods=["DELETE"])
def api_delete_user(user_id):
    """API эндпоинт для полного удаления пользователя из базы данных и панели"""
    try:
        import asyncio
        import sys
        import os
        from config import DATABASE_FILE
        import sqlite3

        # Добавляем путь к модулям бота
        sys.path.append(os.path.join(os.path.dirname(__file__), "bot"))

        from bot.services.working_panel_api import WorkingPanelAPI
        from config_keys import get_panel_credentials

        # Получаем данные пользователя перед удалением
        conn = sqlite3.connect(DATABASE_FILE)
        cursor = conn.cursor()

        cursor.execute("SELECT vpn_email FROM users WHERE id = ?", (user_id,))
        user_result = cursor.fetchone()

        if not user_result:
            conn.close()
            return jsonify({"success": False, "error": "Пользователь не найден"})

        vpn_email = user_result[0]

        # Удаляем из панели 3X-UI если есть VPN email
        if vpn_email:
            try:
                panel_url, panel_username, panel_password = get_panel_credentials()

                async def delete_from_panel():
                    from bot.services.strict_panel_api import delete_client as strict_delete_client

                    panel_api = WorkingPanelAPI(panel_url, panel_username, panel_password)
                    try:
                        await panel_api.login()
                        clients = await panel_api.get_clients()

                        if clients:
                            for client in clients:
                                if client.get("email") == vpn_email:
                                    # Удаляем клиента из inbound
                                    inbound_id = client.get("inbound_id")
                                    client_id = client.get("id")
                                    if inbound_id and client_id:
                                        await strict_delete_client(inbound_id, client_id)
                                    break
                    finally:
                        await panel_api.close()

                try:
                    run_async_safely(delete_from_panel())
                except Exception as delete_error:
                    print(f"Ошибка удаления из панели: {delete_error}")
            except Exception as panel_error:
                print(f"Ошибка удаления из панели: {panel_error}")

        # Удаляем все связанные данные из базы
        cursor.execute("DELETE FROM payments WHERE user_id = ?", (user_id,))
        cursor.execute("DELETE FROM subscriptions WHERE user_id = ?", (user_id,))
        cursor.execute("DELETE FROM users WHERE id = ?", (user_id,))

        conn.commit()
        conn.close()

        return jsonify(
            {"success": True, "message": f"Пользователь {user_id} и все связанные данные удалены"}
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)}), 500


if __name__ == "__main__":
    app.run(host="0.0.0.0", port=5000, debug=True)
