import logging
from aiogram import Bot, types
from aiogram.utils.keyboard import InlineKeyboardBuilder

from bot.database.models import User, Subscription

logger = logging.getLogger(__name__)


async def send_expiry_notification(bot: Bot, user: User, subscription: Subscription, days_left: int) -> None:
    """Send notification about subscription expiry"""
    from config import SUBSCRIPTION_PLANS
    from bot.utils.formatting import format_bytes_to_gb
    
    # Create keyboard for renewal
    keyboard = InlineKeyboardBuilder()
    keyboard.button(
        text="💰 Продлить подписку",
        callback_data=f"extend_subscription"
    )
    
    # Получаем информацию о тарифе
    plan_id = subscription.plan_id or "basic"
    plan = SUBSCRIPTION_PLANS.get(plan_id, {"name": "Стандартный"})
    plan_name = plan.get("name", "Стандартный")
    
    # Специальное сообщение для тестового тарифа
    if plan_id == "trial":
        # Форматируем данные о трафике
        remaining_gb = format_bytes_to_gb(subscription.traffic_remaining)
        percent_used = int(100 - (subscription.traffic_remaining / subscription.traffic_limit * 100))
        
        if days_left == 0:
            message = (
                f"⚠️ <b>Важное уведомление:</b> Ваш тестовый период <b>заканчивается сегодня</b>!\n\n"
                f"• Осталось времени: <b>менее 24 часов</b>\n"
                f"• Доступный трафик: <b>{remaining_gb} ГБ</b> (использовано {percent_used}%)\n\n"
                f"Чтобы продолжить пользоваться сервисом без ограничений, "
                f"пожалуйста, приобретите полноценную подписку, нажав на кнопку ниже "
                f"или выбрав «Продлить Подписку» в главном меню бота."
            )
        elif days_left == 1:
            message = (
                f"⚠️ <b>Важное уведомление:</b> Ваш тестовый период заканчивается <b>завтра</b>!\n\n"
                f"• Осталось времени: <b>1 день</b>\n"
                f"• Доступный трафик: <b>{remaining_gb} ГБ</b> (использовано {percent_used}%)\n\n"
                f"Чтобы продолжить пользоваться сервисом без ограничений, "
                f"пожалуйста, приобретите полноценную подписку, нажав на кнопку ниже "
                f"или выбрав «Продлить Подписку» в главном меню бота."
            )
        else:
            message = (
                f"ℹ️ <b>Напоминание:</b> Ваш тестовый период действует еще <b>{days_left} дня</b>.\n\n"
                f"• Доступный трафик: <b>{remaining_gb} ГБ</b> (использовано {percent_used}%)\n\n"
                f"Когда тестовый период закончится, вы сможете продолжить пользоваться сервисом, "
                f"приобретя полноценную подписку. Вы уже можете выбрать тариф, "
                f"нажав на кнопку ниже или через меню бота."
            )
    # Для других тарифов
    else:
        if days_left == 1:
            message = (
                f"⚠️ <b>Внимание!</b> Ваша подписка по тарифу <b>{plan_name}</b> истекает <b>завтра</b>.\n\n"
                f"Для продления подписки нажмите кнопку ниже или выберите "
                f"«Продлить Подписку» в главном меню бота."
            )
        else:
            message = (
                f"ℹ️ <b>Напоминание:</b> Ваша подписка по тарифу <b>{plan_name}</b> истекает через <b>{days_left} дней</b>.\n\n"
                f"Для продления подписки нажмите кнопку ниже или выберите "
                f"«Продлить Подписку» в главном меню бота."
            )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message,
            reply_markup=keyboard.as_markup()
        )
        
    except Exception as e:
        logger.error(f"Error sending expiry notification to user {user.id}: {e}")


async def send_traffic_warning(bot: Bot, user: User, subscription: Subscription) -> None:
    """Send notification about low traffic"""
    from config import SUBSCRIPTION_PLANS
    from bot.utils.formatting import format_bytes_to_gb
    
    # Create keyboard for additional traffic purchase
    keyboard = InlineKeyboardBuilder()
    keyboard.button(
        text="💰 Купить трафик",
        callback_data=f"buy_traffic"
    )
    keyboard.button(
        text="📋 Информация о подписке",
        callback_data=f"subscription_info"
    )
    keyboard.adjust(1)
    
    # Получаем информацию о тарифе
    plan_id = subscription.plan_id or "basic"
    plan = SUBSCRIPTION_PLANS.get(plan_id, {"name": "Стандартный"})
    plan_name = plan.get("name", "Стандартный")
    
    # Форматируем данные о трафике
    remaining_gb = format_bytes_to_gb(subscription.traffic_remaining)
    percent_used = int(100 - (subscription.traffic_remaining / subscription.traffic_limit * 100))
    
    message = (
        f"⚠️ <b>Внимание!</b> У вас заканчивается трафик по тарифу <b>{plan_name}</b>.\n\n"
        f"Осталось: <b>{remaining_gb} ГБ</b> (использовано {percent_used}%)\n\n"
        f"Для продления подписки или покупки дополнительного трафика "
        f"нажмите кнопку ниже."
    )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message,
            reply_markup=keyboard.as_markup()
        )
        
    except Exception as e:
        logger.error(f"Error sending traffic warning to user {user.id}: {e}")


async def send_payment_confirmation(bot: Bot, user_id: int, payment_id: str, amount: float, 
                                   plan_name: str) -> None:
    """Send payment confirmation notification"""
    message = (
        f"✅ <b>Оплата успешно выполнена!</b>\n\n"
        f"💰 Сумма: {amount} руб.\n"
        f"📦 Тариф: {plan_name}\n"
        f"🧾 ID платежа: {payment_id}\n\n"
        f"Ваша подписка активирована.\n"
        f"Для просмотра деталей перейдите в «Личный Кабинет» → «Текущая Подписка»."
    )
    
    try:
        await bot.send_message(
            chat_id=user_id,
            text=message
        )
        
    except Exception as e:
        logger.error(f"Error sending payment confirmation to user {user_id}: {e}")


async def send_subscription_activated(bot: Bot, user_id: int, subscription: Subscription, 
                                     plan_name: str) -> None:
    """Send notification about subscription activation"""
    expires_at_str = subscription.expires_at.strftime("%d.%m.%Y")
    
    message = (
        f"🎉 <b>Ваша подписка успешно активирована!</b>\n\n"
        f"📦 Тариф: {plan_name}\n"
        f"⏱️ Действует до: {expires_at_str}\n"
        f"📊 Доступный трафик: {subscription.traffic_limit} ГБ\n"
        f"🔑 Ключ VPN: <code>{subscription.vpn_key}</code>\n\n"
        f"Сохраните ключ для настройки приложения.\n"
        f"Для просмотра инструкций перейдите в «Личный Кабинет» → «Текущая Подписка»."
    )
    
    try:
        await bot.send_message(
            chat_id=user_id,
            text=message
        )
        
    except Exception as e:
        logger.error(f"Error sending subscription activation to user {user_id}: {e}")
