"""
Сервисный модуль для работы с панелью 3X-UI, предоставляющий унифицированный интерфейс
для различных функций работы с панелью.
"""

import logging
import asyncio
from typing import Dict, Optional, Any, List, Tuple

# Настройка логирования
logger = logging.getLogger(__name__)

# Импортируем оптимизированное API
from bot.services.optimized_panel_api import (
    login,
    get_inbounds,
    create_client,
    create_client_without_link,
    get_client_link
)

class PanelError(Exception):
    """Ошибка при работе с панелью"""
    pass

async def check_panel_connection() -> bool:
    """
    Проверяет соединение с панелью
    
    Returns:
        bool: True если соединение установлено, иначе False
    """
    try:
        auth_result = await login()
        return auth_result
    except Exception as e:
        logger.error(f"Ошибка при проверке соединения с панелью: {e}")
        return False

async def create_vpn_account(user_id: int, email: str = None, user_name: str = None, 
                          protocol: str = "VLESS", traffic_limit_gb: int = 5, days: int = 30) -> Dict[str, Any]:
    """
    Создает новый VPN аккаунт в панели
    
    Args:
        user_id: ID пользователя Telegram
        email: Email пользователя (если None, будет сгенерирован)
        user_name: Имя пользователя для генерации email
        protocol: Протокол (VLESS, VMESS, SHADOWSOCKS, TROJAN)
        traffic_limit_gb: Лимит трафика в ГБ
        days: Срок действия в днях
        
    Returns:
        Dict[str, Any]: Информация о созданном аккаунте (без ссылки)
    """
    # Генерируем email если не предоставлен
    if not email:
        # Получаем username из базы данных для нового формата
        from bot.database.operations import get_user
        try:
            user = await get_user(user_id)
            username = user.username if user and user.username else (user_name if user_name else "user")
        except:
            username = user_name if user_name else "user"
        
        # Новая схема email: ID@Username
        email = f"{user_id}@{username}"
    
    logger.info(f"Создание нового VPN аккаунта: {email}, {protocol}, {traffic_limit_gb}GB, {days} days")
    
    try:
        # Создаем клиента через рабочий API
        from bot.services.working_panel_api import create_client_in_panel
        
        client_id = await create_client_in_panel(
            email=email,
            traffic_gb=traffic_limit_gb,
            days=days
        )
        
        if client_id:
            return {
                "client_id": client_id,
                "email": email,
                "protocol": protocol,
                "traffic_limit_gb": traffic_limit_gb,
                "days": days
            }
        else:
            raise PanelError("Не удалось создать VPN аккаунт")
            
    except Exception as e:
        logger.error(f"Ошибка при создании VPN аккаунта: {e}")
        raise PanelError(f"Ошибка при создании VPN аккаунта: {e}")
        
async def update_subscription_in_panel(subscription_id: int, email: str, expiry_days: int = 30) -> bool:
    """
    Обновляет существующую подписку в панели
    
    Args:
        subscription_id: ID подписки в базе данных
        email: Email пользователя (используется как идентификатор в панели)
        expiry_days: Новый срок действия в днях
        
    Returns:
        bool: True если обновление успешно, иначе False
    """
    logger.info(f"Обновление подписки в панели: {subscription_id}, {email}, {expiry_days} days")
    
    try:
        # Пока просто пытаемся создать нового клиента с таким же email
        # В будущем можно реализовать обновление существующего
        result = await create_client(
            email=email,
            traffic_gb=10,  # Стандартный лимит трафика при продлении
            days=expiry_days
        )
        
        if result:
            client_id, link = result
        else:
            client_id, link = None, None
        
        return bool(client_id and link)
            
    except Exception as e:
        logger.error(f"Ошибка при обновлении подписки в панели: {e}")
        return False
        
async def get_vpn_key(email: str) -> Optional[str]:
    """
    Получает ключ VPN для пользователя по email
    
    Args:
        email: Email пользователя
        
    Returns:
        Optional[str]: VPN ключ или None при ошибке
    """
    logger.info(f"Получение VPN ключа для: {email}")
    
    try:
        # Получаем ссылку
        link = await get_client_link(email)
        
        if link:
            # Из ссылки вида vless://UUID@server:port?encryption=... извлекаем UUID
            if link.startswith("vless://"):
                # Извлекаем UUID между vless:// и @
                parts = link.split("://", 1)[1].split("@", 1)
                if len(parts) > 0:
                    return parts[0]  # Это UUID клиента
            
            return link  # Возвращаем всю ссылку, если не смогли извлечь UUID
        
        return None
    except Exception as e:
        logger.error(f"Ошибка при получении VPN ключа: {e}")
        return None
        
async def get_vpn_link(email: str, protocol: str = "vless") -> Optional[str]:
    """
    Получает полную VPN ссылку подключения для пользователя
    
    Args:
        email: Email пользователя
        protocol: Протокол (vless или shadowsocks)
        
    Returns:
        Optional[str]: Полная ссылка для подключения или None при ошибке
    """
    logger.info(f"Получение полной VPN ссылки для: {email}, протокол: {protocol}")
    
    try:
        # Попробуем получить ссылку несколько раз, с небольшими задержками
        max_attempts = 3
        link = None
        
        for attempt in range(max_attempts):
            link = await get_client_link(email, protocol)
            if link:
                logger.info(f"Получена ссылка для {email} с попытки {attempt+1}")
                break
                
            logger.warning(f"Попытка {attempt+1}/{max_attempts} получить ссылку для {email} не удалась, ожидание...")
            await asyncio.sleep(1)  # Небольшая задержка между попытками
        
        if not link:
            logger.warning(f"Не удалось получить ссылку для {email} после {max_attempts} попыток")
            
        return link
    except Exception as e:
        logger.error(f"Ошибка при получении VPN ссылки: {e}")
        return None


async def get_traffic_stats(client_id: str = None, email: str = None) -> Optional[Dict[str, Any]]:
    """
    Получает статистику использования трафика для клиента через Subscription URL API
    
    Args:
        client_id: ID клиента в панели
        email: Email клиента (если client_id не указан)
        
    Returns:
        Optional[Dict[str, Any]]: Статистика трафика или None при ошибке
    """
    from bot.services.sub_url_fetcher import fetch_subscription_data
    
    logger.info(f"Получение статистики трафика: client_id={client_id}, email={email}")
    
    if not client_id and not email:
        logger.error("Не указан идентификатор клиента для получения статистики трафика")
        return None
    
    try:
        # Получаем данные клиента через Subscription URL
        client_data = await fetch_subscription_data(client_id=client_id, email=email)
        
        if not client_data:
            logger.warning(f"Не удалось получить данные о клиенте для трафика: {client_id or email}")
            return None
        
        # Извлекаем информацию о трафике из данных клиента
        # Поле traffic_usage содержит использованный трафик в байтах
        traffic_used = client_data.get("traffic_usage", 0)
        if isinstance(traffic_used, str):
            try:
                traffic_used = int(traffic_used)
            except ValueError:
                traffic_used = 0
        
        # Возвращаем данные в байтах для совместимости с форматированием
        return {
            "traffic_used": traffic_used
        }
    except Exception as e:
        logger.error(f"Ошибка при получении статистики трафика: {e}")
        return None
        
# Примечание: update_subscription_in_panel используется для обновления подписки в панели 3X-UI
# Для обновления подписки в базе данных используйте update_subscription из bot.database.operations