"""
Модуль для получения информации о клиентах VPN
"""

import asyncio
import logging
import base64
import aiohttp
import ssl
import json
from typing import Dict, Optional, Any, List

# Настройка логирования
logger = logging.getLogger(__name__)

class ClientInfoError(Exception):
    """Ошибка при получении информации о клиенте"""
    pass

async def get_client_info(email: str, client_id: Optional[str] = None) -> Dict[str, Any]:
    """
    Получить информацию о клиенте по email или client_id
    
    Args:
        email: Email клиента
        client_id: ID клиента (опционально)
        
    Returns:
        Dict[str, Any]: Информация о клиенте
        
    Raises:
        ClientInfoError: Если не удалось получить информацию о клиенте
    """
    logger.info(f"Получение информации о клиенте: {email}")
    
    try:
        from bot.services.optimized_panel_api import get_client_link
        # Получаем ссылку клиента
        link = await get_client_link(email)
        
        if link:
            return {
                "email": email,
                "direct_link": link,
                "status": "ACTIVE",
                "protocol": "VLESS" if link.startswith("vless://") else "SHADOWSOCKS"
            }
        else:
            raise ClientInfoError(f"Не удалось найти ссылку для клиента {email}")
    
    except Exception as e:
        logger.error(f"Ошибка при получении информации о клиенте: {e}")
        raise ClientInfoError(f"Ошибка при получении информации о клиенте: {e}")