"""
Оптимизированный модуль для повторного получения ссылки на подписку
"""

import asyncio
import logging
from aiogram import F, Router
from aiogram.types import CallbackQuery, InlineKeyboardButton, Message
from aiogram import types

from bot.services.optimized_panel_api import get_client_link

# Настройка логирования
logger = logging.getLogger(__name__)

# Создаем роутер
router = Router(name="refresh_link_router")


@router.callback_query(F.data.startswith("refresh_link|"))
async def refresh_client_link(callback_query: CallbackQuery):
    """Повторная попытка получить ссылку для клиента"""
    try:
        # Получаем данные из callback_data
        parts = callback_query.data.split("|")
        if len(parts) < 2:
            await callback_query.message.edit_text("❌ Некорректные данные для обновления ссылки.")
            return
        
        # Email клиента
        email = parts[1]
        # Протокол, если есть
        protocol = parts[2] if len(parts) > 2 else "vless"
        
        await callback_query.message.edit_text(
            "⏳ <b>Повторная попытка получения ссылки из панели...</b>",
            parse_mode="HTML"
        )
        
        logger.info(f"Повторная попытка получения ссылки для клиента {email}")
        
        # Увеличиваем задержку до 5 секунд, чтобы дать панели время обновить данные
        # (Задержка уже встроена в функцию get_client_link)
        
        # Получаем ссылку через оптимизированный API
        direct_link = await get_client_link(email, protocol)
        
        # Формируем сообщение с результатами
        if direct_link:
            message = (
                f"✅ <b>Ссылка успешно получена!</b>\n\n"
                f"📋 <b>Прямая ссылка для импорта:</b>\n"
                f"<code>{direct_link}</code>\n\n"
            )
            
            # Обычная клавиатура без кнопки повторного получения
            keyboard = types.InlineKeyboardMarkup(inline_keyboard=[])
        else:
            message = (
                f"⚠️ <b>Не удалось получить ссылку!</b>\n\n"
                f"Возможно, нужно подождать больше времени, пока панель обновит данные.\n\n"
            )
            
            # Клавиатура с кнопкой для повторного получения
            keyboard = types.InlineKeyboardMarkup(inline_keyboard=[
                [types.InlineKeyboardButton(
                    text="🔄 Попробовать еще раз",
                    callback_data=f"refresh_link|{email}|{protocol}"
                )]
            ])
        
        await callback_query.message.edit_text(message, reply_markup=keyboard, parse_mode="HTML")
        
    except Exception as e:
        logger.error(f"Ошибка при обновлении ссылки клиента: {e}")
        await callback_query.message.edit_text(
            f"❌ Произошла ошибка при обновлении ссылки:\n{e}",
            reply_markup=types.InlineKeyboardMarkup(inline_keyboard=[
                [types.InlineKeyboardButton(
                    text="🔄 Попробовать еще раз",
                    callback_data=callback_query.data
                )]
            ]),
            parse_mode="HTML"
        )


def register_handlers(dp):
    """Регистрирует обработчики"""
    dp.include_router(router)