"""
Модуль для работы с активными подписками
"""
import logging
from datetime import datetime
from typing import List, Optional

from bot.database.models import Subscription
from bot.database.operations import db_error_handler, get_db_connection

logger = logging.getLogger(__name__)


@db_error_handler
async def get_all_active_subscriptions() -> List[Subscription]:
    """
    Получает все активные подписки из базы данных.
    
    Активная подписка должна:
    1. Иметь действующий срок (не истекший)
    2. Иметь статус активной подписки (не EXPIRED и не SUSPENDED)
    
    Returns:
        List[Subscription]: Список активных подписок
    """
    async with get_db_connection() as db:
        try:
            # Активные подписки: не истекшие и со статусом отличным от EXPIRED или SUSPENDED
            cursor = await db.execute(
                """
                SELECT * FROM subscriptions 
                WHERE (expires_at IS NULL OR expires_at > CURRENT_TIMESTAMP)
                AND (status IS NULL OR status NOT IN ('EXPIRED', 'SUSPENDED'))
                """
            )
            rows = await cursor.fetchall()
            
            subscriptions = []
            for row in rows:
                try:
                    # Преобразуем строковые даты в объекты datetime
                    created_at = datetime.fromisoformat(row['created_at']) if row['created_at'] else None
                    expires_at = datetime.fromisoformat(row['expires_at']) if row['expires_at'] else None
                    
                    # Приводим значения трафика к float для совместимости
                    traffic_limit = float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0
                    traffic_used = float(row['traffic_used']) if row['traffic_used'] is not None else 0.0
                    
                    subscription = Subscription(
                        id=row['id'],
                        user_id=row['user_id'],
                        plan_id=row['plan_id'],
                        created_at=created_at,
                        expires_at=expires_at,
                        traffic_limit=traffic_limit,
                        traffic_used=traffic_used,
                        vpn_user_id=row['vpn_user_id'],
                        vpn_key=row['vpn_key'],
                        status=row['status'],
                        vpn_email=row['vpn_email']
                    )
                    subscriptions.append(subscription)
                except Exception as e:
                    logger.error(f"Ошибка при обработке данных активной подписки для ID {row['id']}: {e}")
            
            logger.info(f"Найдено {len(subscriptions)} активных подписок")
            return subscriptions
        except Exception as e:
            logger.error(f"Ошибка при получении активных подписок: {e}")
            return []