import asyncio
import logging
from aiogram import Bot, Dispatcher
from aiogram.fsm.storage.memory import MemoryStorage
from aiogram.client.default import DefaultBotProperties

from config import BOT_TOKEN, PANEL_API_URL, PANEL_USERNAME, PANEL_PASSWORD
from bot.database.operations import init_db
from bot.handlers import main_menu, subscription, payment, personal, admin_handlers, optimized_refresh_link
from bot.services import traffic_manager
# Старый планировщик удален - используем только EnhancedNotificationScheduler
from bot.jobs.enhanced_notification_scheduler import EnhancedNotificationScheduler
from bot.services.simple_panel_api import simple_panel_api

# Импортируем Flask-приложение для Gunicorn
from app import app

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Initialize bot and dispatcher
bot = Bot(
    token=BOT_TOKEN,
    default=DefaultBotProperties(parse_mode="HTML")
)
dp = Dispatcher(storage=MemoryStorage())


async def on_startup():
    # Initialize database
    await init_db()
    
    # Обязательная авторизация в панели при запуске
    logger.info("Авторизация в панели 3X-UI при запуске...")
    try:
        from bot.services.working_panel_api import working_api
        login_success = await working_api.login()
        if login_success:
            logger.info("✓ Успешная авторизация в панели 3X-UI")
        else:
            logger.error("✗ Ошибка авторизации в панели - создание новых клиентов будет недоступно")
    except Exception as e:
        logger.error(f"Критическая ошибка авторизации в панели: {e}")
    
    # Тестируем подключение к панели
    logger.info("Проверка подключения к панели...")
    try:
        # Тестовый запрос для проверки доступности API панели
        test_result = await simple_panel_api.get_traffic_by_email("test@example.com")
        if test_result is not None or True:  # API доступен даже если email не найден
            logger.info("✓ Подключение к панели установлено")
        else:
            logger.warning("⚠ Проблемы с подключением к панели, но система продолжит работу")
    except Exception as e:
        logger.error(f"Ошибка подключения к панели: {e}")
    
    # Установка команд бота
    from aiogram.types import BotCommand, BotCommandScopeDefault, BotCommandScopeChat
    from config import ADMIN_IDS
    
    # Общие команды для всех пользователей
    await bot.set_my_commands([
        BotCommand(command="start", description="Запустить бота / Вернуться в главное меню"),
        BotCommand(command="subscription", description="Оформить новую подписку"),
        BotCommand(command="extend", description="Продлить подписку"),
        BotCommand(command="account", description="Личный кабинет")
    ], scope=BotCommandScopeDefault())
    
    # Административные команды только для админов
    admin_commands = [
        BotCommand(command="start", description="Запустить бота / Вернуться в главное меню"),
        BotCommand(command="subscription", description="Оформить новую подписку"),
        BotCommand(command="extend", description="Продлить подписку"),
        BotCommand(command="account", description="Личный кабинет"),
        # Административные команды
        BotCommand(command="admin", description="Список административных команд"),
        BotCommand(command="stats", description="Статистика бота"),
        BotCommand(command="users", description="Список пользователей"),
        BotCommand(command="add_client", description="Добавить нового VPN-клиента"),
        BotCommand(command="reset_subscription", description="Сбросить подписку пользователя")
    ]
    
    # Устанавливаем админ-команды для каждого админа
    for admin_id in ADMIN_IDS:
        try:
            await bot.set_my_commands(
                admin_commands,
                scope=BotCommandScopeChat(chat_id=admin_id)
            )
        except Exception as e:
            logger.error(f"Ошибка при установке команд для админа {admin_id}: {e}")
    
    # Set up command handlers
    logger.info("Registering handlers...")
    
    # Register all handlers
    main_menu.register_handlers(dp)
    subscription.register_handlers(dp)

    payment.register_handlers(dp)  # payment должен быть последним из-за универсального обработчика
    personal.register_handlers(dp)
    admin_handlers.register_admin_handlers(dp)  # Регистрируем обработчики административных команд
    optimized_refresh_link.register_handlers(dp)  # Регистрируем обработчики повторного получения ссылки
    
    # Старый планировщик удален - используем только EnhancedNotificationScheduler
    
    # Start notification service
    notification_scheduler = EnhancedNotificationScheduler(bot)
    await notification_scheduler.start()
    logger.info("Notification service started")
    
    # Register traffic management handlers
    traffic_manager.register_handlers(dp)
    logger.info("Traffic management handlers registered")
    
    logger.info("Bot started!")


async def on_shutdown():
    """Функция корректного завершения работы бота"""
    logger.info("Завершение работы бота...")
    logger.info("✓ Завершение работы выполнено")


async def main():
    logger.info("Starting bot...")
    
    # Set up startup and shutdown actions
    dp.startup.register(on_startup)
    dp.shutdown.register(on_shutdown)
    
    # Start polling
    await dp.start_polling(bot)


if __name__ == "__main__":
    try:
        asyncio.run(main())
    except (KeyboardInterrupt, SystemExit):
        logger.info("Bot stopped!")
