"""
Конфигурационный файл для хранения API ключей и токенов
Используется для синхронизации между различными сервисами
"""

import os
from typing import Dict, Any, Optional

class APIKeysConfig:
    """Класс для управления API ключами и токенами"""
    
    def __init__(self):
        self._keys = {}
        self._load_from_env()
    
    def _load_from_env(self):
        """Загрузка ключей из переменных окружения"""
        
        # Telegram Bot
        self._keys['telegram'] = {
            'bot_token': os.environ.get('BOT_TOKEN', ''),
            'bot_username': '@RazDvaVPN_bot'
        }
        
        # YooKassa (платежная система)
        self._keys['yookassa'] = {
            'shop_id': os.environ.get('YOOKASSA_SHOP_ID', ''),
            'secret_key': os.environ.get('YOOKASSA_SECRET_KEY', ''),
            'test_mode': True  # Переключение между тестовым и боевым режимом
        }
        
        # 3X-UI Panel (VPN панель)
        self._keys['panel'] = {
            'url': os.environ.get('PANEL_URL', ''),
            'username': os.environ.get('PANEL_USERNAME', ''),
            'password': os.environ.get('PANEL_PASSWORD', ''),
            'timeout': 30
        }
        
        # Database (используем только SQLite)
        self._keys['database'] = {
            'sqlite_path': './database/vpnbot.db',
            'url': 'sqlite:///database/vpnbot.db'
        }
        
        # Email Service (для будущего использования)
        self._keys['email'] = {
            'smtp_server': '',
            'smtp_port': 587,
            'username': '',
            'password': '',
            'use_tls': True
        }
        
        # Backup Service (для резервного копирования)
        self._keys['backup'] = {
            'enabled': True,
            'interval_hours': 24,
            'max_backups': 7
        }
    
    def get_telegram_config(self) -> Dict[str, str]:
        """Получить конфигурацию Telegram"""
        return self._keys['telegram']
    
    def get_yookassa_config(self) -> Dict[str, Any]:
        """Получить конфигурацию YooKassa"""
        return self._keys['yookassa']
    
    def get_panel_config(self) -> Dict[str, Any]:
        """Получить конфигурацию панели 3X-UI"""
        return self._keys['panel']
    
    def get_database_config(self) -> Dict[str, str]:
        """Получить конфигурацию базы данных"""
        return self._keys['database']
    
    def get_email_config(self) -> Dict[str, Any]:
        """Получить конфигурацию email сервиса"""
        return self._keys['email']
    
    def get_backup_config(self) -> Dict[str, Any]:
        """Получить конфигурацию резервного копирования"""
        return self._keys['backup']
    
    def is_yookassa_configured(self) -> bool:
        """Проверить, настроена ли YooKassa"""
        config = self._keys['yookassa']
        return bool(config['shop_id'] and config['secret_key'])
    
    def is_panel_configured(self) -> bool:
        """Проверить, настроена ли панель"""
        config = self._keys['panel']
        return bool(config['url'] and config['username'])
    
    def is_telegram_configured(self) -> bool:
        """Проверить, настроен ли Telegram бот"""
        config = self._keys['telegram']
        return bool(config['bot_token'])
    
    def update_key(self, service: str, key: str, value: Any):
        """Обновить конкретный ключ"""
        if service in self._keys and key in self._keys[service]:
            self._keys[service][key] = value
    
    def get_missing_keys(self) -> Dict[str, list]:
        """Получить список отсутствующих ключей"""
        missing = {}
        
        # Проверяем обязательные ключи
        if not self.is_telegram_configured():
            missing['telegram'] = ['BOT_TOKEN']
        
        if not self.is_yookassa_configured():
            missing['yookassa'] = []
            if not self._keys['yookassa']['shop_id']:
                missing['yookassa'].append('YOOKASSA_SHOP_ID')
            if not self._keys['yookassa']['secret_key']:
                missing['yookassa'].append('YOOKASSA_SECRET_KEY')
        
        if not self._keys['panel']['password']:
            missing['panel'] = ['PANEL_PASSWORD']
        
        return missing
    
    def get_status_report(self) -> Dict[str, str]:
        """Получить отчет о статусе конфигурации"""
        return {
            'telegram': '✓ Настроен' if self.is_telegram_configured() else '❌ Не настроен',
            'yookassa': '✓ Настроен' if self.is_yookassa_configured() else '❌ Не настроен', 
            'panel': '✓ Настроен' if self.is_panel_configured() else '⚠️ Частично настроен',
            'database': '✓ Доступна' if self._keys['database']['url'] else '❌ Не настроена'
        }

# Глобальный экземпляр конфигурации
api_keys = APIKeysConfig()

# Функции для быстрого доступа
def get_telegram_token() -> str:
    """Получить токен Telegram бота"""
    return api_keys.get_telegram_config()['bot_token']

def get_yookassa_credentials() -> tuple:
    """Получить учетные данные YooKassa"""
    config = api_keys.get_yookassa_config()
    return config['shop_id'], config['secret_key']

def get_panel_credentials() -> tuple:
    """Получить учетные данные панели"""
    config = api_keys.get_panel_config()
    return config['url'], config['username'], config['password']

def get_database_url() -> str:
    """Получить URL базы данных"""
    return api_keys.get_database_config()['url']

def print_configuration_status():
    """Вывести статус конфигурации"""
    print("\n=== Статус конфигурации API ключей ===")
    status = api_keys.get_status_report()
    for service, status_text in status.items():
        print(f"{service.capitalize()}: {status_text}")
    
    missing = api_keys.get_missing_keys()
    if missing:
        print("\n=== Отсутствующие ключи ===")
        for service, keys in missing.items():
            print(f"{service.capitalize()}: {', '.join(keys)}")
    else:
        print("\n✅ Все ключи настроены корректно!")

if __name__ == "__main__":
    print_configuration_status()