"""
Утилиты для валидации и проверки корректности расчета дат
Предотвращает ошибки в алгоритмах расчета подписок
"""

from datetime import datetime, timedelta
from typing import Optional, Dict, Any
import logging

logger = logging.getLogger(__name__)

def validate_subscription_dates(
    payment_date: datetime,
    plan_duration_days: int,
    calculated_expiry: datetime,
    user_id: int,
    plan_id: str
) -> bool:
    """
    Валидация корректности расчета даты истечения подписки
    
    Args:
        payment_date: Дата оплаты
        plan_duration_days: Продолжительность плана в днях
        calculated_expiry: Рассчитанная дата истечения
        user_id: ID пользователя
        plan_id: ID тарифного плана
        
    Returns:
        bool: True если расчет корректен, False если есть ошибки
    """
    try:
        # Убираем timezone info для корректного сравнения
        if payment_date.tzinfo is not None:
            payment_date = payment_date.replace(tzinfo=None)
        if calculated_expiry.tzinfo is not None:
            calculated_expiry = calculated_expiry.replace(tzinfo=None)
        
        # Рассчитываем ожидаемую дату
        expected_expiry = payment_date + timedelta(days=plan_duration_days)
        
        # Проверяем точность расчета (допускаем погрешность в 1 минуту)
        time_diff = abs((calculated_expiry - expected_expiry).total_seconds())
        
        if time_diff > 60:  # больше 1 минуты расхождения
            logger.error(
                f"КРИТИЧЕСКАЯ ОШИБКА РАСЧЕТА ДАТ для пользователя {user_id}, план {plan_id}:\n"
                f"Дата оплаты: {payment_date}\n"
                f"Продолжительность: {plan_duration_days} дней\n"
                f"Ожидалось: {expected_expiry}\n"
                f"Рассчитано: {calculated_expiry}\n"
                f"Расхождение: {time_diff} секунд"
            )
            return False
        
        # Проверяем, что дата истечения в будущем
        if calculated_expiry <= payment_date:
            logger.error(
                f"ОШИБКА: Дата истечения не в будущем для пользователя {user_id}\n"
                f"Дата оплаты: {payment_date}\n"
                f"Дата истечения: {calculated_expiry}"
            )
            return False
        
        # Проверяем разумность продолжительности (от 1 дня до 2 лет)
        actual_duration = (calculated_expiry - payment_date).days
        if not (1 <= actual_duration <= 730):
            logger.error(
                f"ОШИБКА: Неразумная продолжительность подписки {actual_duration} дней "
                f"для пользователя {user_id}, план {plan_id}"
            )
            return False
        
        logger.info(
            f"✓ Валидация прошла успешно для пользователя {user_id}, план {plan_id}:\n"
            f"Дата оплаты: {payment_date.strftime('%d.%m.%Y %H:%M:%S')}\n"
            f"Дата истечения: {calculated_expiry.strftime('%d.%m.%Y %H:%M:%S')}\n"
            f"Продолжительность: {plan_duration_days} дней"
        )
        return True
        
    except Exception as e:
        logger.error(f"Ошибка при валидации дат для пользователя {user_id}: {e}")
        return False

def format_date_for_display(date_obj: datetime, format_type: str = "full") -> str:
    """
    Форматирование даты для отображения пользователю
    
    Args:
        date_obj: Объект datetime
        format_type: Тип формата ("full", "date_only", "short")
        
    Returns:
        str: Отформатированная дата
    """
    if date_obj is None:
        return "Не установлено"
    
    # Убираем timezone info если есть
    if date_obj.tzinfo is not None:
        date_obj = date_obj.replace(tzinfo=None)
    
    if format_type == "date_only":
        return date_obj.strftime("%d.%m.%Y")
    elif format_type == "short":
        return date_obj.strftime("%d.%m.%Y %H:%M")
    else:  # full
        return date_obj.strftime("%d.%m.%Y %H:%M:%S")

def calculate_days_remaining(expiry_date: datetime) -> int:
    """
    Рассчитывает количество дней до истечения подписки
    
    Args:
        expiry_date: Дата истечения подписки
        
    Returns:
        int: Количество дней (отрицательное если просрочено)
    """
    if expiry_date is None:
        return -999
    
    # Убираем timezone info
    if expiry_date.tzinfo is not None:
        expiry_date = expiry_date.replace(tzinfo=None)
    
    now = datetime.now()
    delta = expiry_date - now
    return delta.days

def get_subscription_status(expiry_date: datetime) -> str:
    """
    Определяет статус подписки на основе даты истечения
    
    Args:
        expiry_date: Дата истечения подписки
        
    Returns:
        str: Статус подписки
    """
    days_remaining = calculate_days_remaining(expiry_date)
    
    if days_remaining < 0:
        return "EXPIRED"
    elif days_remaining == 0:
        return "EXPIRES_TODAY" 
    elif days_remaining <= 3:
        return "EXPIRES_SOON"
    else:
        return "ACTIVE"

def create_expiry_summary(expiry_date: datetime, plan_name: str) -> Dict[str, Any]:
    """
    Создает сводку по истечению подписки
    
    Args:
        expiry_date: Дата истечения
        plan_name: Название тарифного плана
        
    Returns:
        Dict: Сводная информация
    """
    days_remaining = calculate_days_remaining(expiry_date)
    status = get_subscription_status(expiry_date)
    
    return {
        "expiry_date": format_date_for_display(expiry_date, "date_only"),
        "expiry_time": format_date_for_display(expiry_date, "short"),
        "days_remaining": days_remaining,
        "status": status,
        "plan_name": plan_name,
        "is_active": status in ["ACTIVE", "EXPIRES_SOON"],
        "needs_renewal": status in ["EXPIRED", "EXPIRES_TODAY", "EXPIRES_SOON"]
    }

def log_date_calculation(
    user_id: int, 
    plan_id: str, 
    payment_date: datetime, 
    calculated_expiry: datetime,
    is_extension: bool = False
) -> None:
    """
    Логирование расчета дат для аудита
    
    Args:
        user_id: ID пользователя
        plan_id: ID плана
        payment_date: Дата платежа
        calculated_expiry: Рассчитанная дата истечения
        is_extension: Является ли это продлением
    """
    operation_type = "ПРОДЛЕНИЕ" if is_extension else "НОВАЯ ПОДПИСКА"
    
    logger.info(
        f"[АУДИТ РАСЧЕТА ДАТ] {operation_type}\n"
        f"Пользователь: {user_id}\n"
        f"План: {plan_id}\n"
        f"Дата платежа: {payment_date.strftime('%d.%m.%Y %H:%M:%S')}\n"
        f"Дата истечения: {calculated_expiry.strftime('%d.%m.%Y %H:%M:%S')}\n"
        f"Продолжительность: {(calculated_expiry - payment_date).days} дней"
    )