"""
Модуль для уведомлений об изменениях подписок
"""
import logging
from typing import Optional
from aiogram import Bot
from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton

from bot.database.operations import get_user, get_user_subscription
from bot.database.models import Subscription

logger = logging.getLogger(__name__)


class SubscriptionNotifier:
    """Класс для отправки уведомлений об изменениях подписок"""
    
    def __init__(self, bot: Bot):
        self.bot = bot
    
    async def notify_subscription_upgraded(self, user_id: int, old_plan: str, new_plan: str):
        """
        Уведомляет пользователя об обновлении подписки
        
        Args:
            user_id: ID пользователя
            old_plan: Старый тарифный план
            new_plan: Новый тарифный план
        """
        try:
            # Получаем данные пользователя и подписки
            user = await get_user(user_id)
            subscription = await get_user_subscription(user_id)
            
            if not user or not subscription:
                logger.error(f"Не удалось найти пользователя {user_id} или его подписку")
                return False
            
            # Формируем сообщение
            plan_names = {
                'trial': 'Пробный период',
                'basic': 'Базовый',
                'premium': 'Премиум',
                'pro': 'Профессиональный'
            }
            
            old_plan_name = plan_names.get(old_plan, old_plan)
            new_plan_name = plan_names.get(new_plan, new_plan)
            
            message_text = f"""🎉 Поздравляем! Ваша подписка обновлена!

📈 Изменения:
• Старый тариф: {old_plan_name}
• Новый тариф: {new_plan_name}

📊 Ваши новые возможности:
• Действует до: {subscription.expires_at.strftime('%d.%m.%Y %H:%M')}
• Лимит трафика: {subscription.traffic_limit / (1024**3):.0f} ГБ
• Дней осталось: {subscription.days_remaining}

🔗 Ваша VPN ссылка остается прежней - никаких дополнительных настроек не требуется!"""

            # Создаем клавиатуру
            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="📱 Мой профиль", callback_data="profile")],
                [InlineKeyboardButton(text="🔗 Получить ключ VPN", callback_data="get_vpn_key")],
                [InlineKeyboardButton(text="📊 Статистика", callback_data="stats")]
            ])
            
            # Отправляем уведомление
            await self.bot.send_message(
                chat_id=user_id,
                text=message_text,
                reply_markup=keyboard
            )
            
            logger.info(f"Отправлено уведомление об обновлении подписки для пользователя {user_id}")
            return True
            
        except Exception as e:
            logger.error(f"Ошибка при отправке уведомления об обновлении подписки: {e}")
            return False
    
    async def notify_subscription_extended(self, user_id: int, additional_days: int):
        """
        Уведомляет пользователя о продлении подписки
        
        Args:
            user_id: ID пользователя
            additional_days: Количество добавленных дней
        """
        try:
            subscription = await get_user_subscription(user_id)
            
            if not subscription:
                logger.error(f"Не удалось найти подписку для пользователя {user_id}")
                return False
            
            message_text = f"""⏰ Ваша подписка продлена!

✅ Добавлено дней: {additional_days}
📅 Действует до: {subscription.expires_at.strftime('%d.%m.%Y %H:%M')}
🗓 Дней осталось: {subscription.days_remaining}

Спасибо за продление! Продолжайте пользоваться нашим VPN сервисом без ограничений."""

            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="🔗 Получить ключ VPN", callback_data="get_vpn_key")],
                [InlineKeyboardButton(text="📊 Статистика", callback_data="stats")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=message_text,
                reply_markup=keyboard
            )
            
            logger.info(f"Отправлено уведомление о продлении подписки для пользователя {user_id}")
            return True
            
        except Exception as e:
            logger.error(f"Ошибка при отправке уведомления о продлении: {e}")
            return False
    
    async def notify_traffic_added(self, user_id: int, additional_gb: float):
        """
        Уведомляет пользователя о добавлении трафика
        
        Args:
            user_id: ID пользователя  
            additional_gb: Количество добавленных ГБ
        """
        try:
            subscription = await get_user_subscription(user_id)
            
            if not subscription:
                logger.error(f"Не удалось найти подписку для пользователя {user_id}")
                return False
            
            remaining_gb = (subscription.traffic_limit - subscription.traffic_used) / (1024**3)
            
            message_text = f"""📈 Дополнительный трафик добавлен!

✅ Добавлено: {additional_gb:.0f} ГБ
📊 Доступно трафика: {remaining_gb:.1f} ГБ
🏷 Общий лимит: {subscription.traffic_limit / (1024**3):.0f} ГБ

Продолжайте пользоваться интернетом без ограничений!"""

            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="🔗 Получить ключ VPN", callback_data="get_vpn_key")],
                [InlineKeyboardButton(text="📊 Статистика", callback_data="stats")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=message_text,
                reply_markup=keyboard
            )
            
            logger.info(f"Отправлено уведомление о добавлении трафика для пользователя {user_id}")
            return True
            
        except Exception as e:
            logger.error(f"Ошибка при отправке уведомления о трафике: {e}")
            return False