"""
Модуль для получения информации о клиенте через Subscription URL и JSON URL
"""
import logging
import aiohttp
import ssl
import json
import re
from typing import Dict, Any, Optional, Tuple

# Настройка логирования
logger = logging.getLogger(__name__)

class SubUrlFetcherError(Exception):
    """Исключение при ошибках получения данных через Subscription URL"""
    pass

async def fetch_subscription_data(client_id: Optional[str] = None, email: Optional[str] = None, inbound_id: int = 1) -> Optional[Dict[str, Any]]:
    """
    Получает данные о подписке клиента через Subscription URL
    
    Args:
        client_id: ID клиента (опционально)
        email: Email клиента (опционально) - имеет приоритет над client_id
        inbound_id: ID inbound
        
    Returns:
        Optional[Dict[str, Any]]: Данные о подписке клиента или None при ошибке
        
    Note:
        Для поиска клиента должен быть указан хотя бы один из параметров: client_id или email
    """
    
    # Проверяем, что хотя бы один идентификатор указан
    if not client_id and not email:
        logger.error("Не указан ни ID, ни email клиента для поиска")
        return None
    # Формируем URL для получения данных
    sub_url = f"http://31.172.75.92:2096/subavto/{inbound_id}"
    json_url = f"http://31.172.75.92:2096/jsonauto/{inbound_id}"
    
    # Определяем параметры поиска
    search_term = email if email else client_id
    search_type = "email" if email else "ID"
    
    logger.info(f"Запрос данных подписки для клиента ({search_type}: {search_term}), inbound {inbound_id}")
    logger.info(f"Subscription URL: {sub_url}")
    logger.info(f"JSON URL: {json_url}")
    
    # Пробуем получить данные из JSON URL (более полная информация)
    json_data, success_json = await _fetch_url(json_url)
    if success_json:
        try:
            # Преобразуем в JSON
            config_data = json.loads(json_data)
            logger.info(f"Получены данные JSON конфигурации: {len(str(config_data))} символов")
            
            # Извлекаем информацию о клиенте из конфигурации
            client_info = None
            if email:
                # Ищем по email (приоритет) в remarks и других полях
                client_info = extract_client_from_json_config(config_data, client_id=None, email=email)
                if not client_info:
                    # Резервный метод - ищем в полях remark
                    client_info = extract_client_by_email_from_json(config_data, email)
            elif client_id:
                # Ищем по ID
                client_info = extract_client_from_json_config(config_data, client_id=client_id)
                
            if client_info:
                logger.info(f"Найдена информация о клиенте ({search_type}: {search_term}) в JSON конфигурации")
                return client_info
            else:
                logger.warning(f"Клиент ({search_type}: {search_term}) не найден в JSON конфигурации")
        except json.JSONDecodeError as e:
            logger.error(f"Ошибка разбора JSON: {e}")
    
    # Если JSON не сработал, пробуем получить данные из Subscription URL
    sub_data, success_sub = await _fetch_url(sub_url)
    if success_sub:
        try:
            # Преобразуем в список строк и ищем клиента
            lines = sub_data.strip().split('\n')
            
            for line in lines:
                found = False
                
                # Декодируем строку из base64, если возможно, для поиска в данных конфигурации
                decoded_line = line
                try:
                    # Пытаемся декодировать части строки
                    if line.startswith("vless://"):
                        # Для VLESS строк, возможно, remark (после #) содержит email
                        parts = line.split("#", 1)
                        if len(parts) > 1:
                            remark = parts[1]
                            if email and email.lower() in remark.lower():
                                logger.info(f"Найдена строка с email {email} в Subscription (в remark)")
                                found = True
                except Exception as e:
                    logger.debug(f"Ошибка при попытке декодирования строки: {e}")
                
                # Если не нашли в декодированной строке, ищем в оригинальной
                if not found:
                    if email and email.lower() in line.lower():
                        logger.info(f"Найдена строка с email {email} в Subscription")
                        found = True
                    elif client_id and client_id.lower() in line.lower():
                        logger.info(f"Найдена строка с ID {client_id} в Subscription")
                        found = True
                
                if found:
                    # Определяем ID из строки для корректного парсинга
                    extracted_id = extract_id_from_subscription_line(line)
                    if extracted_id:
                        return parse_subscription_line(line, extracted_id)
                    else:
                        return parse_subscription_line(line, client_id or "unknown")
            
            logger.warning(f"Клиент ({search_type}: {search_term}) не найден в Subscription данных")
        except Exception as e:
            logger.error(f"Ошибка при обработке Subscription данных: {e}")
    
    logger.error(f"Не удалось получить данные клиента ({search_type}: {search_term}) через Subscription/JSON URL")
    return None

async def _fetch_url(url: str) -> Tuple[str, bool]:
    """
    Получает данные по URL
    
    Args:
        url: URL для запроса
        
    Returns:
        Tuple[str, bool]: (Данные, Успех)
    """
    try:
        # SSL контекст без проверки сертификата
        ssl_context = ssl.create_default_context()
        ssl_context.check_hostname = False
        ssl_context.verify_mode = ssl.CERT_NONE
        
        # Настройка сессии
        timeout = aiohttp.ClientTimeout(total=30)
        connector = aiohttp.TCPConnector(ssl=ssl_context, force_close=True)
        
        async with aiohttp.ClientSession(connector=connector, timeout=timeout) as session:
            async with session.get(url) as response:
                if response.status == 200:
                    data = await response.text()
                    return data, True
                else:
                    logger.error(f"Ошибка запроса {url}: статус {response.status}")
                    return "", False
    except Exception as e:
        logger.error(f"Ошибка при запросе {url}: {e}")
        return "", False

def extract_id_from_subscription_line(line: str) -> Optional[str]:
    """
    Извлекает ID клиента из строки подписки
    
    Args:
        line: Строка подписки
        
    Returns:
        Optional[str]: ID клиента или None
    """
    try:
        # Проверяем, что это строка VLESS
        if line.startswith("vless://"):
            # Убираем префикс протокола
            content = line[8:]
            
            # Получаем часть до @
            id_part = content.split("@", 1)[0]
            
            # Это и есть ID (UUID)
            return id_part
        
        # Для других форматов пока не реализовано
        return None
    
    except Exception as e:
        logger.error(f"Ошибка при извлечении ID из строки подписки: {e}")
        return None

def extract_client_by_email_from_json(config_data: Dict[str, Any], email: str) -> Optional[Dict[str, Any]]:
    """
    Извлекает информацию о клиенте по email из JSON конфигурации
    
    Args:
        config_data: Данные JSON конфигурации
        email: Email клиента для поиска
        
    Returns:
        Optional[Dict[str, Any]]: Информация о клиенте или None
    """
    try:
        # Ищем Remark в конфигурации, который может содержать email
        if "outbounds" in config_data:
            for outbound in config_data["outbounds"]:
                if outbound.get("protocol") == "vless" and "settings" in outbound:
                    settings = outbound["settings"]
                    if "vnext" in settings:
                        for vnext in settings["vnext"]:
                            server = vnext.get("address", "")
                            port = vnext.get("port", 0)
                            
                            if "users" in vnext:
                                for user in vnext["users"]:
                                    # Ищем email в remark или в тегах
                                    remark = user.get("remark", "")
                                    
                                    # Email может быть в remark или в других метаданных
                                    if email.lower() in remark.lower():
                                        # Это наш клиент
                                        client_id = user.get("id", "")
                                        
                                        # Формируем результат
                                        client_info = {
                                            "id": client_id,
                                            "protocol": "vless",
                                            "server": server,
                                            "port": port,
                                            "flow": user.get("flow", ""),
                                            "encryption": user.get("encryption", "none"),
                                            "email": email,
                                            "remarks": remark,
                                        }
                                        
                                        # Добавляем информацию о реальности, если есть
                                        stream_settings = outbound.get("streamSettings", {})
                                        if stream_settings.get("security") == "reality":
                                            reality_settings = stream_settings.get("realitySettings", {})
                                            client_info.update({
                                                "security": "reality",
                                                "sni": reality_settings.get("serverName", ""),
                                                "fingerprint": reality_settings.get("fingerprint", ""),
                                                "publicKey": reality_settings.get("publicKey", ""),
                                                "shortId": reality_settings.get("shortId", ""),
                                                "spiderX": reality_settings.get("spiderX", ""),
                                            })
                                        
                                        # Добавляем информацию о TCP/WS, если есть
                                        if "network" in stream_settings:
                                            client_info["network"] = stream_settings["network"]
                                            
                                            if stream_settings["network"] == "tcp":
                                                tcp_settings = stream_settings.get("tcpSettings", {})
                                                header = tcp_settings.get("header", {})
                                                client_info["headerType"] = header.get("type", "none")
                                        
                                        # Формируем прямую ссылку
                                        client_info["direct_link"] = build_vless_link(client_info)
                                        
                                        return client_info
        
        # Если клиент не найден
        logger.warning(f"Клиент с email {email} не найден в JSON конфигурации")
        return None
    
    except Exception as e:
        logger.error(f"Ошибка при поиске клиента по email в JSON: {e}")
        return None

def extract_client_from_json_config(config_data: Dict[str, Any], client_id: Optional[str] = None, email: Optional[str] = None) -> Optional[Dict[str, Any]]:
    """
    Извлекает информацию о клиенте из JSON конфигурации
    
    Args:
        config_data: Данные JSON конфигурации, может содержать один объект или массив объектов
        client_id: ID клиента (опционально)
        email: Email клиента (опционально)
        
    Returns:
        Optional[Dict[str, Any]]: Информация о клиенте или None
    """
    try:
        # Если config_data это массив, проверяем каждый элемент
        if isinstance(config_data, list):
            for config_item in config_data:
                result = _extract_client_from_single_config(config_item, client_id, email)
                if result:
                    return result
            return None
        else:
            # Если это одиночный объект конфигурации
            return _extract_client_from_single_config(config_data, client_id, email)
    
    except Exception as e:
        logger.error(f"Ошибка при извлечении информации о клиенте из JSON: {e}")
        return None

def _extract_client_from_single_config(config_item: Dict[str, Any], client_id: Optional[str] = None, email: Optional[str] = None) -> Optional[Dict[str, Any]]:
    """
    Извлекает информацию о клиенте из одного объекта конфигурации
    
    Args:
        config_item: Один объект JSON конфигурации
        client_id: ID клиента (опционально)
        email: Email клиента (опционально)
        
    Returns:
        Optional[Dict[str, Any]]: Информация о клиенте или None
    """
    try:
        # Сначала проверяем remarks, который содержит email и другую информацию
        remarks = config_item.get("remarks", "")
        
        # Если ищем по email и он есть в remarks - это наш клиент
        if email and email.lower() in remarks.lower():
            logger.info(f"Найден клиент по email {email} в remarks: {remarks}")
            # Нашли совпадение в remarks, теперь извлекаем информацию из outbounds
            if "outbounds" in config_item:
                for outbound in config_item["outbounds"]:
                    if outbound.get("protocol") == "vless" and "settings" in outbound:
                        settings = outbound["settings"]
                        if "vnext" in settings:
                            for vnext in settings["vnext"]:
                                server = vnext.get("address", "")
                                port = vnext.get("port", 0)
                                
                                if "users" in vnext and len(vnext["users"]) > 0:
                                    # Берем первого пользователя (в контексте одного config_item)
                                    user = vnext["users"][0]
                                    client_id_from_config = user.get("id", "")
                                    
                                    # Парсим информацию из remarks
                                    traffic_info = extract_traffic_info_from_remarks(remarks)
                                    
                                    # Формируем результат
                                    client_info = {
                                        "id": client_id_from_config,
                                        "protocol": "vless",
                                        "server": server,
                                        "port": port,
                                        "flow": user.get("flow", ""),
                                        "encryption": user.get("encryption", "none"),
                                        "email": email,
                                        "remarks": remarks,
                                    }
                                    
                                    # Добавляем информацию о трафике, если есть
                                    if traffic_info:
                                        client_info.update(traffic_info)
                                    
                                    # Добавляем информацию о реальности, если есть
                                    stream_settings = outbound.get("streamSettings", {})
                                    if stream_settings.get("security") == "reality":
                                        reality_settings = stream_settings.get("realitySettings", {})
                                        client_info.update({
                                            "security": "reality",
                                            "sni": reality_settings.get("serverName", ""),
                                            "fingerprint": reality_settings.get("fingerprint", ""),
                                            "publicKey": reality_settings.get("publicKey", ""),
                                            "shortId": reality_settings.get("shortId", ""),
                                            "spiderX": reality_settings.get("spiderX", ""),
                                        })
                                    
                                    # Добавляем информацию о TCP/WS, если есть
                                    if "network" in stream_settings:
                                        client_info["network"] = stream_settings["network"]
                                        
                                        if stream_settings["network"] == "tcp":
                                            tcp_settings = stream_settings.get("tcpSettings", {})
                                            header = tcp_settings.get("header", {})
                                            client_info["headerType"] = header.get("type", "none")
                                    
                                    # Формируем прямую ссылку
                                    client_info["direct_link"] = build_vless_link(client_info)
                                    
                                    return client_info
        
        # Если ищем по client_id, проверяем традиционный путь
        if client_id and "outbounds" in config_item:
            for outbound in config_item["outbounds"]:
                # Проверяем vless протокол
                if outbound.get("protocol") == "vless" and "settings" in outbound:
                    settings = outbound["settings"]
                    if "vnext" in settings:
                        for vnext in settings["vnext"]:
                            server = vnext.get("address", "")
                            port = vnext.get("port", 0)
                            
                            if "users" in vnext:
                                for user in vnext["users"]:
                                    if user.get("id") == client_id:
                                        # Формируем результат
                                        client_info = {
                                            "id": client_id,
                                            "protocol": "vless",
                                            "server": server,
                                            "port": port,
                                            "flow": user.get("flow", ""),
                                            "encryption": user.get("encryption", "none"),
                                            "remarks": remarks,
                                        }
                                        
                                        # Парсим информацию из remarks
                                        traffic_info = extract_traffic_info_from_remarks(remarks)
                                        if traffic_info:
                                            client_info.update(traffic_info)
                                        
                                        # Добавляем информацию о реальности, если есть
                                        stream_settings = outbound.get("streamSettings", {})
                                        if stream_settings.get("security") == "reality":
                                            reality_settings = stream_settings.get("realitySettings", {})
                                            client_info.update({
                                                "security": "reality",
                                                "sni": reality_settings.get("serverName", ""),
                                                "fingerprint": reality_settings.get("fingerprint", ""),
                                                "publicKey": reality_settings.get("publicKey", ""),
                                                "shortId": reality_settings.get("shortId", ""),
                                                "spiderX": reality_settings.get("spiderX", ""),
                                            })
                                        
                                        # Добавляем информацию о TCP/WS, если есть
                                        if "network" in stream_settings:
                                            client_info["network"] = stream_settings["network"]
                                            
                                            if stream_settings["network"] == "tcp":
                                                tcp_settings = stream_settings.get("tcpSettings", {})
                                                header = tcp_settings.get("header", {})
                                                client_info["headerType"] = header.get("type", "none")
                                        
                                        # Формируем прямую ссылку
                                        client_info["direct_link"] = build_vless_link(client_info)
                                        
                                        return client_info
        
        # Если клиент не найден
        return None
    
    except Exception as e:
        logger.error(f"Ошибка при извлечении информации о клиенте из конфигурации: {e}")
        return None

def extract_traffic_info_from_remarks(remarks: str) -> Dict[str, Any]:
    """
    Извлекает информацию о трафике и сроке действия из строки remarks
    Пример: "RazDvaVPN Vless-test@ya.ru-10.00GB📊-2D,21H⏳"
    
    Args:
        remarks: Строка remarks
        
    Returns:
        Dict[str, Any]: Словарь с информацией о трафике и сроке действия
    """
    result = {}
    
    try:
        # Ищем информацию о трафике (пример: 10.00GB📊)
        traffic_match = re.search(r'(\d+\.\d+)GB', remarks)
        if traffic_match:
            result["traffic_limit_gb"] = float(traffic_match.group(1))
            
        # Ищем информацию о сроке действия (пример: 2D,21H⏳)
        days_match = re.search(r'(\d+)D,(\d+)H', remarks)
        if days_match:
            days = int(days_match.group(1))
            hours = int(days_match.group(2))
            result["days_remaining"] = days
            result["hours_remaining"] = hours
            
        # Пытаемся извлечь email
        email_match = re.search(r'[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+', remarks)
        if email_match:
            result["email"] = email_match.group(0)
            
    except Exception as e:
        logger.error(f"Ошибка при извлечении информации из remarks: {e}")
        
    return result

def parse_subscription_line(line: str, client_id: str) -> Optional[Dict[str, Any]]:
    """
    Разбирает строку подписки для извлечения информации о клиенте
    
    Args:
        line: Строка подписки
        client_id: ID клиента
        
    Returns:
        Optional[Dict[str, Any]]: Информация о клиенте или None
    """
    try:
        # Проверяем, что это строка VLESS
        if line.startswith("vless://"):
            # Убираем префикс протокола
            content = line[8:]
            
            # Разделяем на части: id@server:port?params#remarks
            main_parts = content.split("#", 1)
            connection_part = main_parts[0]
            remarks = main_parts[1] if len(main_parts) > 1 else ""
            
            # Разбираем часть соединения
            conn_parts = connection_part.split("?", 1)
            server_part = conn_parts[0]
            params_part = conn_parts[1] if len(conn_parts) > 1 else ""
            
            # Разбираем server_part: id@server:port
            id_server_parts = server_part.split("@", 1)
            uuid = id_server_parts[0]
            server_port = id_server_parts[1]
            
            # Используем ID из строки, даже если оно отличается от искомого
            if uuid != client_id:
                logger.info(f"ID в строке ({uuid}) отличается от искомого ({client_id}), но мы используем оригинальную ссылку из панели")
            
            # Разбираем сервер и порт
            server_port_parts = server_port.split(":", 1)
            server = server_port_parts[0]
            port = int(server_port_parts[1]) if len(server_port_parts) > 1 else 443
            
            # Разбираем параметры
            params = {}
            if params_part:
                param_pairs = params_part.split("&")
                for pair in param_pairs:
                    if "=" in pair:
                        key, value = pair.split("=", 1)
                        params[key] = value
            
            # Формируем результат
            result = {
                "id": uuid,
                "protocol": "vless",
                "server": server,
                "port": port,
                "remarks": remarks,
                "direct_link": line,
            }
            
            # Добавляем параметры
            for key, value in params.items():
                result[key] = value
            
            return result
        
        # Проверяем, что это строка Shadowsocks
        elif line.startswith("ss://"):
            # Для Shadowsocks пока просто возвращаем строку как direct_link
            return {
                "id": client_id,
                "protocol": "shadowsocks",
                "direct_link": line
            }
        
        else:
            logger.warning(f"Неизвестный формат строки: {line}")
            return None
    
    except Exception as e:
        logger.error(f"Ошибка при разборе строки подписки: {e}")
        return None

def build_vless_link(client_info: Dict[str, Any]) -> str:
    """
    Строит VLESS ссылку из информации о клиенте
    
    Args:
        client_info: Информация о клиенте
        
    Returns:
        str: VLESS ссылка
    """
    try:
        # Обязательные параметры
        uuid = client_info["id"]
        server = client_info["server"]
        port = client_info.get("port", 443)
        
        # Базовая часть ссылки
        link = f"vless://{uuid}@{server}:{port}?"
        
        # Добавляем параметры
        params = []
        
        # Основные параметры
        params.append(f"encryption={client_info.get('encryption', 'none')}")
        
        if "flow" in client_info and client_info["flow"]:
            params.append(f"flow={client_info['flow']}")
        
        if "security" in client_info and client_info["security"]:
            params.append(f"security={client_info['security']}")
        
        if "sni" in client_info and client_info["sni"]:
            params.append(f"sni={client_info['sni']}")
        
        if "fingerprint" in client_info and client_info["fingerprint"]:
            params.append(f"fp={client_info['fingerprint']}")
        
        if "publicKey" in client_info and client_info["publicKey"]:
            params.append(f"pbk={client_info['publicKey']}")
        
        if "shortId" in client_info and client_info["shortId"]:
            params.append(f"sid={client_info['shortId']}")
        
        if "spiderX" in client_info and client_info["spiderX"]:
            params.append(f"spx={client_info['spiderX']}")
        
        if "network" in client_info and client_info["network"]:
            params.append(f"type={client_info['network']}")
        
        if "headerType" in client_info and client_info["headerType"]:
            params.append(f"headerType={client_info['headerType']}")
        
        # Добавляем параметры к ссылке
        link += "&".join(params)
        
        # Добавляем метку, если есть
        if "remarks" in client_info and client_info["remarks"]:
            link += f"#{client_info['remarks']}"
        else:
            link += f"#client_{uuid[:8]}"
        
        return link
    
    except Exception as e:
        logger.error(f"Ошибка при построении VLESS ссылки: {e}")
        return ""