"""
Модуль для работы с API панели 3X-UI с повышенной надежностью и контролем ошибок
"""
import asyncio
import logging
import base64
import json
import os
import time
import random
import re
from typing import Dict, List, Optional, Any, Union

import aiohttp
from yarl import URL

from config import PANEL_API_URL, PANEL_USERNAME, PANEL_PASSWORD

logger = logging.getLogger(__name__)

# Максимальное количество попыток для операций API
MAX_RETRIES = 3
# Задержка между попытками в секундах (с случайной составляющей)
RETRY_DELAY = 2


async def get_auth_cookies() -> Optional[Dict[str, str]]:
    """
    Получение авторизационных cookies для доступа к API панели
    
    Returns:
        Optional[Dict[str, str]]: Словарь с cookie или None при ошибке
    """
    try:
        login_url = f"{PANEL_API_URL}/login"
        
        # Данные для авторизации
        payload = {
            "username": PANEL_USERNAME,
            "password": PANEL_PASSWORD
        }
        
        async with aiohttp.ClientSession() as session:
            async with session.post(login_url, json=payload, timeout=30) as response:
                if response.status == 200:
                    # Извлекаем cookies
                    cookies = {}
                    for cookie in response.cookies.values():
                        cookies[cookie.key] = cookie.value
                    
                    logger.info("Успешная авторизация в панели")
                    return cookies
                else:
                    logger.error(f"Ошибка авторизации в панели: {response.status} - {await response.text()}")
                    return None
    except Exception as e:
        logger.error(f"Исключение при авторизации в панели: {e}")
        return None


async def retry_api_call(api_func, *args, **kwargs):
    """
    Функция для повторных попыток выполнения API запросов при ошибках
    
    Args:
        api_func: Функция API для выполнения
        *args, **kwargs: Аргументы для API функции
        
    Returns:
        Результат выполнения API функции или None при всех неудачных попытках
    """
    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = await api_func(*args, **kwargs)
            if result is not None:
                return result
            
            logger.warning(f"Попытка {attempt}/{MAX_RETRIES} не вернула результатов")
        except Exception as e:
            logger.error(f"Ошибка в попытке {attempt}/{MAX_RETRIES}: {e}")
        
        if attempt < MAX_RETRIES:
            # Добавляем случайную задержку для предотвращения конфликтов
            delay = RETRY_DELAY + random.uniform(0.1, 1.0)
            logger.info(f"Пауза {delay:.2f} секунд перед следующей попыткой")
            await asyncio.sleep(delay)
    
    logger.error(f"Все {MAX_RETRIES} попытки выполнения API запроса завершились неудачно")
    return None


async def get_client_by_id(client_id: str) -> Optional[Dict[str, Any]]:
    """
    Получение информации о клиенте по ID
    
    Args:
        client_id: Идентификатор клиента
        
    Returns:
        Optional[Dict[str, Any]]: Информация о клиенте или None при ошибке
    """
    try:
        # Получаем cookies для авторизации
        cookies = await get_auth_cookies()
        if not cookies:
            logger.error("Не удалось получить cookies для авторизации")
            return None
        
        # Формируем URL для API запроса
        api_url = f"{PANEL_API_URL}/panel/api/inbounds/getClientTraffics/{client_id}"
        
        async with aiohttp.ClientSession(cookies=cookies) as session:
            async with session.get(api_url, timeout=30) as response:
                if response.status == 200:
                    data = await response.json()
                    if data.get("success", False):
                        client_info = data.get("obj", {})
                        logger.info(f"Получена информация о клиенте {client_id}")
                        return client_info
                    else:
                        logger.warning(f"API вернул ошибку: {data.get('msg', 'Unknown error')}")
                        return None
                else:
                    logger.error(f"Ошибка при получении клиента: {response.status} - {await response.text()}")
                    return None
    except Exception as e:
        logger.error(f"Исключение при получении клиента {client_id}: {e}")
        return None


async def get_all_clients() -> Optional[List[Dict[str, Any]]]:
    """
    Получение информации о всех клиентах
    
    Returns:
        Optional[List[Dict[str, Any]]]: Список с информацией о клиентах или None при ошибке
    """
    try:
        # Получаем cookies для авторизации
        cookies = await get_auth_cookies()
        if not cookies:
            logger.error("Не удалось получить cookies для авторизации")
            return None
        
        # Формируем URL для API запроса
        api_url = f"{PANEL_API_URL}/panel/api/inbounds/list"
        
        async with aiohttp.ClientSession(cookies=cookies) as session:
            async with session.get(api_url, timeout=30) as response:
                if response.status == 200:
                    data = await response.json()
                    if data.get("success", False):
                        inbounds = data.get("obj", [])
                        
                        # Собираем клиентов из всех inbounds
                        all_clients = []
                        for inbound in inbounds:
                            settings = inbound.get("settings", "{}")
                            if isinstance(settings, str):
                                try:
                                    settings_json = json.loads(settings)
                                    clients = settings_json.get("clients", [])
                                    
                                    # Добавляем информацию об inbound к каждому клиенту
                                    for client in clients:
                                        client["inbound_id"] = inbound.get("id")
                                        client["protocol"] = inbound.get("protocol")
                                        all_clients.append(client)
                                except json.JSONDecodeError:
                                    logger.error(f"Ошибка при декодировании settings: {settings}")
                                    continue
                        
                        logger.info(f"Получено {len(all_clients)} клиентов из панели")
                        return all_clients
                    else:
                        logger.warning(f"API вернул ошибку: {data.get('msg', 'Unknown error')}")
                        return None
                else:
                    logger.error(f"Ошибка при получении списка inbounds: {response.status} - {await response.text()}")
                    return None
    except Exception as e:
        logger.error(f"Исключение при получении списка клиентов: {e}")
        return None


async def add_client(inbound_id: int, email: str, traffic_limit_gb: int = 0, expire_days: int = 0,
                    protocol: str = "vless") -> Optional[str]:
    """
    Добавление нового клиента в панель
    
    Args:
        inbound_id: ID inbound
        email: Email клиента (используется как идентификатор)
        traffic_limit_gb: Лимит трафика в ГБ (0 = безлимитно)
        expire_days: Срок действия в днях (0 = безлимитно)
        protocol: Протокол (vless, vmess, trojan)
        
    Returns:
        Optional[str]: ID созданного клиента или None при ошибке
    """
    try:
        # Получаем cookies для авторизации
        cookies = await get_auth_cookies()
        if not cookies:
            logger.error("Не удалось получить cookies для авторизации")
            return None
        
        # Формируем URL для API запроса
        api_url = f"{PANEL_API_URL}/panel/api/inbounds/addClient"
        
        # Генерируем клиента в зависимости от протокола
        client_id = base64.urlsafe_b64encode(os.urandom(16)).decode('utf-8').rstrip('=')
        
        # Общие параметры клиента
        client_data = {
            "id": client_id,
            "email": email,
            "limitIp": 0,
            "totalGB": traffic_limit_gb * 1024 * 1024 * 1024 if traffic_limit_gb > 0 else 0,
            "expiryTime": int(time.time() + expire_days * 86400) * 1000 if expire_days > 0 else 0,
            "enable": True,
            "tgId": "",
            "subId": ""
        }
        
        # Специфичные параметры для разных протоколов
        if protocol.lower() == "vless":
            client_data["flow"] = "xtls-rprx-vision"
        
        # Добавляем клиента
        payload = {
            "id": inbound_id,
            "settings": json.dumps(client_data)
        }
        
        async with aiohttp.ClientSession(cookies=cookies) as session:
            async with session.post(api_url, json=payload, timeout=30) as response:
                if response.status == 200:
                    data = await response.json()
                    if data.get("success", False):
                        logger.info(f"Клиент {email} успешно добавлен, ID: {client_id}")
                        return client_id
                    else:
                        logger.warning(f"API вернул ошибку: {data.get('msg', 'Unknown error')}")
                        return None
                else:
                    logger.error(f"Ошибка при добавлении клиента: {response.status} - {await response.text()}")
                    return None
    except Exception as e:
        logger.error(f"Исключение при добавлении клиента {email}: {e}")
        return None


async def update_client(inbound_id: int, client_id: str, traffic_limit_gb: int = 0, expire_days: int = 0,
                       enable: bool = True) -> bool:
    """
    Обновление параметров клиента
    
    Args:
        inbound_id: ID inbound
        client_id: ID клиента
        traffic_limit_gb: Новый лимит трафика в ГБ (0 = безлимитно)
        expire_days: Новый срок действия в днях (0 = не менять)
        enable: Активировать или деактивировать клиента
        
    Returns:
        bool: True если обновление успешно, иначе False
    """
    try:
        # Получаем cookies для авторизации
        cookies = await get_auth_cookies()
        if not cookies:
            logger.error("Не удалось получить cookies для авторизации")
            return False
        
        # Получаем текущие данные клиента
        client_info = await get_client_by_id(client_id)
        if not client_info:
            logger.error(f"Не удалось получить информацию о клиенте {client_id}")
            return False
        
        # Формируем URL для API запроса
        api_url = f"{PANEL_API_URL}/panel/api/inbounds/updateClient/{inbound_id}/{client_id}"
        
        # Обновляем параметры клиента
        update_data = {
            "id": client_id,
            "email": client_info.get("email", ""),
            "limitIp": client_info.get("limitIp", 0),
            "totalGB": traffic_limit_gb * 1024 * 1024 * 1024 if traffic_limit_gb > 0 else client_info.get("totalGB", 0),
            "enable": enable,
            "tgId": client_info.get("tgId", ""),
            "subId": client_info.get("subId", "")
        }
        
        # Если указан срок действия, обновляем его
        if expire_days > 0:
            update_data["expiryTime"] = int(time.time() + expire_days * 86400) * 1000
        else:
            update_data["expiryTime"] = client_info.get("expiryTime", 0)
        
        # Отправляем запрос на обновление
        async with aiohttp.ClientSession(cookies=cookies) as session:
            async with session.post(api_url, json=update_data, timeout=30) as response:
                if response.status == 200:
                    data = await response.json()
                    if data.get("success", False):
                        logger.info(f"Клиент {client_id} успешно обновлен")
                        return True
                    else:
                        logger.warning(f"API вернул ошибку: {data.get('msg', 'Unknown error')}")
                        return False
                else:
                    logger.error(f"Ошибка при обновлении клиента: {response.status} - {await response.text()}")
                    return False
    except Exception as e:
        logger.error(f"Исключение при обновлении клиента {client_id}: {e}")
        return False


async def delete_client(inbound_id: int, client_id: str) -> bool:
    """
    Удаление клиента из панели
    
    Args:
        inbound_id: ID inbound
        client_id: ID клиента
        
    Returns:
        bool: True если удаление успешно, иначе False
    """
    try:
        # Получаем cookies для авторизации
        cookies = await get_auth_cookies()
        if not cookies:
            logger.error("Не удалось получить cookies для авторизации")
            return False
        
        # Формируем URL для API запроса
        api_url = f"{PANEL_API_URL}/panel/api/inbounds/{inbound_id}/delClient/{client_id}"
        
        # Отправляем запрос на удаление
        async with aiohttp.ClientSession(cookies=cookies) as session:
            async with session.post(api_url, timeout=30) as response:
                if response.status == 200:
                    data = await response.json()
                    if data.get("success", False):
                        logger.info(f"Клиент {client_id} успешно удален")
                        return True
                    else:
                        logger.warning(f"API вернул ошибку: {data.get('msg', 'Unknown error')}")
                        return False
                else:
                    logger.error(f"Ошибка при удалении клиента: {response.status} - {await response.text()}")
                    return False
    except Exception as e:
        logger.error(f"Исключение при удалении клиента {client_id}: {e}")
        return False


# Функции с автоматическими повторными попытками (НЕ РЕКУРСИВНЫЕ)
async def get_client_by_id_with_retry(client_id: str) -> Optional[Dict[str, Any]]:
    """Получение информации о клиенте с повторными попытками"""
    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = await get_client_by_id(client_id)
            if result is not None:
                return result
            logger.warning(f"Попытка {attempt}/{MAX_RETRIES} не вернула результатов")
        except Exception as e:
            logger.error(f"Ошибка в попытке {attempt}/{MAX_RETRIES}: {e}")
        
        if attempt < MAX_RETRIES:
            delay = RETRY_DELAY + random.uniform(0.1, 1.0)
            logger.info(f"Пауза {delay:.2f} секунд перед следующей попыткой")
            await asyncio.sleep(delay)
    
    logger.error(f"Все {MAX_RETRIES} попытки получения клиента {client_id} завершились неудачно")
    return None

async def get_all_clients_with_retry() -> Optional[List[Dict[str, Any]]]:
    """Получение всех клиентов с повторными попытками"""
    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = await get_all_clients()
            if result is not None:
                return result
            logger.warning(f"Попытка {attempt}/{MAX_RETRIES} не вернула результатов")
        except Exception as e:
            logger.error(f"Ошибка в попытке {attempt}/{MAX_RETRIES}: {e}")
        
        if attempt < MAX_RETRIES:
            delay = RETRY_DELAY + random.uniform(0.1, 1.0)
            logger.info(f"Пауза {delay:.2f} секунд перед следующей попыткой")
            await asyncio.sleep(delay)
    
    logger.error(f"Все {MAX_RETRIES} попытки получения списка клиентов завершились неудачно")
    return None

async def add_client_with_retry(inbound_id: int, email: str, traffic_limit_gb: int = 0, expire_days: int = 0,
                               protocol: str = "vless") -> Optional[str]:
    """Добавление клиента с повторными попытками"""
    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = await add_client(inbound_id, email, traffic_limit_gb, expire_days, protocol)
            if result is not None:
                return result
            logger.warning(f"Попытка {attempt}/{MAX_RETRIES} не вернула результатов")
        except Exception as e:
            logger.error(f"Ошибка в попытке {attempt}/{MAX_RETRIES}: {e}")
        
        if attempt < MAX_RETRIES:
            delay = RETRY_DELAY + random.uniform(0.1, 1.0)
            logger.info(f"Пауза {delay:.2f} секунд перед следующей попыткой")
            await asyncio.sleep(delay)
    
    logger.error(f"Все {MAX_RETRIES} попытки добавления клиента {email} завершились неудачно")
    return None

async def update_client_with_retry(inbound_id: int, client_id: str, traffic_limit_gb: int = 0, expire_days: int = 0,
                                  enable: bool = True) -> bool:
    """Обновление клиента с повторными попытками"""
    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = await update_client(inbound_id, client_id, traffic_limit_gb, expire_days, enable)
            if result is not None:
                return bool(result)
            logger.warning(f"Попытка {attempt}/{MAX_RETRIES} не вернула результатов")
        except Exception as e:
            logger.error(f"Ошибка в попытке {attempt}/{MAX_RETRIES}: {e}")
        
        if attempt < MAX_RETRIES:
            delay = RETRY_DELAY + random.uniform(0.1, 1.0)
            logger.info(f"Пауза {delay:.2f} секунд перед следующей попыткой")
            await asyncio.sleep(delay)
    
    logger.error(f"Все {MAX_RETRIES} попытки обновления клиента {client_id} завершились неудачно")
    return False

async def delete_client_with_retry(inbound_id: int, client_id: str) -> bool:
    """Удаление клиента с повторными попытками"""
    for attempt in range(1, MAX_RETRIES + 1):
        try:
            result = await delete_client(inbound_id, client_id)
            if result is not None:
                return bool(result)
            logger.warning(f"Попытка {attempt}/{MAX_RETRIES} не вернула результатов")
        except Exception as e:
            logger.error(f"Ошибка в попытке {attempt}/{MAX_RETRIES}: {e}")
        
        if attempt < MAX_RETRIES:
            delay = RETRY_DELAY + random.uniform(0.1, 1.0)
            logger.info(f"Пауза {delay:.2f} секунд перед следующей попыткой")
            await asyncio.sleep(delay)
    
    logger.error(f"Все {MAX_RETRIES} попытки удаления клиента {client_id} завершились неудачно")
    return False


# Экспортируем публичные функции с более простыми именами для использования в других модулях
get_client_by_id = get_client_by_id_with_retry
get_all_clients = get_all_clients_with_retry
add_client = add_client_with_retry
update_client = update_client_with_retry
delete_client = delete_client_with_retry