"""
Менеджер сессий для панели 3X-UI с автоматическим обновлением cookie
"""
import asyncio
import logging
import time
import aiohttp
import ssl
from typing import Optional, Dict, Any
from datetime import datetime, timedelta

logger = logging.getLogger(__name__)

class PanelSessionManager:
    """Управляет сессиями и cookie для панели 3X-UI"""
    
    def __init__(self):
        import os
        # Получаем URL из переменных окружения и правильно формируем базовый URL
        panel_url_env = os.environ.get('PANEL_URL', 'https://31.172.75.92:47773/Sh1YLmgVQZ7cgcS/panel/')
        # Убираем /panel/ из конца для получения базового URL
        self.base_url = panel_url_env.rstrip('/').removesuffix('/panel')
        self.username = os.environ.get('PANEL_USERNAME', 'VVShilin')
        self.password = os.environ.get('PANEL_PASSWORD', 'Kat@11122012')
        
        self._session: Optional[aiohttp.ClientSession] = None
        self._auth_cookie: Optional[str] = None
        self._last_login_time = 0
        self._login_lifetime = 3600  # Cookie живет 1 час
        self._login_lock = asyncio.Lock()
        
        # SSL контекст для самоподписанных сертификатов
        self._ssl_context = ssl.create_default_context()
        self._ssl_context.check_hostname = False
        self._ssl_context.verify_mode = ssl.CERT_NONE
        
    async def _create_session(self) -> aiohttp.ClientSession:
        """Создает новую HTTP сессию"""
        connector = aiohttp.TCPConnector(ssl=self._ssl_context)
        return aiohttp.ClientSession(connector=connector)
    
    async def _login(self) -> bool:
        """Выполняет вход в панель согласно PANEL_API_INSTRUCTIONS.md"""
        try:
            if self._session:
                await self._session.close()
            
            self._session = await self._create_session()
            
            # Используем правильный формат согласно инструкции
            login_url = f"{self.base_url}/login"  # БЕЗ /panel/
            credentials = {
                "username": self.username,
                "password": self.password
            }
            
            headers = {
                "Content-Type": "application/x-www-form-urlencoded"
            }
            
            logger.info(f"Авторизация в панели: {login_url}")
            
            async with self._session.post(login_url, data=credentials, headers=headers) as response:
                if response.status == 200:
                    try:
                        response_data = await response.json()
                        
                        if response_data.get("success"):
                            logger.info("✅ Авторизация успешна")
                            
                            # Извлекаем cookie '3x-ui' из заголовков согласно инструкции
                            import re
                            set_cookie_headers = response.headers.getall('Set-Cookie', [])
                            cookie_3x_ui = None
                            
                            for cookie_header in set_cookie_headers:
                                if '3x-ui=' in cookie_header:
                                    match = re.search(r'3x-ui=([^;]+)', cookie_header)
                                    if match:
                                        cookie_3x_ui = match.group(1)
                                        break
                            
                            if cookie_3x_ui:
                                # Сохраняем cookie для последующих запросов
                                self._auth_cookie = f"3x-ui={cookie_3x_ui}"
                                self._last_login_time = time.time()
                                logger.info("🔐 Cookie '3x-ui' успешно получен")
                                return True
                            else:
                                logger.error("❌ Cookie '3x-ui' не найден в заголовках")
                                return False
                        else:
                            error_msg = response_data.get('msg', 'Неизвестная ошибка')
                            logger.error(f"❌ Ошибка авторизации: {error_msg}")
                            return False
                            
                    except Exception as e:
                        response_text = await response.text()
                        logger.error(f"❌ Ошибка парсинга ответа: {e}")
                        logger.debug(f"Ответ сервера: {response_text[:300]}...")
                        return False
                else:
                    logger.error(f"❌ HTTP ошибка авторизации: {response.status}")
                    return False
                
        except Exception as e:
            logger.error(f"❌ Критическая ошибка авторизации: {e}")
            return False
    

    
    async def ensure_authenticated(self) -> bool:
        """Гарантирует наличие активной аутентифицированной сессии"""
        async with self._login_lock:
            current_time = time.time()
            
            # Проверяем нужно ли обновить сессию
            if (not self._session or 
                not self._last_login_time or 
                current_time - self._last_login_time > self._login_lifetime):
                
                logger.info("Сессия истекла или отсутствует, выполняем повторный вход...")
                return await self._login()
            
            return True
    
    async def make_authenticated_request(self, method: str, endpoint: str, **kwargs) -> Optional[Dict[str, Any]]:
        """Выполняет аутентифицированный запрос к панели с автоматическим retry при истечении cookie"""
        max_attempts = 2
        
        for attempt in range(max_attempts):
            if not await self.ensure_authenticated():
                logger.error("Не удалось получить аутентифицированную сессию")
                return None
            
            # Формируем URL для API запросов (с /panel/api/)
            url = f"{self.base_url}/panel/api/{endpoint.lstrip('/')}"
            
            try:
                if not self._session:
                    logger.error("Сессия не инициализирована")
                    return None
                
                # Добавляем cookie в заголовки
                headers = kwargs.pop('headers', {})
                if self._auth_cookie:
                    headers['Cookie'] = self._auth_cookie
                
                logger.debug(f"API запрос: {method} {url}")
                
                async with self._session.request(method, url, headers=headers, **kwargs) as response:
                    response_text = await response.text()
                    
                    # Проверяем, вернулся ли HTML вместо JSON (признак истекшего cookie)
                    if response_text.strip().startswith('<') or '<html' in response_text.lower():
                        logger.warning(f"Получен HTML вместо JSON (попытка {attempt + 1})")
                        
                        if attempt < max_attempts - 1:
                            logger.info("Принудительно обновляем авторизацию и повторяем запрос")
                            # Сбрасываем время для принудительной переавторизации
                            self._last_login_time = 0
                            self._auth_cookie = None
                            continue
                        else:
                            logger.error("Cookie истек, не удалось получить JSON ответ")
                            return None
                    
                    if response.status != 200:
                        logger.error(f"Ошибка HTTP: {response.status} для {endpoint}")
                        return None
                    
                    # Пытаемся распарсить JSON
                    try:
                        import json
                        return json.loads(response_text)
                    except json.JSONDecodeError:
                        logger.error(f"Не удалось распарсить ответ как JSON: {response_text[:200]}...")
                        return None
                        
            except Exception as e:
                logger.error(f"Ошибка запроса к {endpoint}: {e}")
                if attempt < max_attempts - 1:
                    await asyncio.sleep(1)
                    continue
                else:
                    return None
        
        return None
    
    async def get_client_traffic(self, email: str) -> Optional[Dict[str, Any]]:
        """Получает данные о трафике клиента по email"""
        endpoint = f"/api/inbounds/getClientTraffics/{email}"
        
        response_data = await self.make_authenticated_request("GET", endpoint)
        
        if not response_data:
            return None
        
        if not response_data.get("success"):
            logger.error(f"API ошибка для {email}: {response_data.get('msg', 'Unknown')}")
            return None
        
        obj = response_data.get("obj")
        if not obj:
            logger.warning(f"Нет данных о трафике для {email}")
            return None
        
        # Возвращаем данные в нужном формате
        return {
            "up": obj.get("up", 0),
            "down": obj.get("down", 0),
            "traffic_used": obj.get("up", 0) + obj.get("down", 0)
        }
    
    async def close(self):
        """Закрывает сессию"""
        if self._session:
            await self._session.close()
            self._session = None
            self._auth_cookie = None
            self._last_login_time = 0

# Глобальный экземпляр менеджера сессий
panel_session_manager = PanelSessionManager()