"""
Модуль для отправки уведомлений пользователям бота
"""
import logging
import asyncio
from datetime import datetime, timedelta, timezone
from typing import List, Tuple, Dict, Optional

from aiogram import Bot
from aiogram.types import InlineKeyboardMarkup, InlineKeyboardButton

from bot.database.models import User, Subscription
from bot.database.operations import (
    get_expiring_subscriptions,
    get_low_traffic_subscriptions
)

logger = logging.getLogger(__name__)


async def was_notification_recently_sent(user_id: int, subscription_id: int, notification_type: str, days_before_expiry: int = None, hours: int = 4) -> bool:
    """
    Проверяет, было ли уже отправлено уведомление за последние N часов
    
    Args:
        user_id: ID пользователя
        subscription_id: ID подписки
        notification_type: Тип уведомления
        days_before_expiry: Количество дней до истечения
        hours: Количество часов для проверки (по умолчанию 4)
    
    Returns:
        bool: True если уведомление уже отправлялось недавно
    """
    try:
        import aiosqlite
        from config import DATABASE_FILE
        from datetime import datetime, timedelta
        
        # Время N часов назад
        cutoff_time = (datetime.now() - timedelta(hours=hours)).isoformat()
        
        async with aiosqlite.connect(DATABASE_FILE) as db:
            query = """
                SELECT COUNT(*) FROM notification_log 
                WHERE user_id = ? AND subscription_id = ? AND notification_type = ? 
                AND sent_at > ? AND success = 1
            """
            params = [user_id, subscription_id, notification_type, cutoff_time]
            
            # Добавляем проверку по дням если указано
            if days_before_expiry is not None:
                query += " AND days_before_expiry = ?"
                params.append(days_before_expiry)
            
            async with db.execute(query, params) as cursor:
                result = await cursor.fetchone()
                return result[0] > 0 if result else False
                
    except Exception as e:
        logger.error(f"Ошибка при проверке недавних уведомлений: {e}")
        return False


async def is_notification_time() -> bool:
    """
    Проверяет, подходящее ли время для отправки уведомлений на основе настроек из базы данных
    
    Returns:
        bool: True если время подходящее, False иначе
    """
    try:
        import aiosqlite
        from config import DATABASE_FILE
        
        # Московское время (UTC+3)
        moscow_tz = timezone(timedelta(hours=3))
        moscow_time = datetime.now(moscow_tz)
        current_hour = moscow_time.hour
        
        # Получаем настройки времени из базы данных
        async with aiosqlite.connect(DATABASE_FILE) as db:
            async with db.execute(
                "SELECT send_time_start, send_time_end FROM notification_settings WHERE enabled = 1 LIMIT 1"
            ) as cursor:
                row = await cursor.fetchone()
                
                if row:
                    start_hour, end_hour = row
                    is_time_ok = start_hour <= current_hour <= end_hour
                    logger.info(f"ПРОВЕРКА ВРЕМЕНИ: БД настройки {start_hour}:00-{end_hour}:00, текущее: {current_hour}:00 MSK, результат: {is_time_ok}")
                    return is_time_ok
                else:
                    # Если настроек нет, используем по умолчанию
                    is_time_ok = NOTIFICATION_START_HOUR <= current_hour <= NOTIFICATION_END_HOUR
                    logger.warning(f"ПРОВЕРКА ВРЕМЕНИ: Настройки не найдены, время по умолчанию {NOTIFICATION_START_HOUR}:00-{NOTIFICATION_END_HOUR}:00, текущее: {current_hour}:00 MSK, результат: {is_time_ok}")
                    return is_time_ok
                    
    except Exception as e:
        logger.error(f"Ошибка при получении настроек времени уведомлений: {e}")
        # В случае ошибки используем время по умолчанию
        moscow_tz = timezone(timedelta(hours=3))
        moscow_time = datetime.now(moscow_tz)
        return NOTIFICATION_START_HOUR <= moscow_time.hour <= NOTIFICATION_END_HOUR


# Пороги для уведомлений
TRIAL_EXPIRY_DAYS = 1  # За сколько дней уведомлять о истечении тестового периода
REGULAR_EXPIRY_DAYS = 1  # За день до истечения обычной подписки
LOW_TRAFFIC_THRESHOLD = 1.0  # 1 ГБ остатка трафика для уведомления (в ГБ)

# Настройки времени уведомлений
NOTIFICATION_START_HOUR = 9   # Начало периода уведомлений (9:00)
NOTIFICATION_END_HOUR = 21    # Конец периода уведомлений (21:00)

# Интервал уведомлений для тестового периода (каждый час)
TRIAL_NOTIFICATION_INTERVAL_HOURS = 1

# Тарифы на дополнительный трафик
TRAFFIC_PACKAGES = {
    '10g': {'name': '10 ГБ', 'price': 150, 'size': 10},
    '25g': {'name': '25 ГБ', 'price': 350, 'size': 25},
    '50g': {'name': '50 ГБ', 'price': 600, 'size': 50},
    '100g': {'name': '100 ГБ', 'price': 1000, 'size': 100},
}


async def check_and_notify_expiring_subscriptions(bot: Bot) -> None:
    """
    Проверяет подписки, срок действия которых истекает, и отправляет уведомления
    
    Args:
        bot: Экземпляр бота для отправки сообщений
    """
    logger.info("Проверка истекающих подписок...")
    
    # Получаем тестовые подписки, истекающие через 1 день
    try:
        trial_expiring = await get_expiring_subscriptions(TRIAL_EXPIRY_DAYS)
        logger.info(f"Найдено {len(trial_expiring)} истекающих тестовых подписок")
        
        # Отправляем уведомления о истечении тестового периода
        for user, subscription in trial_expiring:
            # Проверяем, является ли подписка тестовой (без plan_id)
            if not subscription.plan_id:
                await notify_trial_expiring(bot, user, subscription)
                await asyncio.sleep(0.3)  # Небольшая задержка между отправками сообщений
    except Exception as e:
        logger.error(f"Ошибка при обработке истекающих тестовых подписок: {e}")
    
    # Получаем обычные подписки, истекающие через 2 дня
    try:
        regular_expiring = await get_expiring_subscriptions(REGULAR_EXPIRY_DAYS)
        logger.info(f"Найдено {len(regular_expiring)} истекающих обычных подписок")
        
        # Отправляем уведомления о истечении обычной подписки (только в рабочее время 9-21)
        if await is_notification_time():
            for user, subscription in regular_expiring:
                # Проверяем, что это не тестовая подписка (имеет plan_id)
                if subscription.plan_id:
                    await notify_subscription_expiring(bot, user, subscription)
                    await asyncio.sleep(0.3)  # Небольшая задержка между отправками сообщений
        else:
            logger.info(f"Уведомления о истечении подписок отложены до рабочего времени (9:00-21:00). Текущее время: {datetime.now().hour}:00")
    except Exception as e:
        logger.error(f"Ошибка при обработке истекающих обычных подписок: {e}")


async def check_and_notify_low_traffic(bot: Bot) -> None:
    """
    Проверяет подписки с низким остатком трафика и отправляет уведомления
    
    Args:
        bot: Экземпляр бота для отправки сообщений
    """
    logger.info("Проверка подписок с низким остатком трафика...")
    
    try:
        # Получаем подписки с низким остатком трафика
        low_traffic = await get_low_traffic_subscriptions(LOW_TRAFFIC_THRESHOLD)
        logger.info(f"Найдено {len(low_traffic)} подписок с низким остатком трафика")
        
        # Отправляем уведомления о низком остатке трафика
        for user, subscription in low_traffic:
            await notify_low_traffic(bot, user, subscription)
            await asyncio.sleep(0.3)  # Небольшая задержка между отправками сообщений
    except Exception as e:
        logger.error(f"Ошибка при обработке подписок с низким остатком трафика: {e}")


async def notify_trial_expiring(bot: Bot, user: User, subscription: Subscription) -> None:
    """
    Отправляет уведомление о скором истечении тестового периода
    
    Args:
        bot: Экземпляр бота для отправки сообщений
        user: Пользователь, которому отправляется уведомление
        subscription: Подписка, срок действия которой истекает
    """
    message_text = (
        f"⚠️ <b>Ваш тестовый период скоро закончится</b>\n\n"
        f"Здравствуйте, {user.full_name}!\n\n"
        f"Напоминаем, что ваш тестовый период VPN заканчивается через {TRIAL_EXPIRY_DAYS} день.\n\n"
        f"📅 Дата окончания: <b>{subscription.expires_at.strftime('%d.%m.%Y')}</b>\n"
        f"📊 Использовано трафика: <b>{subscription.traffic_percent_used:.1f}%</b>\n\n"
        f"Чтобы продолжить пользоваться VPN без перерыва, оформите подписку прямо сейчас."
    )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message_text
        )
        
        # Записываем уведомление в лог
        import aiosqlite
        from config import DATABASE_FILE
        from datetime import datetime
        
        async with aiosqlite.connect(DATABASE_FILE) as db:
            await db.execute("""
                INSERT INTO notification_log 
                (user_id, subscription_id, notification_type, days_before_expiry, message_text, sent_at, success)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """, (user.id, subscription.id, "expiry", TRIAL_EXPIRY_DAYS, message_text, datetime.now().isoformat(), True))
            await db.commit()
        
        logger.info(f"Отправлено уведомление об истечении тестового периода пользователю {user.id}")
    except Exception as e:
        # Записываем ошибку в лог
        try:
            import aiosqlite
            from config import DATABASE_FILE
            from datetime import datetime
            
            async with aiosqlite.connect(DATABASE_FILE) as db:
                await db.execute("""
                    INSERT INTO notification_log 
                    (user_id, subscription_id, notification_type, days_before_expiry, message_text, sent_at, success, error_message)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                """, (user.id, subscription.id, "expiry", TRIAL_EXPIRY_DAYS, message_text, datetime.now().isoformat(), False, str(e)))
                await db.commit()
        except:
            pass
        
        logger.error(f"Ошибка при отправке уведомления пользователю {user.id}: {e}")


async def notify_subscription_expiring(bot: Bot, user: User, subscription: Subscription) -> None:
    """
    Отправляет уведомление о скором истечении обычной подписки
    
    Args:
        bot: Экземпляр бота для отправки сообщений
        user: Пользователь, которому отправляется уведомление
        subscription: Подписка, срок действия которой истекает
    """
    message_text = (
        f"⚠️ <b>Ваша подписка скоро закончится</b>\n\n"
        f"Здравствуйте, {user.full_name}!\n\n"
        f"Напоминаем, что срок действия вашей подписки VPN заканчивается через {REGULAR_EXPIRY_DAYS} дня.\n\n"
        f"📅 Дата окончания: <b>{subscription.expires_at.strftime('%d.%m.%Y')}</b>\n"
        f"📊 Использовано трафика: <b>{subscription.traffic_percent_used:.1f}%</b>\n\n"
        f"Чтобы продолжить пользоваться VPN без перерыва, продлите подписку прямо сейчас."
    )
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message_text
        )
        
        # Записываем уведомление в лог
        import aiosqlite
        from config import DATABASE_FILE
        from datetime import datetime
        
        async with aiosqlite.connect(DATABASE_FILE) as db:
            await db.execute("""
                INSERT INTO notification_log 
                (user_id, subscription_id, notification_type, days_before_expiry, message_text, sent_at, success)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            """, (user.id, subscription.id, "expiry", REGULAR_EXPIRY_DAYS, message_text, datetime.now().isoformat(), True))
            await db.commit()
        
        logger.info(f"Отправлено уведомление об истечении подписки пользователю {user.id}")
    except Exception as e:
        # Записываем ошибку в лог
        try:
            import aiosqlite
            from config import DATABASE_FILE
            from datetime import datetime
            
            async with aiosqlite.connect(DATABASE_FILE) as db:
                await db.execute("""
                    INSERT INTO notification_log 
                    (user_id, subscription_id, notification_type, days_before_expiry, message_text, sent_at, success, error_message)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                """, (user.id, subscription.id, "expiry", REGULAR_EXPIRY_DAYS, message_text, datetime.now().isoformat(), False, str(e)))
                await db.commit()
        except:
            pass
        
        logger.error(f"Ошибка при отправке уведомления пользователю {user.id}: {e}")


async def notify_low_traffic(bot: Bot, user: User, subscription: Subscription) -> None:
    """
    Отправляет уведомление о низком остатке трафика
    
    Args:
        bot: Экземпляр бота для отправки сообщений
        user: Пользователь, которому отправляется уведомление
        subscription: Подписка с низким остатком трафика
    """
    # Проверяем, является ли это тестовой подпиской
    is_trial = subscription.plan_id == "trial"
    
    # Вычисляем оставшийся трафик в ГБ
    remaining_gb = subscription.traffic_remaining / (1024 * 1024 * 1024)
    
    if is_trial:
        # Для тестового периода показываем оставшиеся дни
        from datetime import datetime
        days_remaining = subscription.days_remaining
        
        if remaining_gb <= 0:
            # Трафик исчерпан в тестовом периоде
            message_text = (
                f"⚠️ <b>Трафик тестового периода исчерпан</b>\n\n"
                f"Здравствуйте, {user.full_name}!\n\n"
                f"Ваш тестовый трафик (15 ГБ) полностью использован.\n\n"
                f"📅 Тестовый период действует еще <b>{days_remaining} {'день' if days_remaining == 1 else 'дня' if days_remaining < 5 else 'дней'}</b>\n\n"
                f"Для продолжения использования VPN рекомендуем оформить полную подписку:"
            )
        else:
            # Остался трафик, но мало
            message_text = (
                f"⚠️ <b>Тестовый период подходит к концу</b>\n\n"
                f"Здравствуйте, {user.full_name}!\n\n"
                f"📊 Осталось трафика: <b>{remaining_gb:.1f} ГБ</b>\n"
                f"📅 Тестовый период: еще <b>{days_remaining} {'день' if days_remaining == 1 else 'дня' if days_remaining < 5 else 'дней'}</b>\n\n"
                f"Рекомендуем оформить полную подписку для непрерывного доступа:"
            )
        
        # Клавиатура с тарифами для тестового периода
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [
                InlineKeyboardButton(text="💼 На Недельку (75₽)", callback_data="subscription:weekly")
            ],
            [
                InlineKeyboardButton(text="📊 Базовый (150₽)", callback_data="subscription:basic")
            ],
            [
                InlineKeyboardButton(text="⭐ Стандарт (425₽)", callback_data="subscription:standard")
            ],
            [
                InlineKeyboardButton(text="💎 Премиум (810₽)", callback_data="subscription:premium")
            ]
        ])
    else:
        # Для обычных подписок показываем как раньше
        message_text = (
            f"⚠️ <b>У вас заканчивается трафик VPN</b>\n\n"
            f"Здравствуйте, {user.full_name}!\n\n"
            f"Напоминаем, что ваш трафик VPN подходит к концу.\n\n"
            f"📊 Осталось: <b>{remaining_gb:.2f} ГБ</b> (примерно {int(remaining_gb)} часов видео)\n"
            f"📅 Дата окончания подписки: <b>{subscription.expires_at.strftime('%d.%m.%Y') if subscription.expires_at else 'Не указана'}</b>\n\n"
            f"Вы можете приобрести дополнительный трафик:"
        )
        
        # Клавиатура с дополнительным трафиком для обычных подписок
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [
                InlineKeyboardButton(text="10 ГБ (150₽)", callback_data="buy_traffic:10g"),
                InlineKeyboardButton(text="25 ГБ (350₽)", callback_data="buy_traffic:25g")
            ],
            [
                InlineKeyboardButton(text="50 ГБ (600₽)", callback_data="buy_traffic:50g"),
                InlineKeyboardButton(text="100 ГБ (1000₽)", callback_data="buy_traffic:100g")
            ]
        ])
    
    try:
        await bot.send_message(
            chat_id=user.id,
            text=message_text,
            reply_markup=keyboard
        )
        logger.info(f"Отправлено уведомление о низком остатке трафика пользователю {user.id}")
    except Exception as e:
        logger.error(f"Ошибка при отправке уведомления пользователю {user.id}: {e}")


async def run_notification_service(bot: Bot) -> None:
    """
    Запускает службу уведомлений
    
    Args:
        bot: Экземпляр бота для отправки сообщений
    """
    logger.info("Запуск службы уведомлений...")
    
    while True:
        try:
            # Проверяем и уведомляем о истекающих подписках
            await check_and_notify_expiring_subscriptions(bot)
            
            # Проверяем и уведомляем о низком остатке трафика
            await check_and_notify_low_traffic(bot)
            
            # Ждем до следующей проверки (1 раз в день)
            logger.info("Проверка завершена, следующая через 24 часа")
            await asyncio.sleep(24 * 60 * 60)  # 24 часа
            
        except Exception as e:
            logger.error(f"Ошибка в службе уведомлений: {e}")
            await asyncio.sleep(60 * 60)  # Повторяем через час в случае ошибки