"""
Нормализованный создатель клиентов с правильным получением UUID
"""
import logging
import json
import uuid
from datetime import datetime, timedelta
from typing import Optional, Tuple
from bot.services.working_panel_api import WorkingPanelAPI
from config_keys import get_panel_credentials

logger = logging.getLogger(__name__)

async def create_normalized_client(user_id: int, traffic_gb: int = 15, days: int = 30, subscription_id: int = None) -> Tuple[Optional[str], Optional[str]]:
    """
    Создает клиента в панели и сразу получает правильный UUID
    
    Args:
        user_id: ID пользователя Telegram
        traffic_gb: Лимит трафика в ГБ
        days: Срок действия в днях
        
    Returns:
        Tuple[Optional[str], Optional[str]]: (реальный_UUID, VPN_ключ) или (None, None)
    """
    try:
        # Защита от использования ID бота (8071791443)
        if user_id == 8071791443:
            logger.error("ОШИБКА: в create_normalized_client передан ID бота!")
            return None, None
        
        # Получаем username пользователя из базы данных
        from bot.database.operations import get_user
        user = await get_user(user_id)
        username = user.username if user and user.username else "user"
        
        # Новая схема email: ID + Username без домена
        email = f"{user_id}@{username}"
        
        # Инициализируем API панели
        panel_url, panel_username, panel_password = get_panel_credentials()
        panel_api = WorkingPanelAPI()
        panel_api.base_url = panel_url
        panel_api.username = panel_username
        panel_api.password = panel_password
        
        # Авторизуемся
        if not await panel_api.login():
            logger.error("Не удалось авторизоваться в панели для создания клиента")
            return None, None
        
        # Проверяем, не существует ли уже клиент с таким email
        existing_client = await find_client_by_email(panel_api, email)
        if existing_client:
            logger.info(f"Клиент с email {email} уже существует, используем существующий")
            vpn_key = await get_client_vpn_key(panel_api, existing_client['id'])
            return existing_client['id'], vpn_key
        
        # Генерируем UUID для создания (3X-UI требует UUID в определенном формате)
        client_uuid = str(uuid.uuid4())
        
        # Вычисляем параметры
        expiry_time = int((datetime.now() + timedelta(days=days)).timestamp() * 1000)
        traffic_bytes = traffic_gb * 1024 * 1024 * 1024
        
        # Создаем клиента
        success = await create_client_in_panel(panel_api, client_uuid, email, traffic_bytes, expiry_time, user_id, subscription_id)
        
        if success:
            # Получаем VPN ключ через subscription URL (надежный метод)
            vpn_key = await get_client_vpn_key(panel_api, client_uuid)
            
            logger.info(f"Клиент успешно создан: {email}, UUID: {client_uuid}")
            return client_uuid, vpn_key
        else:
            logger.error(f"Не удалось создать клиента в панели для пользователя {user_id}")
            return None, None
            
    except Exception as e:
        logger.error(f"Ошибка при создании нормализованного клиента: {e}")
        return None, None
    
    finally:
        try:
            await panel_api.close()
        except:
            pass


async def find_client_by_email(panel_api, email: str) -> Optional[dict]:
    """
    Находит клиента в панели по email
    
    Args:
        panel_api: Не используется, оставлен для совместимости
        email: Email для поиска
        
    Returns:
        dict или None: Данные клиента, если найден
    """
    try:
        from bot.services.panel_session_manager import PanelSessionManager
        panel_session_manager = PanelSessionManager()
        
        # Получаем данные inbound
        endpoint = "/api/inbounds/get/1"
        data = await panel_session_manager.make_authenticated_request("GET", endpoint)
        
        if not data:
            logger.error("Не удалось получить данные inbound")
            return None
            
        if data.get('success'):
            obj = data.get('obj', {})
            settings_str = obj.get('settings', '{}')
            settings = json.loads(settings_str)
            clients = settings.get('clients', [])
            
            # Ищем клиента по email
            for client in clients:
                if client.get('email') == email:
                    return client
            
            return None
        else:
            logger.error(f"Панель вернула ошибку при поиске клиента: {data.get('msg')}")
            return None
                
    except Exception as e:
        logger.error(f"Ошибка при поиске клиента по email: {e}")
        return None


async def create_client_in_panel(panel_api, client_uuid: str, email: str, 
                                traffic_bytes: int, expiry_time: int, user_id: int = None, subscription_id: int = None) -> bool:
    """
    Создает клиента в панели
    
    Args:
        panel_api: Не используется, оставлен для совместимости
        client_uuid: UUID клиента
        email: Email клиента
        traffic_bytes: Лимит трафика в байтах
        expiry_time: Время истечения в миллисекундах
        
    Returns:
        bool: Успешность создания
    """
    try:
        from bot.services.panel_session_manager import PanelSessionManager
        panel_session_manager = PanelSessionManager()
        
        # Данные для VLESS клиента
        client_data = {
            "id": client_uuid,
            "email": email,
            "limitIp": 0,
            "totalGB": traffic_bytes,
            "expiryTime": expiry_time,
            "enable": True,
            "tgId": str(user_id) if user_id else "",
            "subId": str(subscription_id) if subscription_id else "",
            "flow": "xtls-rprx-vision"
        }
        
        # Формат для добавления клиента в inbound
        api_data = {
            "id": 1,  # ID inbound для VLESS
            "settings": json.dumps({"clients": [client_data]})
        }
        
        logger.info(f"Создание клиента в панели: {email}")
        
        # Используем менеджер cookie для создания клиента
        endpoint = "/api/inbounds/addClient"
        headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        }
        
        response_data = await panel_session_manager.make_authenticated_request(
            "POST", endpoint, json=api_data, headers=headers
        )
        
        if response_data and response_data.get('success'):
            logger.info(f"Клиент успешно создан в панели: {email}")
            return True
        else:
            error_msg = response_data.get('msg', 'Неизвестная ошибка') if response_data else 'Нет ответа от панели'
            logger.error(f"Панель вернула ошибку при создании клиента: {error_msg}")
            return False
                
    except Exception as e:
        logger.error(f"Ошибка при создании клиента в панели: {e}")
        return False


async def get_client_vpn_key(panel_api, client_uuid: str) -> Optional[str]:
    """
    Получает VPN ключ клиента
    
    Args:
        panel_api: Не используется, оставлен для совместимости
        client_uuid: UUID клиента
        
    Returns:
        str или None: VPN ключ или None при ошибке
    """
    try:
        from bot.services.panel_session_manager import PanelSessionManager
        import aiohttp
        import ssl
        
        # Создаем SSL контекст
        ssl_context = ssl.create_default_context()
        ssl_context.check_hostname = False
        ssl_context.verify_mode = ssl.CERT_NONE
        
        # Используем session manager для авторизации
        panel_session_manager = PanelSessionManager()
        if not await panel_session_manager.ensure_authenticated():
            logger.error("Не удалось получить авторизацию")
            return None
        
        # Получаем ссылку через subscription URL используя session manager
        sub_url = f"https://31.172.75.92:47773/Sh1YLmgVQZ7cgcS/{client_uuid}/"
        
        # Используем авторизованную сессию из panel_session_manager
        if not panel_session_manager._session:
            logger.error("Сессия не инициализирована")
            return None
        
        async with panel_session_manager._session.get(sub_url) as response:
            if response.status == 200:
                subscription_data = await response.text()
                if subscription_data and subscription_data.startswith('vless://'):
                    logger.info(f"VPN ключ получен для клиента {client_uuid}")
                    return subscription_data.strip()
                else:
                    logger.warning(f"Получены некорректные данные подписки для {client_uuid}")
                    return None
            else:
                logger.error(f"Ошибка получения подписки: {response.status}")
                return None
                
    except Exception as e:
        logger.error(f"Ошибка при получении VPN ключа: {e}")
        return None


async def normalize_existing_subscriptions():
    """
    Нормализует существующие подписки, получая правильные UUID из панели
    """
    try:
        from bot.database.operations import get_all_subscriptions, update_subscription
        
        logger.info("Начинаем нормализацию существующих подписок...")
        
        # Получаем все подписки
        subscriptions = await get_all_subscriptions()
        
        if not subscriptions:
            logger.info("Нет подписок для нормализации")
            return
        
        # Инициализируем API панели
        panel_url, panel_username, panel_password = get_panel_credentials()
        panel_api = WorkingPanelAPI()
        panel_api.base_url = panel_url
        panel_api.username = panel_username
        panel_api.password = panel_password
        
        if not await panel_api.login():
            logger.error("Не удалось авторизоваться в панели для нормализации")
            return
        
        normalized_count = 0
        
        for subscription in subscriptions:
            try:
                # Формируем ожидаемый email в новом формате
                # Используем сохраненный email из подписки или генерируем новый
                if subscription.vpn_email and '@' in subscription.vpn_email and '@razdvavpn.ru' not in subscription.vpn_email:
                    expected_email = subscription.vpn_email
                else:
                    expected_email = f"{subscription.user_id}@user"
                
                # Ищем клиента в панели по email
                panel_client = await find_client_by_email(panel_api, expected_email)
                
                if panel_client:
                    real_uuid = panel_client['id']
                    
                    # Если UUID в базе не совпадает с реальным, обновляем
                    if subscription.vpn_user_id != real_uuid:
                        logger.info(f"Обновление UUID для пользователя {subscription.user_id}")
                        logger.info(f"  Старый UUID: {subscription.vpn_user_id}")
                        logger.info(f"  Новый UUID:  {real_uuid}")
                        
                        subscription.vpn_user_id = real_uuid
                        
                        # Получаем VPN ключ если его нет
                        if not subscription.vpn_key:
                            vpn_key = await get_client_vpn_key(panel_api, real_uuid)
                            if vpn_key:
                                subscription.vpn_key = vpn_key
                        
                        await update_subscription(subscription)
                        normalized_count += 1
                else:
                    logger.warning(f"Клиент не найден в панели для пользователя {subscription.user_id}")
                    
            except Exception as e:
                logger.error(f"Ошибка при нормализации подписки {subscription.id}: {e}")
                continue
        
        logger.info(f"Нормализация завершена. Обновлено подписок: {normalized_count}")
        
    except Exception as e:
        logger.error(f"Ошибка при нормализации подписок: {e}")
    
    finally:
        try:
            await panel_api.close()
        except:
            pass