import logging
from datetime import datetime
from aiogram import Router, F, types
from aiogram.filters import StateFilter, Command
from aiogram.fsm.context import FSMContext
from aiogram.types import Message, CallbackQuery
from aiogram.utils.keyboard import InlineKeyboardBuilder

from bot.database.operations import (
    get_user, get_user_subscription, update_subscription
)
from bot.utils.states import PaymentStates
from bot.utils.keyboard import get_main_menu_keyboard
from bot.services.payment import create_payment_for_plan
from config import SUBSCRIPTION_PLANS

router = Router()
logger = logging.getLogger(__name__)



async def show_subscription_plans(callback: CallbackQuery, state: FSMContext):
    """Show available subscription plans"""
    keyboard = InlineKeyboardBuilder()
    
    # Добавляем доступные тарифы (исключая тестовый)
    for plan_id, plan in SUBSCRIPTION_PLANS.items():
        if plan_id != "trial":  # Убираем тестовый период из выбора тарифов
            days_text = "3 дня" if plan['duration_days'] == 3 else f"{plan['duration_days']} дней"
            keyboard.button(
                text=f"{plan['name']} - {days_text} ({plan['traffic_gb']} ГБ) - {plan['price']} руб.",
                callback_data=f"subscription:{plan_id}"
            )
    
    keyboard.adjust(1)
    
    plans_text = "📋 <b>Доступные тарифы VPN:</b>\n\n"
    
    for plan_id, plan in SUBSCRIPTION_PLANS.items():
        if plan_id != "trial":  # Exclude trial period from display
            plans_text += (
                f"🔹 <b>{plan['name']}</b> - {plan['price']}₽\n"
                f"   📅 Срок: {plan['duration_days']} дней\n"
                f"   📊 Трафик: {plan['traffic_gb']} ГБ\n\n"
            )
    
    plans_text += "Выберите подходящий тариф:"
    
    try:
        await callback.message.edit_text(
            plans_text,
            reply_markup=keyboard.as_markup(),
            parse_mode="HTML"
        )
        await state.clear()  # Очищаем состояние
    except Exception as e:
        logger.error(f"Ошибка при показе тарифов: {e}")
        await callback.message.answer(
            plans_text,
            reply_markup=keyboard.as_markup(),
            parse_mode="HTML"
        )
    
    await callback.answer()


async def show_detailed_plans_info(callback: CallbackQuery, state: FSMContext):
    """Show detailed information about subscription plans"""
    plans_text = "📋 <b>Подробная информация о тарифах VPN:</b>\n\n"
    
    for plan_id, plan in SUBSCRIPTION_PLANS.items():
        if plan_id != "trial":  # Exclude trial period from detailed display
            plans_text += (
                f"🔸 <b>{plan['name']}</b>\n"
                f"💰 Стоимость: <b>{plan['price']}₽</b>\n"
                f"📅 Длительность: <b>{plan['duration_days']} дней</b>\n"
                f"📊 Трафик: <b>{plan['traffic_gb']} ГБ</b>\n"
                f"📝 {plan['description']}\n\n"
            )
    
    plans_text += "💡 <i>Все тарифы включают:</i>\n"
    plans_text += "• Безлимитную скорость\n"
    plans_text += "• Доступ ко всем серверам\n"
    plans_text += "• Поддержку всех устройств\n"
    plans_text += "• Круглосуточную техподдержку\n\n"
    plans_text += "Готовы оформить подписку?"
    
    # Клавиатура с кнопкой для оформления
    keyboard = InlineKeyboardBuilder()
    keyboard.button(
        text="📲 Оформить подписку",
        callback_data="start_subscription"
    )
    keyboard.adjust(1)
    
    try:
        await callback.message.edit_text(
            plans_text,
            reply_markup=keyboard.as_markup(),
            parse_mode="HTML"
        )
    except Exception as e:
        logger.error(f"Ошибка при показе детальной информации о тарифах: {e}")
        await callback.message.answer(
            plans_text,
            reply_markup=keyboard.as_markup(),
            parse_mode="HTML"
        )
    
    await callback.answer()


@router.message(F.text == "Продлить Подписку")
@router.message(Command("extend"))
async def extend_subscription(message: Message, state: FSMContext):
    """Handle subscription extension request"""
    # Check if user has an active subscription
    subscription = await get_user_subscription(message.from_user.id)
    
    if not subscription:
        await message.answer(
            "❌ У вас нет активной подписки, которую можно продлить.\n\n"
            "Пожалуйста, оформите новую подписку с помощью кнопки «Выбрать Тариф».",
            reply_markup=get_main_menu_keyboard()
        )
        return
    
    # Store current subscription details in state
    await state.update_data(
        current_subscription_id=subscription.id,
        current_plan_id=subscription.plan_id
    )
    
    # Show current plan and options
    current_plan = SUBSCRIPTION_PLANS.get(subscription.plan_id, {"name": "Неизвестный тариф"})
    
    # Create keyboard with plan options
    keyboard = InlineKeyboardBuilder()
    
    # Option to extend current plan
    keyboard.button(
        text=f"🔄 Продлить текущий тариф ({current_plan['name']})",
        callback_data=f"extend_{subscription.plan_id}"
    )
    
    # Option to change plan
    keyboard.button(
        text="📈 Выбрать другой тариф",
        callback_data="change_plan"
    )
    
    keyboard.adjust(1)
    
    await message.answer(
        f"📋 <b>Продление подписки</b>\n\n"
        f"Ваша текущая подписка: <b>{current_plan['name']}</b>\n"
        f"Срок действия: до {subscription.expires_at.strftime('%d.%m.%Y')}\n"
        f"Осталось дней: {subscription.days_remaining}\n"
        f"Осталось трафика: {subscription.traffic_remaining:.2f} ГБ из {subscription.traffic_limit} ГБ\n\n"
        f"Выберите вариант продления:",
        reply_markup=keyboard.as_markup()
    )
    
    await state.set_state(PaymentStates.selecting_extension)


@router.callback_query(StateFilter(PaymentStates.selecting_extension), F.data.startswith("extend_"))
async def process_extend_current_plan(callback: CallbackQuery, state: FSMContext):
    """Process extension of current plan"""
    plan_id = callback.data.split("_")[1]
    plan = SUBSCRIPTION_PLANS.get(plan_id)
    
    if not plan:
        await callback.message.edit_text(
            "❌ Произошла ошибка. Тариф не найден.",
            reply_markup=get_main_menu_keyboard()
        )
        await state.clear()
        return
    
    # Initialize payment process
    try:
        # Get user and subscription information
        user_id = callback.from_user.id
        user_data = await state.get_data()
        
        subscription_id = user_data.get("current_subscription_id")
        
        # Create payment
        payment_info = await create_payment_for_plan(
            user_id=user_id,
            plan_id=plan_id,
            amount=plan['price'],
            description=f"Продление подписки VPN: {plan['name']} на {plan['duration_days']} дней",
            subscription_id=subscription_id
        )
        
        # Store payment info
        await state.update_data(
            payment_id=payment_info["payment_id"],
            payment_url=payment_info["payment_url"]
        )
        
        # Build keyboard with payment link
        keyboard = InlineKeyboardBuilder()
        keyboard.button(
            text="💳 Перейти к оплате",
            url=payment_info["payment_url"]
        )
        keyboard.button(
            text="🏠 В главное меню",
            callback_data="to_main_menu"
        )
        keyboard.adjust(1)
        
        await callback.message.edit_text(
            f"💰 <b>Продление подписки {plan['name']}</b>\n\n"
            f"Сумма к оплате: {plan['price']} руб.\n"
            f"Нажмите на кнопку ниже, чтобы перейти к оплате.\n\n"
            f"💡 Подписка активируется автоматически после успешной оплаты.",
            reply_markup=keyboard.as_markup()
        )
        
        await state.set_state(PaymentStates.awaiting_payment)
        
    except Exception as e:
        logger.error(f"Error creating payment: {e}")
        await callback.message.edit_text(
            "❌ Произошла ошибка при создании платежа. Пожалуйста, попробуйте позже.",
            reply_markup=get_main_menu_keyboard()
        )
        await state.clear()
    
    await callback.answer()


@router.callback_query(StateFilter(PaymentStates.selecting_extension), F.data == "change_plan")
async def show_available_plans(callback: CallbackQuery, state: FSMContext):
    """Show available plans for changing"""
    logger.info(f"Пользователь {callback.from_user.id} нажал кнопку 'Выбрать другой тариф'")
    
    current_state = await state.get_state()
    logger.info(f"Текущее состояние FSM: {current_state}")
    
    keyboard = InlineKeyboardBuilder()
    
    for plan_id, plan in SUBSCRIPTION_PLANS.items():
        if plan_id != "trial":  # Exclude trial period from plan change options
            days_text = "3 дня" if plan['duration_days'] == 3 else f"{plan['duration_days']} дней"
            keyboard.button(
                text=f"{plan['name']} - {days_text} ({plan['traffic_gb']} ГБ) - {plan['price']} руб.",
                callback_data=f"select_plan_{plan_id}"
            )
    
    keyboard.button(
        text="🔙 Назад",
        callback_data="back_to_extension"
    )
    
    keyboard.adjust(1)
    
    await callback.message.edit_text(
        "📋 <b>Выберите новый тарифный план:</b>\n\n"
        "После оплаты выбранный тариф заменит текущий.",
        reply_markup=keyboard.as_markup()
    )
    
    await state.set_state(PaymentStates.selecting_new_plan)
    await callback.answer()


@router.callback_query(StateFilter(PaymentStates.selecting_new_plan), F.data.startswith("select_plan_"))
async def process_plan_change(callback: CallbackQuery, state: FSMContext):
    """Process changing to a new plan"""
    plan_id = callback.data.split("_")[2]
    plan = SUBSCRIPTION_PLANS.get(plan_id)
    
    if not plan:
        await callback.message.edit_text(
            "❌ Произошла ошибка. Тариф не найден.",
            reply_markup=get_main_menu_keyboard()
        )
        await state.clear()
        return
    
    # Initialize payment process
    try:
        # Get user and subscription information
        user_id = callback.from_user.id
        user_data = await state.get_data()
        
        subscription_id = user_data.get("current_subscription_id")
        
        # Create payment
        payment_info = await create_payment_for_plan(
            user_id=user_id,
            plan_id=plan_id,
            amount=plan['price'],
            description=f"Изменение тарифа VPN на: {plan['name']} на {plan['duration_days']} дней",
            subscription_id=subscription_id
        )
        
        # Store payment info
        await state.update_data(
            payment_id=payment_info["payment_id"],
            payment_url=payment_info["payment_url"],
            new_plan_id=plan_id
        )
        
        # Build keyboard with payment link
        keyboard = InlineKeyboardBuilder()
        keyboard.button(
            text="💳 Перейти к оплате",
            url=payment_info["payment_url"]
        )
        keyboard.button(
            text="📈 Выбрать другой тариф",
            callback_data="change_plan"
        )
        keyboard.button(
            text="🏠 В главное меню",
            callback_data="to_main_menu"
        )
        keyboard.adjust(1)
        
        await callback.message.edit_text(
            f"💰 <b>Изменение тарифа на {plan['name']}</b>\n\n"
            f"Сумма к оплате: {plan['price']} руб.\n"
            f"Нажмите на кнопку ниже, чтобы перейти к оплате.\n\n"
            f"💡 Подписка активируется автоматически после успешной оплаты.",
            reply_markup=keyboard.as_markup()
        )
        
        await state.set_state(PaymentStates.awaiting_payment)
        
    except Exception as e:
        logger.error(f"Error creating payment: {e}")
        await callback.message.edit_text(
            "❌ Произошла ошибка при создании платежа. Пожалуйста, попробуйте позже.",
            reply_markup=get_main_menu_keyboard()
        )
        await state.clear()
    
    await callback.answer()


@router.callback_query(StateFilter(PaymentStates.selecting_new_plan), F.data == "back_to_extension")
async def back_to_extension_options(callback: CallbackQuery, state: FSMContext):
    """Return to extension options"""
    user_id = callback.from_user.id
    subscription = await get_user_subscription(user_id)
    
    if not subscription:
        await callback.message.edit_text(
            "❌ У вас нет активной подписки, которую можно продлить.",
            reply_markup=get_main_menu_keyboard()
        )
        await state.clear()
        return
    
    # Show current plan and options
    current_plan = SUBSCRIPTION_PLANS.get(subscription.plan_id, {"name": "Неизвестный тариф"})
    
    # Create keyboard with plan options
    keyboard = InlineKeyboardBuilder()
    
    # Option to extend current plan
    keyboard.button(
        text=f"🔄 Продлить текущий тариф ({current_plan['name']})",
        callback_data=f"extend_{subscription.plan_id}"
    )
    
    # Option to change plan
    keyboard.button(
        text="📈 Выбрать другой тариф",
        callback_data="change_plan"
    )
    
    keyboard.adjust(1)
    
    await callback.message.edit_text(
        f"📋 <b>Продление подписки</b>\n\n"
        f"Ваша текущая подписка: <b>{current_plan['name']}</b>\n"
        f"Срок действия: до {subscription.expires_at.strftime('%d.%m.%Y')}\n"
        f"Осталось дней: {subscription.days_remaining}\n"
        f"Осталось трафика: {subscription.traffic_remaining:.2f} ГБ из {subscription.traffic_limit} ГБ\n\n"
        f"Выберите вариант продления:",
        reply_markup=keyboard.as_markup()
    )
    
    await state.set_state(PaymentStates.selecting_extension)
    await callback.answer()





@router.callback_query(F.data == "refresh_vpn_key")
async def refresh_vpn_key(callback: CallbackQuery):
    """Get VPN key for new clients who just paid"""
    try:
        subscription = await get_user_subscription(callback.from_user.id)
        
        if not subscription:
            await callback.message.edit_text(
                "❌ У вас нет активной подписки.",
                reply_markup=get_main_menu_keyboard()
            )
            await callback.answer()
            return
        
        # Try to get VPN key from panel using the working method
        from bot.services.panel_session_manager import PanelSessionManager
        import json
        try:
            manager = PanelSessionManager()
            
            # Get inbound data and generate the correct VLESS link
            response = await manager.make_authenticated_request('GET', 'inbounds/get/1')
            
            if response and response.get('success'):
                obj = response.get('obj', {})
                
                # Parse settings
                stream_settings = json.loads(obj.get('streamSettings', '{}')) if isinstance(obj.get('streamSettings'), str) else obj.get('streamSettings', {})
                settings = json.loads(obj.get('settings', '{}')) if isinstance(obj.get('settings'), str) else obj.get('settings', {})
                
                reality_settings = stream_settings.get('realitySettings', {})
                clients = settings.get('clients', [])
                
                # Find client by stored client_id
                target_client = None
                for client in clients:
                    if client.get('id') == subscription.vpn_user_id:
                        target_client = client
                        break
                
                if target_client:
                    # Extract key and create VLESS link using correct panel format
                    key = target_client['id']
                    
                    # Get correct port from inbound
                    port = obj.get("port", 443)
                    server_ip = "31.172.75.92"
                    
                    # Get Reality parameters correctly from the proper location
                    reality_inner_settings = reality_settings.get("settings", {})
                    public_key = reality_inner_settings.get("publicKey", "")  # Correct public key location
                    short_ids = reality_settings.get("shortIds", [])
                    short_id = short_ids[0] if short_ids else ""
                    server_names = reality_settings.get("serverNames", [])
                    sni = server_names[0] if server_names else "yahoo.com"
                    
                    # Form VLESS link using correct working format
                    vpn_key = (
                        f"vless://{key}@{server_ip}:{port}"
                        f"?type=tcp&security=reality&pbk={public_key}"
                        f"&fp=chrome&sni={sni}&sid={short_id}"
                        f"&spx=%2F&flow=xtls-rprx-vision"
                        f"#RazDvaVPN%20Vless-{subscription.vpn_email.replace('@', '%40')}"
                    )
                    
                    # Update subscription with the correct VLESS link
                    subscription.vpn_key = vpn_key
                    await update_subscription(subscription)
                    
                    markup = InlineKeyboardBuilder()
                    markup.button(text="📋 Скопировать ссылку", callback_data=f"copy_vpn_link_{subscription.id}")
                    markup.button(text="🏠 В главное меню", callback_data="back_to_main")
                    markup.adjust(1)
                    
                    await callback.message.edit_text(
                        f"✅ <b>Ваш ключ VPN получен!</b>\n\n"
                        f"🔑 <b>Ключ VLESS:</b> <code>{vpn_key}</code>\n\n"
                        f"⚠️ Сохраните ключ! Он также доступен в разделе «Личный Кабинет».",
                        reply_markup=markup.as_markup(),
                        parse_mode="HTML"
                    )
                else:
                    raise Exception("Клиент не найден")
            else:
                raise Exception("Не удалось получить данные из панели")
        except Exception as e:
            logger.error(f"Error getting VPN key for new client: {e}")
            markup = InlineKeyboardBuilder()
            markup.button(text="🔄 Попробовать снова", callback_data="refresh_vpn_key")
            markup.button(text="🏠 В главное меню", callback_data="back_to_main")
            markup.adjust(1)
            
            await callback.message.edit_text(
                "⚠️ Временные проблемы с получением VPN ключа.\n\n"
                "Попробуйте через несколько минут или обратитесь в поддержку.",
                reply_markup=markup.as_markup()
            )
    except Exception as e:
        logger.error(f"Error in refresh_vpn_key handler: {e}")
        await callback.message.edit_text(
            "❌ Произошла ошибка. Пожалуйста, обратитесь в поддержку.",
            reply_markup=get_main_menu_keyboard()
        )
    
    await callback.answer()


@router.callback_query(F.data == "back_to_main")
async def back_to_main(callback: CallbackQuery, state: FSMContext):
    """Handle back to main menu button click"""
    # Очищаем текущее состояние
    await state.clear()
    
    # Отправляем сообщение главного меню
    await callback.message.answer(
        "🏠 <b>Главное меню</b>\n\n"
        "Выберите действие из меню ниже:",
        reply_markup=get_main_menu_keyboard()
    )
    
    # Удаляем предыдущее сообщение с инлайн-кнопкой
    try:
        await callback.message.delete()
    except Exception as e:
        logger.error(f"Error deleting message: {e}")
    
    await callback.answer()


@router.callback_query()
async def debug_unhandled_callbacks(callback: CallbackQuery, state: FSMContext):
    """Debug handler for unhandled callbacks"""
    # Skip trial period callbacks - they are handled by subscription.py
    if callback.data == "activate_test_period":
        return
        
    current_state = await state.get_state()
    logger.info(f"Необработанный callback от пользователя {callback.from_user.id}")
    logger.info(f"Данные callback: {callback.data}")
    logger.info(f"Текущее состояние FSM: {current_state}")
    
    # Отвечаем на callback, чтобы убрать загрузку
    await callback.answer("Обрабатываем ваш запрос...")

def register_handlers(dp):
    """Register all payment handlers"""
    dp.include_router(router)
