from aiogram import Router, F, types
from aiogram.filters import Command
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.utils.keyboard import ReplyKeyboardBuilder

from bot.database.operations import get_user, create_or_update_user, get_user_subscription
from bot.database.models import User, UserStatus
from bot.utils.keyboard import get_main_menu_keyboard
from bot.utils.email_generator import generate_vpn_email
from config import SUBSCRIPTION_PLANS
import os

router = Router()

# Константы для путей к ресурсам
PLASHKA_IMAGE_PATH = "bot/assets/plashka.jpg"
HIDDIFY_LOGO_PATH = "bot/assets/hiddifylogo1280.png"


def generate_tariff_info_text():
    """Генерирует текст с информацией о тарифах на основе конфигурации"""
    text = "<b>Доступные тарифные планы:</b>\n\n"

    # Добавляем тестовый тариф отдельно
    if "trial" in SUBSCRIPTION_PLANS:
        trial = SUBSCRIPTION_PLANS["trial"]
        text += (
            f"<b>Тестовый «Старт»</b> - ⏱️ на {trial['duration_days']} дня, "
            f"💰- {trial['price']} руб.\n"
            f"📦 {trial['traffic_gb']} ГБ трафика\n"
            f"<i>*Разовый бесплатный тестовый период</i>\n"
            f"==================================\n"
            f"<b>Тарифы на платные подписки:</b>\n\n"
        )

    # Добавляем платные тарифы
    for plan_id, plan in SUBSCRIPTION_PLANS.items():
        if plan_id == "trial":
            continue

        # Приблизительное время видео (1 ГБ ≈ 1.1 часа видео)
        video_hours = int(plan["traffic_gb"] * 1.1)

        # Определяем количество звезд по цене
        stars = ""
        if plan["price"] >= 1500:
            stars = "⭐⭐⭐⭐"
        elif plan["price"] >= 700:
            stars = "⭐⭐⭐"
        elif plan["price"] >= 300:
            stars = "⭐⭐"
        elif plan["price"] >= 100:
            stars = "⭐"

        if stars:
            text += f"{stars}\n"

        text += (
            f"<b>{plan['name']}</b> - ⏱️ {plan['duration_days']} дней, "
            f"💰- {plan['price']} руб.\n"
            f"📦 {plan['traffic_gb']} ГБ трафика (~{video_hours}* ч. Видео)\n\n"
        )

    text += (
        "<i>*Значения часов трафика видео даны примерные</i>\n\n"
        "👇 Для оплаты Тарифа, Продления, Изменения используйте Кнопки в ГЛАВНОМ МЕНЮ."
    )

    return text


@router.message(Command("start"))
async def cmd_start(message: Message):
    """
    Handle /start command - entry point for users
    """
    # Create or update user in the database
    user = await get_user(message.from_user.id)

    if not user:
        # Генерируем email для нового пользователя
        vpn_email = generate_vpn_email(message.from_user.id, message.from_user.username)

        user = User(
            id=message.from_user.id,
            username=message.from_user.username,
            first_name=message.from_user.first_name,
            last_name=message.from_user.last_name,
            status="NEW",
            vpn_email=vpn_email,
        )
        await create_or_update_user(user)
    elif not user.vpn_email:
        # Если у существующего пользователя нет email, создаем его
        user.vpn_email = generate_vpn_email(message.from_user.id, message.from_user.username)
        await create_or_update_user(user)

    # Check if user already has a subscription
    subscription = await get_user_subscription(message.from_user.id)

    # Отправляем плашку с логотипом в самом начале
    from aiogram.types import FSInputFile

    try:
        if os.path.exists(PLASHKA_IMAGE_PATH):
            await message.answer_photo(FSInputFile(PLASHKA_IMAGE_PATH))
    except Exception as e:
        # Логируем ошибку, но продолжаем работу
        print(f"Ошибка при отправке изображения: {e}")

    # Welcome message с отступом
    welcome_text = (
        f"    👋 <b>Добро пожаловать в РазДваВПН!</b>\n\n"
        f"    С помощью этого бота вы сможете управлять своей VPN-подпиской.\n\n"
        f"    Что вы можете сделать:\n"
        f"    • 🔑 Приобрести новую подписку\n"
        f"    • ⌛ Продлить существующую подписку\n"
        f"    • ℹ️ Узнать о доступных тарифах\n"
        f"    • 👤 Управлять своим аккаунтом\n\n"
    )

    # Если у пользователя еще нет подписки, новый флоу для установки приложения
    if not subscription:
        welcome_text += (
            f"    🎁 <b>Специальное предложение!</b>\n"
            f"    Попробуйте наш сервис бесплатно в течение 3 дней с лимитом 10 ГБ трафика. "
            f"    Этого будет достаточно на ~10 часов просмотра видео.\n\n"
        )

        # Отправляем приветственное сообщение
        await message.answer(welcome_text, parse_mode="HTML")

        # Рекомендация установить приложение Hiddify
        recommendation_text = (
            "📱 <b>Рекомендуем установить приложение Hiddify</b>\n\n"
            "    Для комфортного использования VPN рекомендуем установить приложение Hiddify "
            "    до активации тестового периода или оплаты подписки.\n\n"
            "    После установки приложения получите ключ, скопируйте его и добавьте "
            '    с помощью кнопки <b>"+ Добавить новый профиль"</b>.'
        )

        # Создаем кнопку для получения информации о установке
        info_kb = InlineKeyboardMarkup(
            inline_keyboard=[
                [
                    InlineKeyboardButton(
                        text="📋 Получить информацию о установке приложения",
                        callback_data="get_install_info",
                    )
                ]
            ]
        )

        await message.answer(recommendation_text, reply_markup=info_kb, parse_mode="HTML")
    else:
        welcome_text += "    Выберите действие в меню ниже:"
        await message.answer(welcome_text, reply_markup=get_main_menu_keyboard(), parse_mode="HTML")


@router.callback_query(F.data == "get_install_info")
async def handle_get_install_info(callback: CallbackQuery):
    """Show Hiddify app installation information"""
    # Отправляем изображение приложения Hiddify
    from aiogram.types import FSInputFile

    caption = (
        "🚀 <b>Приложение Hiddify для РазДваВПН</b>\n\n"
        "• Простой и лаконичный интерфейс\n"
        "• Кроссплатформенность: Android, iOS, Windows, Mac, Linux\n"
        "• Автоматический выбор профиля с наименьшей задержкой\n\n"
        "<b>Скачать приложение:</b>\n"
        "📱 Android: <a href='https://play.google.com/store/apps/details?id=app.hiddify.com'>Google Play</a> или <a href='https://github.com/hiddify/hiddify-next/releases/latest'>GitHub</a>\n"
        "🍎 iOS: <a href='https://apps.apple.com/us/app/hiddify-next/id6476212173'>App Store</a>\n"
        "💻 Windows: <a href='https://github.com/hiddify/hiddify-next/releases/latest/download/Hiddify-Windows-Setup-x64.exe'>Скачать</a>\n"
        "🖥 macOS: <a href='https://github.com/hiddify/hiddify-next/releases/latest'>GitHub</a>\n"
        "🐧 Linux: <a href='https://github.com/hiddify/hiddify-next/releases/latest/download/Hiddify-Linux-x64.AppImage'>Скачать</a>\n\n"
        "После установки нажмите кнопку ниже:"
    )

    # Создаем кнопку "Я установил"
    install_kb = InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="✅ Я установил приложение", callback_data="app_installed")]
        ]
    )

    # Отправляем изображение с подписью и кнопкой
    try:
        if os.path.exists(HIDDIFY_LOGO_PATH):
            await callback.message.answer_photo(
                FSInputFile(HIDDIFY_LOGO_PATH),
                caption=caption,
                reply_markup=install_kb,
                parse_mode="HTML",
            )
        else:
            # Если изображение не найдено, отправляем только текст
            await callback.message.answer(caption, reply_markup=install_kb, parse_mode="HTML")
    except Exception as e:
        print(f"Ошибка при отправке изображения Hiddify: {e}")
        # Fallback - отправляем только текст
        await callback.message.answer(caption, reply_markup=install_kb, parse_mode="HTML")

    await callback.answer()


@router.callback_query(F.data == "app_installed")
async def handle_app_installed(callback: CallbackQuery):
    """Handle when user confirms app installation"""
    # Создаем кнопки для выбора: тестовый период или платный тариф
    choices_kb = InlineKeyboardMarkup(
        inline_keyboard=[
            [
                InlineKeyboardButton(
                    text="🚀 Активировать пробный период (3 дня)",
                    callback_data="activate_test_period",
                )
            ],
            [InlineKeyboardButton(text="💳 Выбрать тариф", callback_data="select_paid_plan")],
        ]
    )

    await callback.message.answer(
        "✅ <b>Отлично! Приложение установлено.</b>\n\n"
        "Теперь выберите один из вариантов:\n\n"
        "🚀 <b>Пробный период</b> - 3 дня бесплатно с 10 ГБ трафика\n"
        "💳 <b>Платный тариф</b> - сразу перейти к выбору подписки\n\n"
        'После получения ключа добавьте его в приложение Hiddify через кнопку "+ Добавить новый профиль".',
        reply_markup=choices_kb,
        parse_mode="HTML",
    )

    await callback.answer()


@router.message(F.text == "Главное меню")
async def return_to_main_menu(message: Message):
    """Handle returning to the main menu"""
    await message.answer(
        "Выберите действие из главного меню:", reply_markup=get_main_menu_keyboard()
    )


@router.message(F.text == "Информация о Тарифах")
async def show_tariff_info(message: Message):
    """Display information about available tariff plans"""
    tariff_text = generate_tariff_info_text()
    await message.answer(tariff_text, reply_markup=get_main_menu_keyboard(), parse_mode="HTML")


def register_handlers(dp):
    dp.include_router(router)
