import os
import aiosqlite
from datetime import datetime, timedelta
import logging
import time
import functools
from typing import List, Optional, Dict, Any, Tuple
from contextlib import asynccontextmanager

from bot.database.models import User, Subscription, Payment, UserStatus, PaymentStatus
from config import DATABASE_FILE

# Ensure directory exists
os.makedirs(os.path.dirname(DATABASE_FILE), exist_ok=True)

logger = logging.getLogger(__name__)

# Пул соединений с базой данных для оптимизации
_connection_pool = {}
_max_pool_size = 5
_connection_timeout = 300  # 5 минут - время жизни соединения

@asynccontextmanager
async def get_db_connection():
    """
    Получение соединения с базой данных из пула или создание нового
    """
    current_time = time.time()
    conn_id = None
    connection = None
    
    # Очистить просроченные соединения из пула
    expired_conns = []
    for conn_id, (conn, last_used) in _connection_pool.items():
        if current_time - last_used > _connection_timeout:
            expired_conns.append(conn_id)
    
    for conn_id in expired_conns:
        conn, _ = _connection_pool.pop(conn_id)
        await conn.close()
        logger.debug(f"Closed expired connection {conn_id}")
    
    # Использовать существующее соединение из пула
    for conn_id, (conn, _) in list(_connection_pool.items()):
        try:
            # Проверка работоспособности
            await conn.execute("SELECT 1")
            _connection_pool[conn_id] = (conn, current_time)
            connection = conn
            break
        except Exception as e:
            logger.warning(f"Removing broken connection {conn_id}: {e}")
            _connection_pool.pop(conn_id)
            await conn.close()
    
    # Создать новое соединение, если нужно
    if connection is None:
        connection = await aiosqlite.connect(DATABASE_FILE)
        connection.row_factory = aiosqlite.Row
        
        # Оптимизация SQLite
        await connection.execute("PRAGMA foreign_keys = ON")
        await connection.execute("PRAGMA journal_mode = WAL")
        await connection.execute("PRAGMA synchronous = NORMAL")
        await connection.execute("PRAGMA cache_size = 10000")
        
        conn_id = id(connection)
        _connection_pool[conn_id] = (connection, current_time)
        logger.debug(f"Created new database connection {conn_id}")
    
    try:
        yield connection
    finally:
        # Обновить временную отметку использования
        if conn_id in _connection_pool:
            _connection_pool[conn_id] = (connection, time.time())

# Декоратор для обработки ошибок базы данных
def db_error_handler(func):
    @functools.wraps(func)
    async def wrapper(*args, **kwargs):
        try:
            return await func(*args, **kwargs)
        except aiosqlite.Error as e:
            logger.error(f"Database error in {func.__name__}: {e}")
            raise
        except Exception as e:
            logger.error(f"Unexpected error in {func.__name__}: {e}")
            raise
    return wrapper


@db_error_handler
async def init_db():
    """Initialize the database with required tables if they don't exist."""
    async with get_db_connection() as db:
        # Create users table
        await db.execute('''
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY,
            username TEXT,
            first_name TEXT,
            last_name TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            status TEXT,
            phone_number TEXT,
            vpn_email TEXT
        )
        ''')
        
        # Create subscriptions table
        await db.execute('''
        CREATE TABLE IF NOT EXISTS subscriptions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            plan_id TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            expires_at TIMESTAMP,
            traffic_limit REAL,
            traffic_used REAL DEFAULT 0,
            vpn_user_id INTEGER,
            vpn_key TEXT,
            status TEXT,
            vpn_email TEXT,
            FOREIGN KEY (user_id) REFERENCES users(id)
        )
        ''')
        
        # Добавляем поле vpn_email если его нет (для существующих баз)
        try:
            await db.execute('ALTER TABLE subscriptions ADD COLUMN vpn_email TEXT')
        except Exception:
            # Поле уже существует
            pass
        
        # Create payments table
        await db.execute('''
        CREATE TABLE IF NOT EXISTS payments (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            subscription_id INTEGER,
            amount REAL,
            currency TEXT DEFAULT 'RUB',
            payment_id TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            completed_at TIMESTAMP,
            status TEXT,
            plan_id TEXT,
            FOREIGN KEY (user_id) REFERENCES users(id),
            FOREIGN KEY (subscription_id) REFERENCES subscriptions(id)
        )
        ''')
        
        # Создаем индексы для оптимизации запросов
        # Индекс по user_id для быстрого поиска подписок пользователя
        await db.execute('CREATE INDEX IF NOT EXISTS idx_subscriptions_user_id ON subscriptions(user_id)')
        
        # Индекс по дате истечения для эффективного поиска скоро истекающих подписок
        await db.execute('CREATE INDEX IF NOT EXISTS idx_subscriptions_expires_at ON subscriptions(expires_at)')
        
        # Индекс по vpn_user_id для быстрого поиска подписок по идентификатору VPN
        await db.execute('CREATE INDEX IF NOT EXISTS idx_subscriptions_vpn_user_id ON subscriptions(vpn_user_id)')
        
        # Индексы для таблицы платежей
        await db.execute('CREATE INDEX IF NOT EXISTS idx_payments_user_id ON payments(user_id)')
        await db.execute('CREATE INDEX IF NOT EXISTS idx_payments_payment_id ON payments(payment_id)')
        await db.execute('CREATE INDEX IF NOT EXISTS idx_payments_status ON payments(status)')
        await db.execute('CREATE INDEX IF NOT EXISTS idx_payments_created_at ON payments(created_at)')
        
        # Оптимизация SQLite для повышения производительности
        await db.execute('PRAGMA journal_mode = WAL')
        await db.execute('PRAGMA synchronous = NORMAL')
        await db.execute('PRAGMA cache_size = 10000')
        await db.execute('PRAGMA foreign_keys = ON')
        
        # Миграция: добавляем поле vpn_email если его нет
        try:
            await db.execute('ALTER TABLE users ADD COLUMN vpn_email TEXT')
            logger.info("Added vpn_email column to users table")
        except Exception:
            # Поле уже существует или другая ошибка
            pass
        
        # Анализ таблиц для оптимизатора запросов
        await db.execute('ANALYZE')
        
        await db.commit()
        logger.info("Database initialized successfully with optimizations and indexes")


# User operations
@db_error_handler
async def get_user(user_id: int) -> Optional[User]:
    """Get user by ID"""
    async with get_db_connection() as db:
        cursor = await db.execute(
            "SELECT * FROM users WHERE id = ?", 
            (user_id,)
        )
        row = await cursor.fetchone()
        
        if row:
            try:
                status = UserStatus[row['status']] if row['status'] else UserStatus.NEW
            except (KeyError, ValueError):
                logger.warning(f"Invalid user status '{row['status']}' for user {user_id}, defaulting to NEW")
                status = UserStatus.NEW
                
            vpn_email = None
            try:
                vpn_email = row['vpn_email']
            except (KeyError, IndexError):
                pass
            
            return User(
                id=row['id'],
                username=row['username'],
                first_name=row['first_name'],
                last_name=row['last_name'],
                created_at=datetime.fromisoformat(row['created_at']) if row['created_at'] else datetime.now(),
                status=status,
                phone_number=row['phone_number'],
                vpn_email=vpn_email
            )
        
        logger.debug(f"User with ID {user_id} not found in database")
        return None


@db_error_handler
async def create_or_update_user(user: User) -> User:
    """Create or update user in the database"""
    async with get_db_connection() as db:
        try:
            # Проверка, существует ли пользователь
            cursor = await db.execute(
                "SELECT * FROM users WHERE id = ?", 
                (user.id,)
            )
            existing_user = await cursor.fetchone()
            
            if existing_user:
                # Обновление существующего пользователя
                logger.debug(f"Updating existing user with ID {user.id}")
                await db.execute(
                    """
                    UPDATE users 
                    SET username = ?, first_name = ?, last_name = ?, 
                        status = ?, phone_number = ?, vpn_email = ?
                    WHERE id = ?
                    """,
                    (
                        user.username, user.first_name, user.last_name, 
                        user.status.value if hasattr(user.status, 'value') else str(user.status), 
                        user.phone_number, user.vpn_email, user.id
                    )
                )
            else:
                # Создание нового пользователя
                logger.info(f"Creating new user with ID {user.id}")
                
                # Проверяем наличие временной метки создания
                created_at = user.created_at
                if not created_at:
                    created_at = datetime.now()
                
                await db.execute(
                    """
                    INSERT INTO users 
                    (id, username, first_name, last_name, created_at, status, phone_number, vpn_email)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    """,
                    (
                        user.id, user.username, user.first_name, user.last_name, 
                        created_at.isoformat(), user.status.value if hasattr(user.status, 'value') else str(user.status), 
                        user.phone_number, user.vpn_email
                    )
                )
                
            await db.commit()
            return user
        except Exception as e:
            logger.error(f"Error creating/updating user {user.id}: {e}")
            await db.rollback()
            raise


@db_error_handler
async def get_all_users() -> List[User]:
    """Get all users from the database"""
    async with get_db_connection() as db:
        cursor = await db.execute("SELECT * FROM users")
        rows = await cursor.fetchall()
        
        users = []
        for row in rows:
            try:
                created_at = datetime.fromisoformat(row['created_at']) if row['created_at'] else None
                status = UserStatus[row['status']] if row['status'] else UserStatus.NEW
                
                vpn_email = None
                try:
                    vpn_email = row['vpn_email']
                except (KeyError, IndexError):
                    pass
                
                user = User(
                    id=row['id'],
                    username=row['username'],
                    first_name=row['first_name'],
                    last_name=row['last_name'],
                    created_at=created_at,
                    status=status,
                    phone_number=row['phone_number'],
                    vpn_email=vpn_email
                )
                users.append(user)
            except Exception as e:
                logger.error(f"Error parsing user data for ID {row['id']}: {e}")
        
        return users


# Subscription operations
@db_error_handler
async def get_subscription_by_id(subscription_id: int) -> Optional[Subscription]:
    """Get subscription by ID"""
    async with get_db_connection() as db:
        cursor = await db.execute(
            """
            SELECT * FROM subscriptions 
            WHERE id = ?
            """, 
            (subscription_id,)
        )
        row = await cursor.fetchone()
        
        if row:
            try:
                created_at = datetime.fromisoformat(row['created_at']) if row['created_at'] else datetime.now()
                expires_at = datetime.fromisoformat(row['expires_at']) if row['expires_at'] else None
                
                return Subscription(
                    id=row['id'],
                    user_id=row['user_id'],
                    plan_id=row['plan_id'],
                    created_at=created_at,
                    expires_at=expires_at,
                    traffic_limit=float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0,
                    traffic_used=float(row['traffic_used']) if row['traffic_used'] is not None else 0.0,
                    vpn_user_id=row['vpn_user_id'],
                    vpn_key=row['vpn_key'],
                    status=row['status'],
                    vpn_email=row['vpn_email']
                )
            except Exception as e:
                logger.error(f"Error parsing subscription data for ID {subscription_id}: {e}")
                
        logger.debug(f"Subscription with ID {subscription_id} not found")
        return None

@db_error_handler
async def get_user_subscription(user_id: int) -> Optional[Subscription]:
    """Get active subscription for user"""
    async with get_db_connection() as db:
        cursor = await db.execute(
            """
            SELECT * FROM subscriptions 
            WHERE user_id = ? 
            AND expires_at > datetime('now')
            AND (status IS NULL OR status != 'EXPIRED')
            ORDER BY expires_at DESC LIMIT 1
            """, 
            (user_id,)
        )
        row = await cursor.fetchone()
        
        if row:
            try:
                created_at = datetime.fromisoformat(row['created_at']) if row['created_at'] else datetime.now()
                expires_at = datetime.fromisoformat(row['expires_at']) if row['expires_at'] else None
                
                return Subscription(
                    id=row['id'],
                    user_id=row['user_id'],
                    plan_id=row['plan_id'],
                    created_at=created_at,
                    expires_at=expires_at,
                    traffic_limit=float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0,
                    traffic_used=float(row['traffic_used']) if row['traffic_used'] is not None else 0.0,
                    vpn_user_id=row['vpn_user_id'],
                    vpn_key=row['vpn_key'],
                    status=row['status'],
                    vpn_email=row['vpn_email']
                )
            except Exception as e:
                logger.error(f"Error parsing subscription data for user {user_id}: {e}")
                
        logger.debug(f"No active subscription found for user {user_id}")
        return None


@db_error_handler
async def create_subscription(subscription: Subscription) -> Subscription:
    """Create a new subscription"""
    async with get_db_connection() as db:
        try:
            # Проверяем, что все необходимые данные присутствуют
            if not subscription.created_at:
                subscription.created_at = datetime.now()
                
            # Логирование создания подписки
            logger.info(f"Creating subscription for user {subscription.user_id} with plan {subscription.plan_id}")
            
            cursor = await db.execute(
                """
                INSERT INTO subscriptions
                (user_id, plan_id, created_at, expires_at, traffic_limit, 
                 traffic_used, vpn_user_id, vpn_key, status, vpn_email)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                RETURNING id
                """,
                (
                    subscription.user_id, subscription.plan_id,
                    subscription.created_at.isoformat() if hasattr(subscription.created_at, 'isoformat') else subscription.created_at,
                    subscription.expires_at.isoformat() if subscription.expires_at and hasattr(subscription.expires_at, 'isoformat') else subscription.expires_at,
                    float(subscription.traffic_limit) if subscription.traffic_limit is not None else 0.0, 
                    float(subscription.traffic_used) if subscription.traffic_used is not None else 0.0,
                    subscription.vpn_user_id, subscription.vpn_key,
                    subscription.status, subscription.vpn_email
                )
            )
            result = await cursor.fetchone()
            if result:
                subscription.id = result[0]
            else:
                raise Exception("Failed to get subscription ID after creation")
            
            await db.commit()
            logger.info(f"Successfully created subscription with ID {subscription.id}")
            return subscription
        except Exception as e:
            logger.error(f"Error creating subscription: {e}")
            await db.rollback()
            raise


@db_error_handler
async def delete_subscription_completely(subscription_id: int) -> bool:
    """
    Полностью удаляет подписку из базы данных
    
    Args:
        subscription_id: ID подписки для удаления
        
    Returns:
        bool: True если удаление успешно, False если подписка не найдена
    """
    async with get_db_connection() as db:
        try:
            # Сначала проверяем, существует ли подписка
            cursor = await db.execute(
                "SELECT id, user_id FROM subscriptions WHERE id = ?",
                (subscription_id,)
            )
            result = await cursor.fetchone()
            
            if not result:
                logger.warning(f"Подписка с ID {subscription_id} не найдена для удаления")
                return False
            
            user_id = result[1]
            logger.info(f"Удаление подписки {subscription_id} для пользователя {user_id}")
            
            # Удаляем связанные платежи (если есть)
            await db.execute(
                "DELETE FROM payments WHERE subscription_id = ?",
                (subscription_id,)
            )
            
            # Удаляем саму подписку
            await db.execute(
                "DELETE FROM subscriptions WHERE id = ?",
                (subscription_id,)
            )
            
            await db.commit()
            logger.info(f"Подписка {subscription_id} полностью удалена из базы данных")
            return True
            
        except Exception as e:
            logger.error(f"Ошибка при удалении подписки {subscription_id}: {e}")
            await db.rollback()
            raise


@db_error_handler
async def delete_user_completely(user_id: int) -> bool:
    """
    ПОЛНОСТЬЮ удаляет пользователя и все связанные данные из базы данных
    
    Args:
        user_id: ID пользователя для удаления
        
    Returns:
        bool: True если удаление успешно, False если пользователь не найден
    """
    async with get_db_connection() as db:
        try:
            # Проверяем, существует ли пользователь
            cursor = await db.execute(
                "SELECT id FROM users WHERE id = ?",
                (user_id,)
            )
            result = await cursor.fetchone()
            
            if not result:
                logger.warning(f"Пользователь с ID {user_id} не найден для удаления")
                return False
            
            logger.info(f"ПОЛНОЕ УДАЛЕНИЕ пользователя {user_id} и всех связанных данных")
            
            # Получаем все подписки пользователя
            cursor = await db.execute(
                "SELECT id FROM subscriptions WHERE user_id = ?",
                (user_id,)
            )
            subscription_ids = [row[0] for row in await cursor.fetchall()]
            
            # Удаляем в правильном порядке согласно зависимостям внешних ключей
            
            # 1. Сначала удаляем самые зависимые записи (которые ни на что не ссылаются)
            # Удаляем notification_log и notification_queue
            await db.execute(
                "DELETE FROM notification_log WHERE user_id = ?", 
                (user_id,)
            )
            await db.execute(
                "DELETE FROM notification_queue WHERE user_id = ?", 
                (user_id,)
            )
            logger.info(f"Удалены уведомления для пользователя {user_id}")
            
            # 2. Удаляем записи, которые ссылаются на подписки
            if subscription_ids:
                placeholders = ','.join('?' * len(subscription_ids))
                
                # Удаляем traffic_usage (ссылается на subscriptions)
                await db.execute(
                    f"DELETE FROM traffic_usage WHERE subscription_id IN ({placeholders})",
                    subscription_ids
                )
                logger.info(f"Удалены записи трафика для подписок: {subscription_ids}")
                
                # Удаляем payments по subscription_id (ссылается на subscriptions)
                await db.execute(
                    f"DELETE FROM payments WHERE subscription_id IN ({placeholders})",
                    subscription_ids
                )
                logger.info(f"Удалены платежи для подписок: {subscription_ids}")
            
            # 2.1. Удаляем ВСЕ платежи пользователя (включая те, что без подписок)
            await db.execute(
                "DELETE FROM payments WHERE user_id = ?",
                (user_id,)
            )
            logger.info(f"Удалены все платежи пользователя {user_id}")
            
            # 3. Удаляем подписки (ссылаются на users)
            await db.execute(
                "DELETE FROM subscriptions WHERE user_id = ?",
                (user_id,)
            )
            logger.info(f"Удалены все подписки пользователя {user_id}")
            
            # 4. Наконец удаляем пользователя (основная запись)
            await db.execute(
                "DELETE FROM users WHERE id = ?",
                (user_id,)
            )
            
            await db.commit()
            logger.info(f"Пользователь {user_id} полностью удален из базы данных со всеми связанными данными")
            return True
            
        except Exception as e:
            logger.error(f"Ошибка при полном удалении пользователя {user_id}: {e}")
            await db.rollback()
            raise


@db_error_handler
async def update_subscription(subscription: Subscription) -> Subscription:
    """Update an existing subscription"""
    async with get_db_connection() as db:
        try:
            # Проверяем ID подписки
            if not subscription.id:
                logger.error("Cannot update subscription without ID")
                raise ValueError("Subscription ID is required for update")
            
            logger.info(f"Updating subscription with ID {subscription.id}")
            
            # Выполняем обновление в базе данных
            await db.execute(
                """
                UPDATE subscriptions
                SET plan_id = ?, expires_at = ?, traffic_limit = ?, 
                    traffic_used = ?, vpn_user_id = ?, vpn_key = ?, status = ?, vpn_email = ?
                WHERE id = ?
                """,
                (
                    subscription.plan_id,
                    subscription.expires_at.isoformat() if subscription.expires_at else None,
                    subscription.traffic_limit, subscription.traffic_used,
                    subscription.vpn_user_id, subscription.vpn_key,
                    subscription.status, subscription.vpn_email, subscription.id
                )
            )
            
            # Проверяем, был ли выполнен запрос успешно
            if db.total_changes == 0:
                logger.warning(f"No subscription found with ID {subscription.id} to update")
            
            await db.commit()
            logger.debug(f"Successfully updated subscription {subscription.id}")
            return subscription
        except Exception as e:
            logger.error(f"Error updating subscription {subscription.id}: {e}")
            await db.rollback()
            raise


@db_error_handler
async def delete_subscription(subscription_id: int) -> bool:
    """Delete a subscription by ID with all related records"""
    async with get_db_connection() as db:
        try:
            logger.info(f"Deleting subscription with ID {subscription_id}")
            
            # Удаляем связанные записи в правильном порядке
            await db.execute("DELETE FROM payments WHERE subscription_id = ?", (subscription_id,))
            
            # Пропускаем notifications, если таблица не существует
            try:
                await db.execute("DELETE FROM notifications WHERE subscription_id = ?", (subscription_id,))
            except Exception:
                pass  # Таблица notifications может не существовать
            
            # Пропускаем traffic_usage, если таблица не существует
            try:
                await db.execute("DELETE FROM traffic_usage WHERE subscription_id = ?", (subscription_id,))
            except Exception:
                pass  # Таблица traffic_usage может не существовать
                
            await db.execute("DELETE FROM subscriptions WHERE id = ?", (subscription_id,))
            
            # Проверяем успешность удаления
            if db.total_changes == 0:
                logger.warning(f"No subscription found with ID {subscription_id} to delete")
                return False
            
            await db.commit()
            logger.info(f"Successfully deleted subscription {subscription_id} with all related records")
            return True
        except Exception as e:
            logger.error(f"Error deleting subscription {subscription_id}: {e}")
            await db.rollback()
            return False


@db_error_handler
async def get_all_subscriptions() -> List[Subscription]:
    """Get all subscriptions from the database"""
    async with get_db_connection() as db:
        try:
            cursor = await db.execute("SELECT * FROM subscriptions")
            rows = await cursor.fetchall()
            
            subscriptions = []
            for row in rows:
                try:
                    # Преобразуем строковые даты в объекты datetime
                    created_at = datetime.fromisoformat(row['created_at']) if row['created_at'] else None
                    expires_at = datetime.fromisoformat(row['expires_at']) if row['expires_at'] else None
                    
                    # Приводим значения трафика к float для совместимости
                    traffic_limit = float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0
                    traffic_used = float(row['traffic_used']) if row['traffic_used'] is not None else 0.0
                    
                    subscription = Subscription(
                        id=row['id'],
                        user_id=row['user_id'],
                        plan_id=row['plan_id'],
                        created_at=created_at,
                        expires_at=expires_at,
                        traffic_limit=traffic_limit,
                        traffic_used=traffic_used,
                        vpn_user_id=row['vpn_user_id'],
                        vpn_key=row['vpn_key'],
                        status=getattr(UserStatus, row['status']) if row['status'] else None,
                        vpn_email=row['vpn_email']
                    )
                    subscriptions.append(subscription)
                except Exception as e:
                    logger.error(f"Error parsing subscription data for row {row['id']}: {e}")
                    continue
            
            logger.debug(f"Retrieved {len(subscriptions)} subscriptions from the database")
            return subscriptions
        except Exception as e:
            logger.error(f"Error getting all subscriptions: {e}")
            return []


@db_error_handler
async def update_traffic_usage(subscription_id: int, traffic_used_bytes: float) -> None:
    """Update traffic usage for a subscription (traffic_used_bytes in bytes)"""
    async with get_db_connection() as db:
        try:
            traffic_used_gb = traffic_used_bytes / (1024**3)
            logger.debug(f"Updating traffic usage for subscription {subscription_id}: {traffic_used_gb:.3f} GB ({traffic_used_bytes} bytes)")
            
            # Получаем текущие данные о подписке для логирования
            cursor = await db.execute(
                "SELECT traffic_used, traffic_limit FROM subscriptions WHERE id = ?",
                (subscription_id,)
            )
            subscription_data = await cursor.fetchone()
            
            if subscription_data:
                old_usage_bytes = float(subscription_data['traffic_used']) if subscription_data['traffic_used'] is not None else 0
                old_usage_gb = old_usage_bytes / (1024**3)
                traffic_limit_bytes = float(subscription_data['traffic_limit']) if subscription_data['traffic_limit'] is not None else 0
                traffic_limit_gb = traffic_limit_bytes / (1024**3)
                
                # Обновляем использование трафика (сохраняем в байтах)
                await db.execute(
                    "UPDATE subscriptions SET traffic_used = ? WHERE id = ?",
                    (traffic_used_bytes, subscription_id)
                )
                
                # Логирование изменений
                if traffic_used_bytes > old_usage_bytes:
                    logger.info(f"Traffic usage increased for subscription {subscription_id}: {old_usage_gb:.2f} GB -> {traffic_used_gb:.2f} GB")
                    
                    # Если использование превышает лимит, добавляем предупреждение в логи
                    if traffic_limit_bytes > 0 and traffic_used_bytes > traffic_limit_bytes * 0.9:
                        percent_used = (traffic_used_bytes / traffic_limit_bytes) * 100
                        logger.warning(f"High traffic usage for subscription {subscription_id}: {traffic_used_gb:.2f}/{traffic_limit_gb:.2f} GB ({percent_used:.1f}%)")
            else:
                logger.warning(f"Subscription {subscription_id} not found while updating traffic usage")
                
            await db.commit()
        except Exception as e:
            logger.error(f"Error updating traffic usage for subscription {subscription_id}: {e}")
            await db.rollback()
            raise


@db_error_handler
def safe_user_status(status_value):
    """Безопасная обработка статуса пользователя"""
    if not status_value:
        return 'NEW'
    if isinstance(status_value, str) and status_value.isdigit():
        return 'NEW'
    if status_value not in ['NEW', 'ACTIVE', 'SUSPENDED', 'EXPIRED']:
        return 'NEW'
    return status_value

async def get_expiring_subscriptions(days: int) -> List[Tuple[User, Subscription]]:
    """Get subscriptions expiring in specified number of days"""
    logger.info(f"Поиск подписок, истекающих в течение {days} дней")
    async with get_db_connection() as db:
        try:
            # Используем алиасы для устранения конфликтов имен полей
            cursor = await db.execute(
                """
                SELECT 
                    u.id as user_id, u.username, u.first_name, u.last_name, 
                    u.created_at as user_created_at, u.status as user_status, u.phone_number,
                    s.id as subscription_id, s.plan_id, s.created_at as subscription_created_at,
                    s.expires_at, s.traffic_limit, s.traffic_used, s.vpn_user_id, s.vpn_key,
                    s.status as subscription_status, s.vpn_email
                FROM subscriptions s
                JOIN users u ON s.user_id = u.id
                WHERE datetime(s.expires_at) <= datetime('now', '+' || ? || ' days')
                AND datetime(s.expires_at) > datetime('now')
                AND (s.status IS NULL OR UPPER(s.status) = 'ACTIVE')
                ORDER BY s.expires_at ASC
                """,
                (days,)
            )
            rows = await cursor.fetchall()
            
            if not rows:
                logger.info(f"Не найдено подписок, истекающих в течение {days} дней")
                return []
                
            logger.info(f"Найдено {len(rows)} подписок, истекающих в течение {days} дней")
            
            result = []
            for row in rows:
                try:
                    # Создаем объект пользователя
                    user = User(
                        id=row['user_id'],
                        username=row['username'],
                        first_name=row['first_name'],
                        last_name=row['last_name'],
                        created_at=datetime.fromisoformat(row['user_created_at']) if row['user_created_at'] else datetime.now(),
                        status=safe_user_status(row['user_status']),
                        phone_number=row['phone_number']
                    )
                    
                    # Создаем объект подписки
                    subscription = Subscription(
                        id=row['subscription_id'],
                        user_id=row['user_id'],
                        plan_id=row['plan_id'],
                        created_at=datetime.fromisoformat(row['subscription_created_at']) if row['subscription_created_at'] else datetime.now(),
                        expires_at=datetime.fromisoformat(row['expires_at']) if row['expires_at'] else None,
                        traffic_limit=float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0,
                        traffic_used=float(row['traffic_used']) if row['traffic_used'] is not None else 0.0,
                        vpn_user_id=row['vpn_user_id'],
                        vpn_key=row['vpn_key'],
                        status=row['subscription_status'],
                        vpn_email=row['vpn_email']
                    )
                    
                    days_remaining = (subscription.expires_at - datetime.now()).days if subscription.expires_at else 0
                    logger.debug(f"Подписка {subscription.id} пользователя {user.id} истекает через {days_remaining} дней")
                    
                    result.append((user, subscription))
                except Exception as parse_error:
                    logger.error(f"Ошибка при обработке данных подписки: {parse_error}")
                    continue
            
            return result
        except Exception as e:
            logger.error(f"Ошибка при получении истекающих подписок: {e}")
            return []


@db_error_handler
async def get_low_traffic_subscriptions(threshold_gb: float) -> List[Tuple[User, Subscription]]:
    """
    Get subscriptions with low traffic remaining
    Args:
        threshold_gb: Threshold in GB (1.0 = 1 GB remaining)
    """
    threshold_bytes = threshold_gb * 1024 * 1024 * 1024
    logger.info(f"Поиск подписок с низким остатком трафика (порог: {threshold_gb} ГБ)")
    
    async with get_db_connection() as db:
        try:
            # Ищем подписки с остатком трафика меньше указанного порога
            cursor = await db.execute(
                """
                SELECT 
                    u.id as user_id, u.username, u.first_name, u.last_name, 
                    u.created_at as user_created_at, u.status as user_status, u.phone_number,
                    s.id as subscription_id, s.plan_id, s.created_at as subscription_created_at,
                    s.expires_at, s.traffic_limit, s.traffic_used, s.vpn_user_id, s.vpn_key,
                    s.status as subscription_status,
                    (s.traffic_limit - s.traffic_used) as traffic_remaining
                FROM subscriptions s
                JOIN users u ON s.user_id = u.id
                WHERE s.expires_at > datetime('now')
                AND s.traffic_limit > 0
                AND (s.traffic_limit - s.traffic_used) <= ?
                AND (s.status IS NULL OR UPPER(s.status) = 'ACTIVE')
                ORDER BY (s.traffic_limit - s.traffic_used) ASC
                """,
                (threshold_bytes,)
            )
            rows = await cursor.fetchall()
            
            if not rows:
                logger.info("Не найдено подписок с низким остатком трафика")
                return []
                
            logger.info(f"Найдено {len(rows)} подписок с низким остатком трафика")
            
            result = []
            for row in rows:
                try:
                    # Создаем объект пользователя
                    user = User(
                        id=row['user_id'],
                        username=row['username'],
                        first_name=row['first_name'],
                        last_name=row['last_name'],
                        created_at=datetime.fromisoformat(row['user_created_at']) if row['user_created_at'] else datetime.now(),
                        status=safe_user_status(row['user_status']),
                        phone_number=row['phone_number']
                    )
                    
                    # Создаем объект подписки
                    subscription = Subscription(
                        id=row['subscription_id'],
                        user_id=row['user_id'],
                        plan_id=row['plan_id'],
                        created_at=datetime.fromisoformat(row['subscription_created_at']) if row['subscription_created_at'] else datetime.now(),
                        expires_at=datetime.fromisoformat(row['expires_at']) if row['expires_at'] else None,
                        traffic_limit=float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0,
                        traffic_used=float(row['traffic_used']) if row['traffic_used'] is not None else 0.0,
                        vpn_user_id=row['vpn_user_id'],
                        vpn_key=row['vpn_key'],
                        status=row['subscription_status'],
                        vpn_email=row['vpn_email']
                    )
                    
                    # Логирование для отладки
                    remaining_gb = subscription.traffic_limit - subscription.traffic_used
                    percent_used = (subscription.traffic_used / subscription.traffic_limit) * 100 if subscription.traffic_limit > 0 else 0
                    logger.debug(
                        f"Подписка {subscription.id} пользователя {user.id} имеет {remaining_gb:.2f} ГБ "
                        f"оставшегося трафика ({percent_used:.1f}% использовано)"
                    )
                    
                    result.append((user, subscription))
                except Exception as parse_error:
                    logger.error(f"Ошибка при обработке данных подписки: {parse_error}")
                    continue
            
            return result
        except Exception as e:
            logger.error(f"Ошибка при получении подписок с низким остатком трафика: {e}")
            return []





# Payment operations
@db_error_handler
async def create_payment(payment: Payment) -> Payment:
    """Create a new payment record"""
    async with get_db_connection() as db:
        cursor = await db.execute(
            """
            INSERT INTO payments
            (user_id, subscription_id, amount, currency, payment_id, 
             created_at, completed_at, status, plan_id)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            RETURNING id
            """,
            (
                payment.user_id, payment.subscription_id,
                payment.amount, payment.currency, payment.payment_id,
                payment.created_at.isoformat(),
                payment.completed_at.isoformat() if payment.completed_at else None,
                payment.status.name, payment.plan_id
            )
        )
        result = await cursor.fetchone()
        if result:
            payment.id = result[0]
        else:
            raise Exception("Failed to get payment ID after creation")
        await db.commit()
        return payment


@db_error_handler
async def get_payment_by_payment_id(payment_id: str) -> Optional[Payment]:
    """Get payment by its payment_id"""
    async with get_db_connection() as db:
        db.row_factory = aiosqlite.Row
        cursor = await db.execute(
            "SELECT * FROM payments WHERE payment_id = ?",
            (payment_id,)
        )
        row = await cursor.fetchone()
        
        if row:
            return Payment(
                id=row['id'],
                user_id=row['user_id'],
                subscription_id=row['subscription_id'],
                amount=row['amount'],
                currency=row['currency'],
                payment_id=row['payment_id'],
                created_at=datetime.fromisoformat(row['created_at']),
                completed_at=datetime.fromisoformat(row['completed_at']) if row['completed_at'] else None,
                status=PaymentStatus[row['status']],
                plan_id=row['plan_id']
            )
        return None


@db_error_handler
async def update_payment_status(payment_id: str, status: PaymentStatus, completed_at: datetime = None) -> None:
    """Update payment status"""
    async with get_db_connection() as db:
        await db.execute(
            "UPDATE payments SET status = ?, completed_at = ? WHERE payment_id = ?",
            (status.name, completed_at.isoformat() if completed_at else None, payment_id)
        )
        await db.commit()


@db_error_handler
async def get_pending_payments() -> List[Payment]:
    """Get all pending payments"""
    async with get_db_connection() as db:
        db.row_factory = aiosqlite.Row
        cursor = await db.execute(
            "SELECT * FROM payments WHERE status = ?",
            (PaymentStatus.PENDING.name,)
        )
        rows = await cursor.fetchall()
        
        return [
            Payment(
                id=row['id'],
                user_id=row['user_id'],
                subscription_id=row['subscription_id'],
                amount=row['amount'],
                currency=row['currency'],
                payment_id=row['payment_id'],
                created_at=datetime.fromisoformat(row['created_at']),
                completed_at=datetime.fromisoformat(row['completed_at']) if row['completed_at'] else None,
                status=PaymentStatus[row['status']],
                plan_id=row['plan_id']
            )
            for row in rows
        ]
