from datetime import datetime, timedelta
from enum import Enum, auto


class UserStatus(Enum):
    NEW = auto()
    ACTIVE = auto()
    SUSPENDED = auto()
    EXPIRED = auto()


class PaymentStatus(Enum):
    PENDING = auto()
    SUCCEEDED = auto()
    CANCELED = auto()
    FAILED = auto()


class User:
    def __init__(self, id, username=None, first_name=None, last_name=None, 
                 created_at=None, status=None, phone_number=None, vpn_email=None):
        self.id = id
        self.username = username
        self.first_name = first_name
        self.last_name = last_name
        self.created_at = created_at or datetime.now()
        self.status = status or UserStatus.NEW
        self.phone_number = phone_number
        self.vpn_email = vpn_email

    @property
    def full_name(self):
        if self.first_name and self.last_name:
            return f"{self.first_name} {self.last_name}"
        elif self.first_name:
            return self.first_name
        elif self.username:
            return self.username
        else:
            return f"User {self.id}"
        

class Subscription:
    def __init__(self, id=None, user_id=None, plan_id=None, created_at=None, 
                 expires_at=None, traffic_limit=0, traffic_used=0, 
                 vpn_user_id=None, vpn_key=None, status=None, vpn_email=None):
        self.id = id
        self.user_id = user_id
        self.plan_id = plan_id
        self.created_at = created_at or datetime.now()
        self.expires_at = expires_at
        self.traffic_limit = float(traffic_limit) if traffic_limit is not None else 0.0
        self.traffic_used = float(traffic_used) if traffic_used is not None else 0.0
        self.vpn_user_id = vpn_user_id
        self.vpn_key = vpn_key
        self.status = status
        self.vpn_email = vpn_email  # Email клиента в панели для синхронизации
        
    @property
    def traffic_remaining(self):
        return max(0, self.traffic_limit - self.traffic_used)
    
    @property
    def traffic_percent_used(self):
        if self.traffic_limit == 0:
            return 100
        return (self.traffic_used / self.traffic_limit) * 100
    
    @property
    def is_active(self):
        return (self.expires_at > datetime.now() and 
                self.traffic_remaining > 0)
    
    @property
    def days_remaining(self):
        if not self.expires_at:
            return 0
        delta = self.expires_at - datetime.now()
        return max(0, delta.days)


class Payment:
    def __init__(self, id=None, user_id=None, subscription_id=None,
                 amount=0, currency="RUB", payment_id=None, 
                 created_at=None, completed_at=None, 
                 status=PaymentStatus.PENDING, plan_id=None):
        self.id = id
        self.user_id = user_id
        self.subscription_id = subscription_id
        self.amount = amount
        self.currency = currency
        self.payment_id = payment_id
        self.created_at = created_at or datetime.now()
        self.completed_at = completed_at
        self.status = status
        self.plan_id = plan_id
