"""
Фабрика для создания объектов Subscription с гарантированной целостностью данных
"""
from typing import Optional, Union
from datetime import datetime

from database.models import Subscription


def create_subscription_object(
    user_id: int,
    plan_id: str,
    expires_at: datetime,
    traffic_limit: Union[int, float],
    traffic_used: Union[int, float] = 0,
    vpn_user_id: Optional[int] = None,
    vpn_key: Optional[str] = None,
    status: str = "ACTIVE",
    vpn_email: Optional[str] = None,
    created_at: Optional[datetime] = None,
    subscription_id: Optional[int] = None
) -> Subscription:
    """
    Создает объект Subscription с гарантированным включением всех обязательных полей.
    
    Args:
        user_id: ID пользователя
        plan_id: ID тарифного плана
        expires_at: Дата истечения подписки
        traffic_limit: Лимит трафика (в байтах или ГБ, будет преобразован)
        traffic_used: Использованный трафик (в байтах или ГБ, будет преобразован)
        vpn_user_id: ID клиента в VPN панели
        vpn_key: VPN ключ
        status: Статус подписки
        vpn_email: Email для VPN клиента
        created_at: Дата создания
        subscription_id: ID подписки (для существующих записей)
    
    Returns:
        Subscription: Созданный объект подписки
    
    Raises:
        ValueError: Если отсутствуют обязательные поля
    """
    # Проверяем обязательные поля
    if not user_id:
        raise ValueError("user_id является обязательным полем")
    
    if not plan_id:
        raise ValueError("plan_id является обязательным полем")
    
    if not expires_at:
        raise ValueError("expires_at является обязательным полем")
    
    # Нормализуем трафик (если передан в ГБ, конвертируем в байты)
    if isinstance(traffic_limit, (int, float)) and traffic_limit < 1000:
        # Предполагаем, что это ГБ, конвертируем в байты
        traffic_limit = traffic_limit * (1024**3)
    
    if isinstance(traffic_used, (int, float)) and traffic_used < 1000:
        # Предполагаем, что это ГБ, конвертируем в байты
        traffic_used = traffic_used * (1024**3)
    
    # Устанавливаем значения по умолчанию
    if created_at is None:
        created_at = datetime.now()
    
    # Создаем объект подписки
    subscription = Subscription(
        id=subscription_id,
        user_id=user_id,
        plan_id=plan_id,
        created_at=created_at,
        expires_at=expires_at,
        traffic_limit=float(traffic_limit),
        traffic_used=float(traffic_used),
        vpn_user_id=vpn_user_id,
        vpn_key=vpn_key,
        status=status,
        vpn_email=vpn_email
    )
    
    return subscription


def create_trial_subscription(
    user_id: int,
    vpn_user_id: int,
    vpn_email: str,
    days: int = 3,
    traffic_gb: float = 30
) -> Subscription:
    """
    Создает пробную подписку с предустановленными параметрами.
    
    Args:
        user_id: ID пользователя
        vpn_user_id: ID клиента в VPN панели
        vpn_email: Email для VPN клиента
        days: Количество дней действия (по умолчанию 3)
        traffic_gb: Лимит трафика в ГБ (по умолчанию 30)
    
    Returns:
        Subscription: Созданная пробная подписка
    """
    from datetime import timedelta
    
    expires_at = datetime.now() + timedelta(days=days)
    
    return create_subscription_object(
        user_id=user_id,
        plan_id="trial",
        expires_at=expires_at,
        traffic_limit=traffic_gb,
        traffic_used=0,
        vpn_user_id=vpn_user_id,
        vpn_key=None,
        status="ACTIVE",
        vpn_email=vpn_email
    )


def create_paid_subscription(
    user_id: int,
    plan_id: str,
    vpn_user_id: int,
    vpn_email: str,
    days: int,
    traffic_gb: float
) -> Subscription:
    """
    Создает платную подписку с указанными параметрами.
    
    Args:
        user_id: ID пользователя
        plan_id: ID тарифного плана
        vpn_user_id: ID клиента в VPN панели
        vpn_email: Email для VPN клиента
        days: Количество дней действия
        traffic_gb: Лимит трафика в ГБ
    
    Returns:
        Subscription: Созданная платная подписка
    """
    from datetime import timedelta
    
    expires_at = datetime.now() + timedelta(days=days)
    
    return create_subscription_object(
        user_id=user_id,
        plan_id=plan_id,
        expires_at=expires_at,
        traffic_limit=traffic_gb,
        traffic_used=0,
        vpn_user_id=vpn_user_id,
        vpn_key=None,
        status="ACTIVE",
        vpn_email=vpn_email
    )


def create_subscription_from_db_row(row) -> Subscription:
    """
    Создает объект Subscription из записи базы данных.
    Гарантирует, что все поля будут включены.
    
    Args:
        row: Запись из базы данных (dict-like объект)
    
    Returns:
        Subscription: Созданный объект подписки
    """
    # Обрабатываем даты
    created_at = None
    if row.get('created_at'):
        created_at = datetime.fromisoformat(row['created_at'])
    
    expires_at = None
    if row.get('expires_at'):
        expires_at = datetime.fromisoformat(row['expires_at'])
    
    return create_subscription_object(
        subscription_id=row.get('id'),
        user_id=row['user_id'],
        plan_id=row['plan_id'],
        created_at=created_at,
        expires_at=expires_at,
        traffic_limit=float(row['traffic_limit']) if row['traffic_limit'] is not None else 0.0,
        traffic_used=float(row['traffic_used']) if row['traffic_used'] is not None else 0.0,
        vpn_user_id=row.get('vpn_user_id'),
        vpn_key=row.get('vpn_key'),
        status=row.get('status', 'ACTIVE'),
        vpn_email=row.get('vpn_email')
    )