"""
Модуль для обновления подписок с автоматическими уведомлениями
"""
import logging
from typing import Optional
from datetime import datetime, timedelta
from aiogram import Bot

from bot.database.operations import (
    get_user_subscription, update_subscription, get_user
)
from bot.database.models import Subscription
from bot.services.subscription_notifier import SubscriptionNotifier

logger = logging.getLogger(__name__)


class SubscriptionUpdater:
    """Класс для обновления подписок с уведомлениями"""
    
    def __init__(self, bot: Bot):
        self.bot = bot
        self.notifier = SubscriptionNotifier(bot)
    
    async def upgrade_subscription(self, user_id: int, new_plan: str, 
                                 days: int, traffic_gb: float) -> bool:
        """
        Обновляет подписку пользователя на новый тариф
        
        Args:
            user_id: ID пользователя
            new_plan: Новый тарифный план
            days: Количество дней действия
            traffic_gb: Лимит трафика в ГБ
            
        Returns:
            bool: True если обновление прошло успешно
        """
        try:
            # Получаем текущую подписку
            current_subscription = await get_user_subscription(user_id)
            
            if not current_subscription:
                logger.error(f"Не найдена текущая подписка для пользователя {user_id}")
                return False
            
            old_plan = current_subscription.plan_id
            
            # Обновляем подписку
            current_subscription.plan_id = new_plan
            current_subscription.expires_at = datetime.now() + timedelta(days=days)
            current_subscription.traffic_limit = traffic_gb * (1024**3)  # Конвертируем в байты
            
            # Сохраняем изменения
            success = await update_subscription(current_subscription)
            
            if success:
                # Отправляем уведомление
                await self.notifier.notify_subscription_upgraded(
                    user_id=user_id,
                    old_plan=old_plan,
                    new_plan=new_plan
                )
                
                logger.info(f"Подписка пользователя {user_id} обновлена с {old_plan} на {new_plan}")
                return True
            else:
                logger.error(f"Не удалось обновить подписку пользователя {user_id}")
                return False
                
        except Exception as e:
            logger.error(f"Ошибка при обновлении подписки пользователя {user_id}: {e}")
            return False
    
    async def extend_subscription(self, user_id: int, additional_days: int) -> bool:
        """
        Продлевает существующую подписку
        
        Args:
            user_id: ID пользователя
            additional_days: Количество дней для продления
            
        Returns:
            bool: True если продление прошло успешно
        """
        try:
            # Получаем текущую подписку
            current_subscription = await get_user_subscription(user_id)
            
            if not current_subscription:
                logger.error(f"Не найдена текущая подписка для пользователя {user_id}")
                return False
            
            # Продлеваем подписку
            if current_subscription.expires_at:
                # Если подписка еще действует, добавляем дни к текущей дате истечения
                if current_subscription.expires_at > datetime.now():
                    current_subscription.expires_at += timedelta(days=additional_days)
                else:
                    # Если подписка истекла, продлеваем с текущего момента
                    current_subscription.expires_at = datetime.now() + timedelta(days=additional_days)
            else:
                current_subscription.expires_at = datetime.now() + timedelta(days=additional_days)
            
            # Сохраняем изменения
            success = await update_subscription(current_subscription)
            
            if success:
                # Отправляем уведомление
                await self.notifier.notify_subscription_extended(
                    user_id=user_id,
                    additional_days=additional_days
                )
                
                logger.info(f"Подписка пользователя {user_id} продлена на {additional_days} дней")
                return True
            else:
                logger.error(f"Не удалось продлить подписку пользователя {user_id}")
                return False
                
        except Exception as e:
            logger.error(f"Ошибка при продлении подписки пользователя {user_id}: {e}")
            return False
    
    async def add_traffic(self, user_id: int, additional_gb: float) -> bool:
        """
        Добавляет дополнительный трафик к подписке
        
        Args:
            user_id: ID пользователя
            additional_gb: Количество дополнительных ГБ
            
        Returns:
            bool: True если добавление прошло успешно
        """
        try:
            # Получаем текущую подписку
            current_subscription = await get_user_subscription(user_id)
            
            if not current_subscription:
                logger.error(f"Не найдена текущая подписка для пользователя {user_id}")
                return False
            
            # Добавляем трафик
            additional_bytes = additional_gb * (1024**3)
            current_subscription.traffic_limit += additional_bytes
            
            # Сохраняем изменения
            success = await update_subscription(current_subscription)
            
            if success:
                # Отправляем уведомление
                await self.notifier.notify_traffic_added(
                    user_id=user_id,
                    additional_gb=additional_gb
                )
                
                logger.info(f"К подписке пользователя {user_id} добавлено {additional_gb} ГБ трафика")
                return True
            else:
                logger.error(f"Не удалось добавить трафик пользователю {user_id}")
                return False
                
        except Exception as e:
            logger.error(f"Ошибка при добавлении трафика пользователю {user_id}: {e}")
            return False
    
    async def process_payment_upgrade(self, user_id: int, plan_data: dict) -> bool:
        """
        Обрабатывает успешную оплату и обновляет подписку
        
        Args:
            user_id: ID пользователя
            plan_data: Данные о тарифном плане
            
        Returns:
            bool: True если обработка прошла успешно
        """
        try:
            plan_id = plan_data.get('id')
            days = plan_data.get('days', 30)
            traffic_gb = plan_data.get('traffic_gb', 100)
            
            # Определяем тип операции
            current_subscription = await get_user_subscription(user_id)
            
            if not current_subscription:
                logger.error(f"Не найдена подписка для обработки платежа пользователя {user_id}")
                return False
            
            # Если это тот же план - продлеваем, если другой - обновляем
            if current_subscription.plan_id == plan_id:
                # Продление существующего плана
                return await self.extend_subscription(user_id, days)
            else:
                # Обновление на новый план
                return await self.upgrade_subscription(user_id, plan_id, days, traffic_gb)
                
        except Exception as e:
            logger.error(f"Ошибка при обработке платежа для пользователя {user_id}: {e}")
            return False