"""
Модуль для синхронизации данных с панелью 3X-UI
"""

import logging
import json
from typing import Optional
from bot.services.working_panel_api import WorkingPanelAPI
from config_keys import get_panel_credentials

logger = logging.getLogger(__name__)


async def ensure_panel_authorization(panel_api: WorkingPanelAPI, context: str = "операции") -> bool:
    """Обеспечивает авторизацию в панели с единообразной обработкой ошибок"""
    logger.info(f"Авторизация в панели для {context}...")
    if not await panel_api.login():
        logger.error(f"Не удалось авторизоваться в панели для {context}")
        return False
    return True





async def update_client_in_panel(
    client_id: str, traffic_limit: int, expiry_time: int, reset_traffic: bool = False,
    user_id: int = 0, subscription_id: int = 0
) -> bool:
    """
    Обновляет клиента в панели 3X-UI с улучшенным поиском

    Args:
        client_id: ID клиента (может быть UUID или email)
        traffic_limit: Лимит трафика в байтах
        expiry_time: Время истечения в миллисекундах
        reset_traffic: Сбросить использованный трафик
        user_id: ID пользователя в Telegram (для поля tgId)
        subscription_id: ID подписки в базе данных (для поля subId)

    Returns:
        bool: Успешность обновления
    """
    panel_api = None
    try:
        # Инициализируем API панели
        panel_url, panel_username, panel_password = get_panel_credentials()
        panel_api = WorkingPanelAPI(panel_url, panel_username, panel_password)

        # Авторизуемся и получаем свежие cookies
        if not await ensure_panel_authorization(panel_api, "обновления клиента"):
            return False

        # Принудительно обновляем сессию после логина
        session = await panel_api._get_session()

        # Получаем данные inbound
        # Используем правильный метод API для формирования URL
        inbound_url = panel_api._get_api_url("inbounds/get/1")
        logger.info(f"Запрос данных inbound: {inbound_url}")

        headers = {"Accept": "application/json", "Content-Type": "application/json"}

        async with session.get(inbound_url, headers=headers) as response:
            logger.info(f"Статус ответа панели: {response.status}")
            logger.info(f"Заголовки ответа: {dict(response.headers)}")

            if response.status == 200:
                try:
                    data = await response.json()
                    logger.info(
                        f"Получены JSON данные: {json.dumps(data, ensure_ascii=False, indent=2)[:500]}..."
                    )
                except Exception as e:
                    # Если не JSON, пробуем получить текст и проверить на успех
                    text_response = await response.text()
                    logger.error(f"Ошибка парсинга JSON: {e}")
                    logger.warning(
                        f"Панель вернула text/plain вместо JSON: '{text_response[:200]}'"
                    )

                    # Проверяем, может ли текстовый ответ содержать полезную информацию
                    if text_response.strip():
                        logger.info(f"Полный текстовый ответ: {text_response}")

                    if '"success":true' in text_response or "success: true" in text_response:
                        logger.info("Операция успешна согласно текстовому ответу")
                        return True
                    return False
            else:
                logger.error(f"HTTP ошибка при получении данных inbound: {response.status}")
                text_response = await response.text()
                logger.error(f"Тело ответа при ошибке: {text_response}")
                return False

        if data.get("success"):
            obj = data.get("obj", {})
            settings_str = obj.get("settings", "{}")
            settings = json.loads(settings_str)
            clients = settings.get("clients", [])
            
            # Проверяем, есть ли clientStats с реальной статистикой
            client_stats = obj.get("clientStats", None)
            if client_stats:
                logger.info(f"Обнаружена статистика клиентов: {json.dumps(client_stats, ensure_ascii=False, indent=2)}")
            else:
                logger.info("clientStats отсутствует или пуст")

            # Ищем клиента по UUID или по email
            target_client = None
            client_found_by = None

            # Сначала ищем по UUID
            for client in clients:
                if client.get("id") == client_id:
                    target_client = client
                    client_found_by = "UUID"
                    break

            # Если не найден по UUID, ищем по email (предполагая что client_id это email)
            if not target_client and "@" in client_id:
                for client in clients:
                    if client.get("email") == client_id:
                        target_client = client
                        client_found_by = "email"
                        break

            # Если не найден по email, пробуем найти по стандартному формату email
            if not target_client:
                # Извлекаем user_id из client_id если это UUID
                for client in clients:
                    client_email = client.get("email", "")
                    # Ищем по части email (username_XXXXX@razdvavpn.ru)
                    if client_email.endswith("@razdvavpn.ru"):
                        target_client = client
                        client_found_by = "email_pattern"
                        break

            if not target_client:
                logger.error(f"Клиент не найден в панели: {client_id}")
                return False

            logger.info(
                f"Клиент найден в панели по {client_found_by}: {target_client.get('email')}"
            )

            # Формируем клиента согласно структуре библиотеки (убираем лишние поля)
            clean_client = {
                "id": target_client["id"],  # UUID клиента
                "email": target_client["email"],
                "enable": target_client.get("enable", True),
                "totalGB": traffic_limit,
                "expiryTime": expiry_time,
                "tgId": str(user_id) if user_id else str(target_client.get("tgId", "")),  # ID Telegram пользователя
                "subId": str(subscription_id) if subscription_id else str(target_client.get("subId", "")),  # ID подписки из БД
                "limitIp": 5  # Устанавливаем ограничение на 5 IP по умолчанию
            }
            
            # Добавляем дополнительные поля если они есть в оригинале
            if "flow" in target_client:
                clean_client["flow"] = target_client["flow"]
            if "alterId" in target_client:
                clean_client["alterId"] = target_client["alterId"]
            if "limitIp" in target_client:
                clean_client["limitIp"] = target_client["limitIp"]

            # КРИТИЧНО: Сбрасываем трафик при reset_traffic=True
            if reset_traffic:
                clean_client["reset"] = 1  # reset=1 ОБНУЛЯЕТ трафик, reset=0 оставляет без изменений
                logger.info(f"Сброс трафика для клиента {target_client['email']} через reset=1")

            # КРИТИЧНО: Передаем только ОДИН обновляемый клиент согласно правилу
            # Формат из рабочего примера: flow="xtls-rprx-vision", subId как строка ID подписки
            single_client_data = {
                "id": target_client["id"],  # UUID клиента
                "email": target_client["email"],
                "totalGB": traffic_limit,  # Лимит в байтах
                "expiryTime": expiry_time,  # Время в миллисекундах
                "enable": True,
                "flow": "xtls-rprx-vision",  # ИСПРАВЛЕНО: правильное значение flow из примера
                "limitIp": 5,  # Ограничение на 5 IP по умолчанию
                "subId": str(subscription_id) if subscription_id else str(target_client.get("subId", "")),  # ID подписки как строка
                "reset": 1 if reset_traffic else 0,  # reset=1 обнуляет трафик
                "tgId": str(user_id) if user_id else str(target_client.get("tgId", ""))  # ID Telegram пользователя из БД
            }
            
            # Создаем настройки с одним клиентом в JSON-строке
            settings_json = json.dumps({"clients": [single_client_data]})

            # Отправляем обновление - используем UUID клиента в URL
            client_uuid = target_client["id"]
            update_url = panel_api._get_api_url(f"inbounds/updateClient/{client_uuid}")

            # Данные для обновления согласно критическому правилу
            update_data = {
                "id": 1,  # ID inbound для VLESS
                "settings": settings_json  # JSON-строка с массивом клиентов
            }

            logger.info(f"Обновление клиента в панели: {update_url}")
            logger.info(
                f"Данные обновления: {json.dumps(update_data, ensure_ascii=False, indent=2)}"
            )

            # Принудительно обновляем авторизацию перед критическим запросом
            if not await ensure_panel_authorization(panel_api, "обновления клиента (критический запрос)"):
                return False

            # Получаем обновленную сессию
            session = await panel_api._get_session()

            # Добавляем заголовки для запроса
            headers = {"Content-Type": "application/json", "Accept": "application/json"}

            async with session.post(
                update_url, json=update_data, headers=headers
            ) as update_response:
                if update_response.status == 200:
                    try:
                        update_result = await update_response.json()
                        if update_result.get("success"):
                            logger.info(
                                f"Клиент успешно обновлен в панели: {target_client.get('email')}"
                            )
                            return True
                        else:
                            logger.error(
                                f"Панель вернула ошибку при обновлении: {update_result.get('msg')}"
                            )
                            return False
                    except Exception:
                        # Если не JSON, проверяем текстовый ответ
                        text_response = await update_response.text()
                        logger.info(f"Панель вернула текстовый ответ: {text_response[:200]}")
                        if (
                            '"success":true' in text_response
                            or "success: true" in text_response
                            or text_response.strip() == "true"
                        ):
                            logger.info(
                                f"Клиент успешно обновлен в панели (текстовый ответ): {target_client.get('email')}"
                            )
                            return True
                        else:
                            logger.error(
                                f"Неожиданный ответ панели при обновлении: {text_response}"
                            )
                            return False
                else:
                    logger.error(f"HTTP ошибка при обновлении клиента: {update_response.status}")
                    return False
        else:
            logger.error("Панель вернула ошибку при получении данных inbound")
            return False

    except Exception as e:
        logger.error(f"Ошибка при обновлении клиента в панели: {e}")
        return False

    finally:
        try:
            if panel_api is not None:
                await panel_api.close()
        except:
            pass


async def sync_subscription_with_panel(subscription) -> bool:
    """
    Синхронизирует подписку с панелью

    Args:
        subscription: Объект подписки

    Returns:
        bool: Успешность синхронизации
    """
    try:
        # Формируем email пользователя в новом формате
        # Используем сохраненный email из подписки или генерируем новый
        if subscription.vpn_email and '@' in subscription.vpn_email and '@razdvavpn.ru' not in subscription.vpn_email:
            user_email = subscription.vpn_email
        else:
            # Генерируем в новом формате user_id@username
            user_email = f"{subscription.user_id}@user"

        # Вычисляем лимит трафика в байтах
        traffic_limit_bytes = subscription.traffic_limit * 1024 * 1024 * 1024

        # Вычисляем время истечения в миллисекундах
        expiry_timestamp = int(subscription.expires_at.timestamp() * 1000)

        # Пробуем обновить по UUID, затем по email
        updated_by_uuid = await update_client_in_panel(
            subscription.vpn_user_id, traffic_limit_bytes, expiry_timestamp, reset_traffic=False
        )

        if updated_by_uuid:
            logger.info(f"Подписка синхронизирована по UUID: {subscription.vpn_user_id}")
            return True

        # Если обновление по UUID не удалось, пробуем по email
        updated_by_email = await update_client_in_panel(
            user_email, traffic_limit_bytes, expiry_timestamp, reset_traffic=False
        )

        if updated_by_email:
            logger.info(f"Подписка синхронизирована по email: {user_email}")
            return True

        logger.error(
            f"Не удалось синхронизировать подписку для пользователя {subscription.user_id}"
        )
        return False

    except Exception as e:
        logger.error(f"Ошибка синхронизации подписки: {e}")
        return False
