"""
Оптимизированный модуль для работы с API панели 3X-UI.
Содержит все необходимые функции для работы с протоколом VLESS.
"""

import json
import asyncio
import logging
import ssl
import base64
import aiohttp
import urllib.parse
from typing import Dict, List, Any, Optional, Tuple
from datetime import datetime

from config import PANEL_BASE_URL, PANEL_PATH, PANEL_USERNAME, PANEL_PASSWORD

# Настройка логирования
logger = logging.getLogger(__name__)


class PanelApiError(Exception):
    """Исключение при ошибках работы с API панели"""
    pass


class PanelApi:
    """
    Оптимизированный API клиент для работы с панелью 3X-UI
    """

    def __init__(self):
        """Инициализация клиента API"""
        self.base_url = f"{PANEL_BASE_URL}{PANEL_PATH}".rstrip('/')
        self.username = PANEL_USERNAME
        self.password = PANEL_PASSWORD
        self.session = None
        self.last_login = 0
        self.login_expiry = 600  # 10 минут
        self.connected = False
        self.connection_error = None
        self.last_connection_attempt = 0

    def _create_ssl_context(self):
        """Создание SSL контекста без проверки сертификата"""
        ssl_context = ssl.create_default_context()
        ssl_context.check_hostname = False
        ssl_context.verify_mode = ssl.CERT_NONE
        return ssl_context

    def _create_connector(self, timeout_seconds=30):
        """Создание TCP connector с SSL контекстом"""
        ssl_context = self._create_ssl_context()
        connector = aiohttp.TCPConnector(ssl=ssl_context)
        timeout = aiohttp.ClientTimeout(total=timeout_seconds)
        return connector, timeout

    def _parse_json_response(self, content_text: str, email: str, protocol: str) -> Optional[str]:
        """Парсинг JSON ответа для поиска VPN ссылки"""
        try:
            # Попытка распарсить как JSON
            json_data = json.loads(content_text)
            
            # Ищем нужную ссылку в JSON
            if isinstance(json_data, list):
                for item in json_data:
                    if isinstance(item, str) and email.lower() in item.lower():
                        if item.startswith(f"{protocol}://"):
                            logger.info(f"Найдена ссылка для {email} в JSON URL")
                            return item
                    elif isinstance(item, dict):
                        if "ps" in item and email.lower() in item.get("ps", "").lower():
                            if "link" in item and item["link"].startswith(f"{protocol}://"):
                                logger.info(f"Найдена ссылка для {email} в JSON URL")
                                return item["link"]
            
            # Также проверяем в тексте, могут быть ссылки не в JSON формате
            if f"{protocol}://" in content_text:
                lines = content_text.split("\n")
                for line in lines:
                    if f"{protocol}://" in line and email.lower() in line.lower():
                        # Извлекаем ссылку из строки
                        start_idx = line.find(f"{protocol}://")
                        end_idx = line.find(" ", start_idx)
                        if end_idx == -1:
                            end_idx = len(line)
                        extracted_link = line[start_idx:end_idx].strip()
                        logger.info(f"Найдена ссылка в тексте ответа: {extracted_link}")
                        return extracted_link
        except Exception as e:
            logger.warning(f"Ошибка при парсинге JSON: {e}")
        
        return None

    async def _get_session(self) -> aiohttp.ClientSession:
        """
        Получение сессии для работы с API
        
        Returns:
            aiohttp.ClientSession: Сессия с необходимыми параметрами
        """
        if self.session is None:
            connector, timeout = self._create_connector(30)
            
            self.session = aiohttp.ClientSession(
                connector=connector,
                timeout=timeout
            )
        
        return self.session

    async def login(self, force: bool = False) -> bool:
        """
        Аутентификация в панели 3X-UI
        
        Args:
            force: Принудительная повторная аутентификация
            
        Returns:
            bool: Успешность аутентификации
        """
        try:
            # Проверяем, нужно ли выполнять повторную аутентификацию
            current_time = datetime.now().timestamp()
            self.last_connection_attempt = current_time
            
            if not force and self.connected and (current_time - self.last_login) < self.login_expiry:
                logger.info("Используем существующую аутентификацию")
                return True
            
            logger.info(f"Аутентификация в панели 3X-UI: {self.base_url}")
            
            session = await self._get_session()
            
            # Пробуем разные методы аутентификации
            login_url = f"{self.base_url}/login"
            
            # Метод 1: Форма
            try:
                form_data = aiohttp.FormData()
                form_data.add_field("username", self.username)
                form_data.add_field("password", self.password)
                
                async with session.post(login_url, data=form_data, allow_redirects=True) as response:
                    if response.status == 200:
                        # Проверка на успешную аутентификацию
                        cookies = session.cookie_jar.filter_cookies(response.url)
                        if len(cookies) > 0:
                            self.last_login = current_time
                            self.connected = True
                            self.connection_error = None
                            logger.info("Аутентификация успешна (метод формы)")
                            return True
                        
                        content = await response.text()
                        if "login-title" not in content and "login-form" not in content:
                            self.last_login = current_time
                            self.connected = True
                            self.connection_error = None
                            logger.info("Аутентификация успешна (проверка по содержимому)")
                            return True
            except Exception as e:
                logger.warning(f"Ошибка при аутентификации через форму: {e}")
            
            # Метод 2: JSON
            try:
                credentials = {
                    "username": self.username,
                    "password": self.password
                }
                
                async with session.post(login_url, json=credentials) as response:
                    if response.status == 200:
                        self.last_login = current_time
                        self.connected = True
                        self.connection_error = None
                        logger.info("Аутентификация успешна (метод JSON)")
                        return True
            except Exception as e:
                logger.warning(f"Ошибка при аутентификации через JSON: {e}")
            
            # Если все методы не сработали
            self.connected = False
            self.connection_error = "Не удалось аутентифицироваться ни одним из методов"
            logger.error(self.connection_error)
            return False
        
        except Exception as e:
            logger.error(f"Ошибка при аутентификации: {e}")
            self.connected = False
            self.connection_error = f"Ошибка при аутентификации: {e}"
            return False

    async def get_client_link_via_subscription(self, email: str, protocol: str = "vless") -> Optional[str]:
        """
        Получить ссылку подключения для клиента через Subscription URL
        
        Args:
            email: Email клиента
            protocol: Протокол (vless или shadowsocks)
            
        Returns:
            Optional[str]: Ссылка для подключения или None если не найдено
        """
        try:
            logger.info(f"Поиск ссылки для {email} через Subscription URL...")
            
            # Короткая задержка для синхронизации данных
            logger.info("Короткая задержка для синхронизации данных панели...")
            await asyncio.sleep(1)
            
            # Создание соединения с коротким таймаутом
            connector, timeout = self._create_connector(10)
            
            direct_link = None
            
            for inbound_id in [1, 2, 3]:  # Проверяем все возможные inbound_id
                # Настраиваем URL
                sub_url = f"http://31.172.75.92:2096/subavto/{inbound_id}"
                json_url = f"http://31.172.75.92:2096/jsonauto/{inbound_id}"
                
                async with aiohttp.ClientSession(connector=connector, timeout=timeout) as session:
                    # Пробуем получить из JSON URL
                    try:
                        async with session.get(json_url) as response:
                            if response.status == 200:
                                try:
                                    # Сначала получаем текст ответа, чтобы проверить его формат
                                    content_text = await response.text()
                                    
                                    # Проверяем, если ответ не является валидным JSON, попытаемся его обработать
                                    if content_text and len(content_text) > 0:
                                        logger.info(f"Получен ответ для inbound {inbound_id}, размер: {len(content_text)} байт")
                                        
                                        # Попытка распарсить как JSON
                                        try:
                                            json_data = json.loads(content_text)
                                            
                                            # Ищем нужную ссылку в JSON
                                            if isinstance(json_data, list):
                                                for item in json_data:
                                                    if isinstance(item, str) and email.lower() in item.lower():
                                                        if item.startswith(f"{protocol}://"):
                                                            logger.info(f"Найдена ссылка для {email} в JSON URL")
                                                            return item
                                                    elif isinstance(item, dict):
                                                        if "ps" in item and email.lower() in item.get("ps", "").lower():
                                                            if "link" in item and item["link"].startswith(f"{protocol}://"):
                                                                logger.info(f"Найдена ссылка для {email} в JSON URL")
                                                                return item["link"]
                                            
                                            # Также проверяем в тексте, могут быть ссылки не в JSON формате
                                            if f"{protocol}://" in content_text:
                                                lines = content_text.split("\n")
                                                for line in lines:
                                                    if f"{protocol}://" in line and email.lower() in line.lower():
                                                        # Извлекаем ссылку из строки
                                                        start_idx = line.find(f"{protocol}://")
                                                        end_idx = line.find(" ", start_idx)
                                                        if end_idx == -1:
                                                            end_idx = len(line)
                                                        extracted_link = line[start_idx:end_idx].strip()
                                                        logger.info(f"Найдена ссылка в тексте ответа: {extracted_link}")
                                                        return extracted_link
                                                
                                        except json.JSONDecodeError:
                                            # Если не JSON, проверяем на наличие ссылок в тексте
                                            logger.warning(f"Ответ не является JSON (inbound={inbound_id}), ищем ссылки в тексте")
                                            if f"{protocol}://" in content_text:
                                                lines = content_text.split("\n")
                                                for line in lines:
                                                    if f"{protocol}://" in line and email.lower() in line.lower():
                                                        # Извлекаем ссылку из строки
                                                        start_idx = line.find(f"{protocol}://")
                                                        end_idx = line.find(" ", start_idx)
                                                        if end_idx == -1:
                                                            end_idx = len(line)
                                                        extracted_link = line[start_idx:end_idx].strip()
                                                        logger.info(f"Найдена ссылка в тексте ответа: {extracted_link}")
                                                        return extracted_link
                                                    
                                except Exception as e:
                                    logger.warning(f"Ошибка при обработке ответа (inbound={inbound_id}): {e}, {str(e)}")
                    except Exception as e:
                        logger.warning(f"Ошибка при запросе JSON URL (inbound={inbound_id}): {e}")
                    
                    # Пробуем получить из Subscription URL
                    try:
                        async with session.get(sub_url) as response:
                            if response.status == 200:
                                content = await response.text()
                                logger.info(f"Получен Subscription для inbound {inbound_id}, размер: {len(content)} байт")
                                
                                # Пробуем декодировать base64
                                try:
                                    decoded = base64.b64decode(content).decode('utf-8')
                                    lines = decoded.split('\n')
                                    
                                    for line in lines:
                                        if email.lower() in line.lower() and line.startswith(f"{protocol}://"):
                                            logger.info(f"Найдена ссылка для {email} в Subscription URL")
                                            return line
                                except Exception as e:
                                    logger.warning(f"Ошибка при декодировании base64 (inbound={inbound_id}): {e}")
                    except Exception as e:
                        logger.warning(f"Ошибка при запросе Subscription URL (inbound={inbound_id}): {e}")
            
            logger.warning(f"Не удалось найти ссылку для {email} через Subscription URL")
            return None
            
        except Exception as e:
            logger.error(f"Ошибка при получении ссылки через Subscription URL: {e}")
            return None

    async def generate_vless_link(self, client_id: str, email: str) -> str:
        """
        Генерирует ссылку VLESS на основе данных клиента
        
        Args:
            client_id: ID клиента
            email: Email клиента
            
        Returns:
            str: Ссылка VLESS
        """
        # Параметры для VLESS
        server = "31.172.75.92"  # Стандартный сервер
        port = 443  # Стандартный порт
        encryption = "none"
        security = "reality"
        flow = "xtls-rprx-vision"
        network = "tcp"
        remarks = f"RazDvaVPN-{email}"
        
        # Формируем параметры
        params = [
            f"encryption={encryption}",
            f"security={security}",
            f"flow={flow}",
            f"type={network}"
        ]
        
        params_str = "&".join(params)
        
        # URL-кодирование описания
        remarks_encoded = urllib.parse.quote(remarks)
        
        # Формируем ссылку VLESS
        vless_link = f"vless://{client_id}@{server}:{port}?{params_str}#{remarks_encoded}"
        
        return vless_link

    async def get_status(self) -> Dict[str, Any]:
        """
        Получить статус подключения к панели
        
        Returns:
            Dict[str, Any]: Статус подключения
        """
        current_time = datetime.now().timestamp()
        
        return {
            "connected": self.connected,
            "connection_error": self.connection_error,
            "last_login": self.last_login,
            "last_connection_attempt": self.last_connection_attempt,
            "session_expires_in": int(self.last_login + self.login_expiry - current_time) if self.connected else 0
        }


# Создаем глобальный экземпляр API
panel_api = PanelApi()


async def login(force: bool = False) -> bool:
    """
    Аутентификация в панели - удобная обертка
    
    Args:
        force: Принудительная повторная аутентификация
        
    Returns:
        bool: Успешность аутентификации
    """
    return await panel_api.login(force)


async def get_client_link(email: str, protocol: str = "vless") -> Optional[str]:
    """
    Получить ссылку клиента - удобная обертка
    
    Args:
        email: Email клиента
        protocol: Протокол (vless или shadowsocks)
        
    Returns:
        Optional[str]: Ссылка для подключения или None
    """
    return await panel_api.get_client_link_via_subscription(email, protocol)


async def get_inbounds() -> List[Dict[str, Any]]:
    """
    Получает список inbounds из панели
    
    Returns:
        List[Dict[str, Any]]: Список inbounds или пустой список при ошибке
    """
    try:
        # Аутентификация
        if not await panel_api.login():
            logger.error("Не удалось аутентифицироваться в панели")
            return []
            
        # Формируем URL и заголовки
        inbounds_url = f"{panel_api.base_url}/panel/api/inbounds/list"
        
        # Получаем сессию
        session = await panel_api._get_session()
        
        # Отправляем запрос
        async with session.get(inbounds_url) as response:
            if response.status == 200:
                try:
                    json_data = await response.json()
                    if "obj" in json_data:
                        return json_data["obj"]
                    return []
                except Exception as e:
                    logger.error(f"Ошибка при обработке ответа: {e}")
                    return []
            else:
                logger.error(f"Ошибка при получении inbounds: {response.status}")
                return []
                
    except Exception as e:
        logger.error(f"Ошибка при получении inbounds: {e}")
        return []


async def create_client_without_link(email: str, traffic_gb: int = 5, days: int = 30) -> Optional[str]:
    """
    Создает нового клиента в панели управления без запроса ссылки
    
    Args:
        email: Email пользователя (используется как идентификатор)
        traffic_gb: Лимит трафика в ГБ
        days: Срок действия в днях
        
    Returns:
        Optional[str]: ID клиента или None при ошибке
    """
    try:
        # Аутентификация
        if not await panel_api.login():
            logger.error("Не удалось аутентифицироваться в панели")
            return None
            
        # Данные для создания клиента
        today = datetime.now()
        expires = today.strftime("%Y-%m-%d")  # По умолчанию сегодня
        
        # Генерируем случайный ID клиента
        import uuid
        client_id = str(uuid.uuid4()).replace("-", "")[:16]
        
        # Настраиваем данные клиента
        client_data = {
            "id": client_id,
            "flow": "xtls-rprx-vision",
            "email": email,
            "limitIp": 0,
            "totalGB": traffic_gb,
            "expiryTime": expires,
            "enable": True,
            "tgId": "",
            "subId": ""
        }
        
        # Формируем URL и заголовки
        add_client_url = f"{panel_api.base_url}/panel/api/inbounds/addClient"
        
        # Получаем сессию
        session = await panel_api._get_session()
        
        # Отправляем запрос
        try:
            async with session.post(add_client_url, json={
                "id": 1,  # ID inbound
                "settings": json.dumps({"clients": [client_data]})
            }) as response:
                if response.status == 200:
                    logger.info(f"Клиент успешно создан в панели: {email}, client_id: {client_id}")
                    return client_id
                else:
                    logger.error(f"Ошибка при создании клиента: {response.status}")
                    content = await response.text()
                    logger.error(f"Ответ сервера: {content[:500]}")
                    return None
        except Exception as e:
            logger.error(f"Ошибка при отправке запроса на создание клиента: {e}")
            return None
            
    except Exception as e:
        logger.error(f"Общая ошибка при создании клиента: {e}")
        return None

async def create_client(email: str, traffic_gb: int = 5, days: int = 30) -> Tuple[Optional[str], Optional[str]]:
    """
    Создает нового клиента в панели управления и получает ссылку
    
    Args:
        email: Email пользователя (используется как идентификатор)
        traffic_gb: Лимит трафика в ГБ
        days: Срок действия в днях
        
    Returns:
        Tuple[Optional[str], Optional[str]]: (ID клиента, ссылка) или (None, None) при ошибке
    """
    try:
        # Аутентификация
        if not await panel_api.login():
            logger.error("Не удалось аутентифицироваться в панели")
            return None, None
            
        # Данные для создания клиента
        today = datetime.now()
        expires = today.strftime("%Y-%m-%d")  # По умолчанию сегодня
        
        # Генерируем случайный ID клиента
        import uuid
        client_id = str(uuid.uuid4()).replace("-", "")[:16]
        
        # Настраиваем данные клиента
        client_data = {
            "id": client_id,
            "flow": "xtls-rprx-vision",
            "email": email,
            "limitIp": 0,
            "totalGB": traffic_gb,
            "expiryTime": expires,
            "enable": True,
            "tgId": "",
            "subId": ""
        }
        
        # Формируем URL и заголовки
        add_client_url = f"{panel_api.base_url}/panel/api/inbounds/addClient"
        
        # Получаем сессию
        session = await panel_api._get_session()
        
        # Отправляем запрос
        try:
            async with session.post(add_client_url, json={
                "id": 1,  # ID inbound
                "settings": json.dumps({"clients": [client_data]})
            }) as response:
                if response.status == 200:
                    logger.info(f"Клиент успешно создан в панели: {email}, client_id: {client_id}")
                    
                    # Даем панели немного времени на обновление данных перед запросом ссылки
                    await asyncio.sleep(1)
                    
                    # Инициализируем переменную link заранее
                    link = None
                    
                    # Попробуем получить ссылку несколько раз, с небольшими задержками
                    max_attempts = 3
                    for attempt in range(max_attempts):
                        link = await panel_api.get_client_link_via_subscription(email)
                        if link:
                            logger.info(f"Получена ссылка для клиента {email} с попытки {attempt+1}")
                            break
                        
                        logger.warning(f"Попытка {attempt+1}/{max_attempts} получить ссылку для {email} не удалась, ожидание...")
                        await asyncio.sleep(1)  # Небольшая задержка между попытками
                    
                    # Если через Subscription URL не удалось получить ссылку, генерируем вручную
                    if not link:
                        logger.warning(f"Не удалось получить ссылку через Subscription URL для {email}, генерируем вручную")
                        link = await panel_api.generate_vless_link(client_id, email)
                    
                    logger.info(f"Клиент полностью создан: {email}, client_id: {client_id}")
                    return client_id, link
                else:
                    logger.error(f"Ошибка при создании клиента: {response.status}")
                    content = await response.text()
                    logger.error(f"Ответ сервера: {content[:500]}")
                    return None, None
        except Exception as e:
            logger.error(f"Ошибка при отправке запроса на создание клиента: {e}")
            return None, None
            
    except Exception as e:
        logger.error(f"Общая ошибка при создании клиента: {e}")
        return None, None


async def test_panel_api() -> Dict[str, Any]:
    """
    Тестирование API панели
    
    Returns:
        Dict[str, Any]: Результаты тестирования
    """
    results = {
        "auth_success": False,
        "errors": []
    }
    
    try:
        # Проверка аутентификации
        auth_result = await panel_api.login(force=True)
        results["auth_success"] = auth_result
        
        if not auth_result:
            results["errors"].append(panel_api.connection_error or "Неизвестная ошибка аутентификации")
        
        # Проверка получения ссылок для тестового email
        test_email = "test@example.com"
        link = await panel_api.get_client_link_via_subscription(test_email)
        
        results["link_found"] = link is not None
        if not link:
            results["errors"].append(f"Не удалось получить ссылку для {test_email}")
        else:
            results["test_link"] = link
        
    except Exception as e:
        results["errors"].append(f"Общая ошибка тестирования: {e}")
    
    return results