"""
Модуль для легкой синхронизации данных с панелью без нагрузки на систему
"""
import logging
import asyncio
from datetime import datetime, timedelta
from typing import Optional, Dict, Any, List

from bot.database.active_subscriptions import get_all_active_subscriptions
from bot.database.operations import update_traffic_usage
from bot.services.working_panel_api import WorkingPanelAPI
from bot.services.panel_session_manager import PanelSessionManager

logger = logging.getLogger(__name__)

def normalize_email_for_panel(vpn_email: str, user_id: int) -> str:
    """
    Нормализует email для поиска в панели.
    Обеспечивает единый формат: user_id@username
    """
    try:
        # Если email уже в новом формате (содержит @ но не @razdvavpn.ru), оставляем как есть
        if vpn_email and '@' in vpn_email and '@razdvavpn.ru' not in vpn_email:
            return vpn_email
        
        # Если email в старом формате, конвертируем в новый
        if vpn_email and '@razdvavpn.ru' in vpn_email:
            # Извлекаем username из старого формата
            email_part = vpn_email.split('@')[0]
            
            if '_' in email_part:
                # Формат: username_userid@razdvavpn.ru
                parts = email_part.split('_')
                if len(parts) >= 2 and str(user_id) in email_part:
                    username = parts[0]
                    new_email = f"{user_id}@{username}"
                    logger.debug(f"Конвертация email: {vpn_email} -> {new_email}")
                    return new_email
            elif email_part.startswith('user_'):
                # Формат: user_userid@razdvavpn.ru
                new_email = f"{user_id}@user"
                logger.debug(f"Конвертация email: {vpn_email} -> {new_email}")
                return new_email
        
        # Возвращаем исходный email, если не смогли нормализовать
        return vpn_email
        
    except Exception as e:
        logger.warning(f"Ошибка при нормализации email {vpn_email}: {e}")
        return vpn_email

# Настройки синхронизации
MAX_CONCURRENT_REQUESTS = 10  # Максимум одновременных запросов к панели
REQUEST_DELAY = 0.5  # Задержка между запросами в секундах
BATCH_SIZE = 20  # Размер батча для обработки (для 1000 пользователей - 50 батчей за час)


class LightPanelSync:
    """Легкая синхронизация с панелью управления"""
    
    def __init__(self):
        self.api = WorkingPanelAPI()
        self._semaphore = asyncio.Semaphore(MAX_CONCURRENT_REQUESTS)
        self._last_sync_time = None
        
    async def sync_traffic_data(self) -> Dict[str, Any]:
        """
        Легкая синхронизация трафика для активных подписок
        
        Returns:
            Dict[str, Any]: Результаты синхронизации
        """
        start_time = datetime.now()
        logger.info("Запуск легкой синхронизации трафика...")
        
        try:
            # Получаем только активные подписки
            active_subscriptions = await get_all_active_subscriptions()
            if not active_subscriptions:
                logger.info("Нет активных подписок для синхронизации")
                return {"status": "success", "updated": 0, "errors": 0}
            
            logger.info(f"Найдено {len(active_subscriptions)} активных подписок")
            
            # Обрабатываем подписки батчами для снижения нагрузки
            updated_count = 0
            error_count = 0
            
            for i in range(0, len(active_subscriptions), BATCH_SIZE):
                batch = active_subscriptions[i:i + BATCH_SIZE]
                logger.debug(f"Обработка батча {i//BATCH_SIZE + 1}: {len(batch)} подписок")
                
                # Обрабатываем батч
                batch_results = await self._process_batch(batch)
                updated_count += batch_results["updated"]
                error_count += batch_results["errors"]
                
                # Пауза между батчами для снижения нагрузки
                if i + BATCH_SIZE < len(active_subscriptions):
                    await asyncio.sleep(REQUEST_DELAY)
            
            duration = (datetime.now() - start_time).total_seconds()
            logger.info(f"Синхронизация завершена за {duration:.2f}с. Обновлено: {updated_count}, ошибок: {error_count}")
            
            self._last_sync_time = datetime.now()
            
            return {
                "status": "success",
                "updated": updated_count,
                "errors": error_count,
                "duration": duration,
                "timestamp": self._last_sync_time.isoformat()
            }
            
        except Exception as e:
            logger.error(f"Ошибка при синхронизации трафика: {e}")
            return {"status": "error", "message": str(e)}
    
    async def _process_batch(self, subscriptions: List) -> Dict[str, int]:
        """
        Обрабатывает батч подписок
        
        Args:
            subscriptions: Список подписок для обработки
            
        Returns:
            Dict[str, int]: Статистика обработки
        """
        updated = 0
        errors = 0
        
        # Создаем задачи для параллельной обработки в рамках семафора
        tasks = []
        for subscription in subscriptions:
            task = self._sync_single_subscription(subscription)
            tasks.append(task)
        
        # Ждем завершения всех задач
        results = await asyncio.gather(*tasks, return_exceptions=True)
        
        for result in results:
            if isinstance(result, Exception):
                errors += 1
                logger.warning(f"Ошибка при синхронизации подписки: {result}")
            elif result:
                updated += 1
        
        return {"updated": updated, "errors": errors}
    
    async def _sync_single_subscription(self, subscription) -> bool:
        """
        Синхронизирует одну подписку
        
        Args:
            subscription: Объект подписки
            
        Returns:
            bool: True если обновление успешно
        """
        async with self._semaphore:
            try:
                if not subscription.vpn_email:
                    logger.debug(f"Подписка {subscription.id} не имеет vpn_email, пропускаем")
                    return False
                
                # Пытаемся получить трафик по исходному email
                panel_api = PanelSessionManager()
                traffic_stats = await panel_api.get_client_traffic(subscription.vpn_email)
                
                # Если не найдено, пытаемся найти клиента по UUID вместо email
                if not traffic_stats and subscription.vpn_user_id:
                    logger.debug(f"Пытаемся получить трафик по UUID: {subscription.vpn_user_id}")
                    traffic_stats = await panel_api.get_client_traffic(subscription.vpn_user_id)
                
                if traffic_stats and "traffic_used" in traffic_stats:
                    current_usage_bytes = traffic_stats["traffic_used"]
                    current_usage_gb = current_usage_bytes / (1024**3)
                    
                    # Обновляем только если есть изменения (сравниваем в байтах)
                    if current_usage_bytes != subscription.traffic_used:
                        await update_traffic_usage(subscription.id, current_usage_bytes)
                        logger.info(f"Обновлен трафик для подписки {subscription.id}: {current_usage_gb:.3f} ГБ")
                        return True
                
                return False
                
            except Exception as e:
                logger.warning(f"Ошибка синхронизации подписки {subscription.id}: {e}")
                return False
            finally:
                # Небольшая задержка между запросами
                await asyncio.sleep(0.1)
    
    async def quick_health_check(self) -> Dict[str, Any]:
        """
        Быстрая проверка состояния панели
        
        Returns:
            Dict[str, Any]: Статус панели
        """
        try:
            # Простая проверка подключения без нагрузки
            is_online = await self.api.login()
            
            return {
                "panel_online": is_online,
                "last_sync": self._last_sync_time.isoformat() if self._last_sync_time else None,
                "timestamp": datetime.now().isoformat()
            }
            
        except Exception as e:
            logger.error(f"Ошибка проверки панели: {e}")
            return {
                "panel_online": False,
                "error": str(e),
                "timestamp": datetime.now().isoformat()
            }
    
    def get_sync_status(self) -> Dict[str, Any]:
        """
        Получает статус синхронизации
        
        Returns:
            Dict[str, Any]: Информация о состоянии синхронизации
        """
        return {
            "last_sync": self._last_sync_time.isoformat() if self._last_sync_time else None,
            "sync_interval": f"{BATCH_SIZE} подписок за раз",
            "max_concurrent": MAX_CONCURRENT_REQUESTS,
            "request_delay": f"{REQUEST_DELAY}с между батчами"
        }


# Глобальный экземпляр синхронизатора
light_sync = LightPanelSync()


async def sync_traffic_lightweight() -> Dict[str, Any]:
    """
    Функция-обертка для легкой синхронизации трафика
    
    Returns:
        Dict[str, Any]: Результаты синхронизации
    """
    return await light_sync.sync_traffic_data()


async def check_panel_health() -> Dict[str, Any]:
    """
    Функция-обертка для проверки состояния панели
    
    Returns:
        Dict[str, Any]: Статус панели
    """
    return await light_sync.quick_health_check()