"""
Обработчики административных команд для бота
"""

from datetime import datetime, timedelta
import base64
import re
import uuid
import asyncio
import ssl
import json
import aiohttp
from typing import Optional

from aiogram import Router, F
from aiogram.filters import Command, StateFilter
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from aiogram.types import Message, CallbackQuery
from aiogram.utils.keyboard import InlineKeyboardBuilder
from aiogram.types.inline_keyboard_markup import InlineKeyboardMarkup
from aiogram.types.inline_keyboard_button import InlineKeyboardButton

from config import ADMIN_IDS as ADMINS, ClientStorage
import logging

logger = logging.getLogger(__name__)
from bot.database.operations import (
    get_user,
    get_user_subscription,
    delete_subscription,
    delete_subscription_completely,
    delete_user_completely,
    get_all_subscriptions,
    get_all_users,
)
from bot.database.models import User, Subscription, UserStatus
from bot.services import strict_panel_api
from bot.services.panel_service import create_vpn_account


# Определение собственных классов ошибок
class StrictPanelApiError(Exception):
    """Ошибка при работе с API панели"""

    pass


# Функция для проверки соединения с панелью
async def check_panel_connection():
    """Проверка соединения с панелью управления"""
    try:
        # Используем функцию получения cookie для проверки соединения
        cookies = await strict_panel_api.get_auth_cookies()
        return cookies is not None
    except Exception as e:
        logger.error(f"Ошибка при проверке соединения с панелью: {e}")
        return False


from bot.utils.formatting import get_user_name


# Определяем класс ошибки для информации о клиенте
class ClientInfoError(Exception):
    """Ошибка при получении информации о клиенте"""

    pass


from bot.services.working_panel_api import WorkingPanelAPI
from config_keys import get_panel_credentials


async def generate_vpn_key_for_admin(client_id, email):
    """Генерация VPN ключа через API панели для админских команд"""
    try:
        panel_url, username, password = get_panel_credentials()
        panel_api = WorkingPanelAPI(panel_url, username, password)

        if await panel_api.login():
            vpn_key = await panel_api.get_client_vpn_key(client_id, email)
            await panel_api.close()
            return {"direct_link": vpn_key, "link": vpn_key} if vpn_key else None
        else:
            await panel_api.close()
            return None
    except Exception as e:
        logger.error(f"Ошибка при генерации VPN ключа: {e}")
        return None


router = Router()


# Статусы для административных команд
class AdminStates(StatesGroup):
    """Состояния для админ команд"""

    waiting_for_user_id = State()
    waiting_for_confirmation = State()
    waiting_for_delete_user_id = State()

    # Добавление нового клиента
    add_client_protocol = State()
    add_client_email = State()
    add_client_traffic = State()
    add_client_days = State()
    add_client_confirmation = State()

    # Повторное получение ссылки
    waiting_for_refresh_link = State()

    # Удаление тестовых подписок
    delete_test_sub_confirmation = State()


def admin_only(func):
    """Декоратор для проверки, что пользователь является администратором"""

    async def wrapper(message: Message, *args, **kwargs):
        # Проверяем, является ли пользователь администратором
        if message.chat.id not in ADMINS:
            await message.answer("❌ У вас нет прав для выполнения этой команды.")
            return
        return await func(message, *args, **kwargs)

    return wrapper


async def validate_user_exists(user_id: int, message: Message) -> Optional[User]:
    """Проверяет существование пользователя и возвращает его данные"""
    user = await get_user(user_id)
    if not user:
        await message.answer(f"❌ Пользователь с ID {user_id} не найден.")
        return None
    return user


def create_confirmation_keyboard(confirm_text: str, confirm_data: str, cancel_data: str = "cancel") -> InlineKeyboardMarkup:
    """Создает клавиатуру подтверждения с кнопками"""
    kb = InlineKeyboardBuilder()
    kb.row(
        InlineKeyboardButton(text=f"✅ {confirm_text}", callback_data=confirm_data),
        InlineKeyboardButton(text="❌ Отмена", callback_data=cancel_data),
    )
    return kb.as_markup()


@router.message(Command("admin"))
@admin_only
async def cmd_admin(message: Message, **kwargs):
    """
    Показать список доступных административных команд
    """
    admin_commands = (
        "🔑 <b>Административные команды:</b>\n\n"
        "/admin - показать это сообщение\n"
        "/stats - показать статистику бота\n"
        "/list_users - список всех пользователей\n"
        "/reset_subscription - сбросить подписку пользователя\n"
        "/delete_subscription_db - полностью удалить пользователя из БД\n"
        "/delete_test_subs - удалить тестовые подписки\n"
        "/add_client - добавить нового VPN-клиента"
    )
    await message.answer(admin_commands, parse_mode="HTML")


@router.message(Command("stats"))
@admin_only
async def cmd_stats(message: Message, **kwargs):
    """
    Показать статистику бота (пользователи, подписки)
    """
    try:
        # Получаем статистику
        users = await get_all_users()
        subscriptions = await get_all_subscriptions()

        # Считаем активные подписки
        active_subs = len([s for s in subscriptions if s.is_active])

        # Общая статистика
        stats_message = (
            "📊 <b>Статистика бота:</b>\n\n"
            f"👤 Всего пользователей: <b>{len(users)}</b>\n"
            f"🔑 Всего подписок: <b>{len(subscriptions)}</b>\n"
            f"✅ Активных подписок: <b>{active_subs}</b>\n"
            f"📅 Неактивных подписок: <b>{len(subscriptions) - active_subs}</b>\n\n"
        )

        # Статистика за последние 24 часа и 7 дней
        now = datetime.now()
        day_ago = now - timedelta(days=1)
        week_ago = now - timedelta(days=7)

        new_day_users = sum(1 for u in users if u.created_at and u.created_at > day_ago)
        new_week_users = sum(1 for u in users if u.created_at and u.created_at > week_ago)

        new_day_subs = sum(1 for s in subscriptions if s.created_at and s.created_at > day_ago)
        new_week_subs = sum(1 for s in subscriptions if s.created_at and s.created_at > week_ago)

        stats_message += (
            "🕒 <b>За последние 24 часа:</b>\n"
            f"👤 Новых пользователей: <b>{new_day_users}</b>\n"
            f"🔑 Новых подписок: <b>{new_day_subs}</b>\n\n"
            "📅 <b>За последние 7 дней:</b>\n"
            f"👤 Новых пользователей: <b>{new_week_users}</b>\n"
            f"🔑 Новых подписок: <b>{new_week_subs}</b>"
        )

        await message.answer(stats_message, parse_mode="HTML")
    except Exception as e:
        logger.error(f"Ошибка при получении статистики: {e}")
        await message.answer(f"❌ Произошла ошибка при получении статистики: {str(e)[:100]}...")


@router.message(Command("list_users"))
@admin_only
async def cmd_list_users(message: Message, **kwargs):
    """
    Показать список всех пользователей с ID
    """
    try:
        users = await get_all_users()

        if not users:
            await message.answer("ℹ️ Список пользователей пуст.")
            return

        # Формируем сообщение со списком пользователей
        users_message = "👥 <b>Список пользователей:</b>\n\n"

        for i, user in enumerate(users, 1):
            sub_data = ""
            subscription = await get_user_subscription(user.id)
            if subscription:
                status = "✅ Активна" if subscription.is_active else "❌ Неактивна"
                days = subscription.days_remaining if subscription.is_active else 0
                sub_data = f" | {status} ({days} дн.)"

            username = f"@{user.username}" if user.username else "Нет username"
            name = get_user_name(user)
            users_message += f"{i}. {name} ({username}) | ID: <code>{user.id}</code>{sub_data}\n"

            # Разбиваем на части, если сообщение слишком длинное
            if i % 30 == 0 and i < len(users):
                await message.answer(users_message, parse_mode="HTML")
                users_message = ""

        if users_message:
            await message.answer(users_message, parse_mode="HTML")
    except Exception as e:
        logger.error(f"Ошибка при получении списка пользователей: {e}")
        await message.answer(
            f"❌ Произошла ошибка при получении списка пользователей: {str(e)[:100]}..."
        )


@router.message(Command("reset_subscription"))
@admin_only
async def cmd_reset_subscription(message: Message, state: FSMContext, **kwargs):
    """
    Удалить подписку пользователя
    Пример: /reset_subscription 12345678
    """
    try:
        # Проверяем, передан ли ID пользователя сразу в команде
        command_parts = message.text.split()
        if len(command_parts) > 1:
            # ID передан сразу в команде
            user_id = int(command_parts[1])
            await process_reset_subscription(message, user_id)
        else:
            # Запрашиваем ID пользователя
            await message.answer("Введите ID пользователя, чью подписку нужно удалить:")
            await state.set_state(AdminStates.waiting_for_user_id)
    except ValueError:
        await message.answer("❌ Некорректный ID пользователя. Должно быть целое число.")
    except Exception as e:
        logger.error(f"Ошибка при обработке команды reset_subscription: {e}")
        await message.answer(f"❌ Произошла ошибка: {str(e)[:100]}...")


@router.message(AdminStates.waiting_for_user_id)
async def process_user_id_input(message: Message, state: FSMContext, **kwargs):
    """Обработка введенного ID пользователя"""
    try:
        user_id = int(message.text.strip())
        await process_reset_subscription(message, user_id)
        await state.clear()
    except ValueError:
        await message.answer("❌ Некорректный ID пользователя. Должно быть целое число.")
    except Exception as e:
        logger.error(f"Ошибка при обработке ID пользователя: {e}")
        await message.answer(f"❌ Произошла ошибка: {str(e)[:100]}...")
        await state.clear()


async def process_reset_subscription(message: Message, user_id: int):
    """Процесс удаления подписки"""
    try:
        # Проверяем существование пользователя
        user = await validate_user_exists(user_id, message)
        if not user:
            return

        # Проверяем наличие подписки
        subscription = await get_user_subscription(user_id)
        if not subscription:
            await message.answer(
                f"ℹ️ У пользователя {get_user_name(user)} (ID: {user_id}) нет активной подписки."
            )
            return

        # Удаляем подписку
        result = await delete_subscription(subscription.id)

        if result:
            await message.answer(
                f"✅ Подписка пользователя {get_user_name(user)} (ID: {user_id}) успешно удалена."
            )
        else:
            await message.answer(
                f"❌ Не удалось удалить подписку пользователя {get_user_name(user)} (ID: {user_id})."
            )
    except Exception as e:
        logger.error(f"Ошибка при удалении подписки: {e}")
        await message.answer(f"❌ Произошла ошибка при удалении подписки: {str(e)[:100]}...")


@router.message(Command("delete_subscription_db"))
@admin_only
async def cmd_delete_subscription_db(message: Message, state: FSMContext, **kwargs):
    """
    ПОЛНОСТЬЮ удалить подписку из базы данных (для тестирования)
    Пример: /delete_subscription_db 12345678
    """
    try:
        # Проверяем, передан ли ID пользователя сразу в команде
        command_parts = message.text.split()
        if len(command_parts) > 1:
            # ID передан сразу в команде
            user_id = int(command_parts[1])
            await process_delete_subscription_db(message, user_id)
        else:
            # Запрашиваем ID пользователя
            await message.answer(
                "⚠️ <b>ВНИМАНИЕ:</b> Эта команда полностью удаляет подписку из БД!\n\nВведите ID пользователя:",
                parse_mode="HTML",
            )
            await state.set_state(AdminStates.waiting_for_delete_user_id)
    except ValueError:
        await message.answer("❌ Некорректный ID пользователя. Должно быть целое число.")
    except Exception as e:
        logger.error(f"Ошибка при обработке команды delete_subscription_db: {e}")
        await message.answer(f"❌ Произошла ошибка: {str(e)[:100]}...")


@router.message(AdminStates.waiting_for_delete_user_id)
async def process_delete_user_id_input(message: Message, state: FSMContext, **kwargs):
    """Обработка введенного ID пользователя для полного удаления"""
    try:
        user_id = int(message.text.strip())
        await process_delete_subscription_db(message, user_id)
        await state.clear()
    except ValueError:
        await message.answer("❌ Некорректный ID пользователя. Должно быть целое число.")
    except Exception as e:
        logger.error(f"Ошибка при обработке ID пользователя для удаления: {e}")
        await message.answer(f"❌ Произошла ошибка: {str(e)[:100]}...")
        await state.clear()


async def process_delete_subscription_db(message: Message, user_id: int):
    """Процесс ПОЛНОГО удаления пользователя из базы данных"""
    try:
        # Проверяем существование пользователя
        user = await validate_user_exists(user_id, message)
        if not user:
            return

        # ПОЛНОСТЬЮ удаляем пользователя и все связанные данные
        result = await delete_user_completely(user_id)

        if result:
            await message.answer(
                f"🗑️ <b>Пользователь ПОЛНОСТЬЮ удален из базы данных!</b>\n"
                f"Пользователь: {get_user_name(user)}\n"
                f"ID: {user_id}\n\n"
                f"✅ Удалено:\n"
                f"   • Профиль пользователя\n"
                f"   • Все подписки\n"
                f"   • Все платежи\n"
                f"   • Все связанные записи\n\n"
                f"📊 Пользователь полностью исчез из веб-интерфейса",
                parse_mode="HTML",
            )
        else:
            await message.answer(
                f"❌ Не удалось полностью удалить пользователя {get_user_name(user)} (ID: {user_id}) из базы данных."
            )
    except Exception as e:
        logger.error(f"Ошибка при полном удалении пользователя из БД: {e}")
        await message.answer(
            f"❌ Произошла ошибка при полном удалении пользователя: {str(e)[:100]}..."
        )


@router.message(Command("add_client"))
@admin_only
async def cmd_add_client(message: Message, state: FSMContext, **kwargs):
    """
    Начать процесс добавления нового VPN-клиента в панель
    """
    kb = await create_protocol_keyboard()
    await message.answer("Выберите протокол для нового VPN-клиента:", reply_markup=kb)
    await state.set_state(AdminStates.add_client_protocol)


async def create_protocol_keyboard():
    """Создать клавиатуру с протоколами"""
    kb = InlineKeyboardBuilder()
    kb.row(
        InlineKeyboardButton(text="VLESS (ID: 1)", callback_data="protocol_vless_1"),
        InlineKeyboardButton(text="Shadowsocks (ID: 2)", callback_data="protocol_ss_2"),
    )
    return kb.as_markup()


@router.callback_query(
    StateFilter(AdminStates.add_client_protocol), lambda c: c.data.startswith("protocol_")
)
async def process_protocol_selection(callback: CallbackQuery, state: FSMContext):
    """Обработка выбора протокола"""
    await callback.answer()

    # Извлекаем протокол и inbound_id из callback data
    data = callback.data.split("_")
    protocol = data[1]  # vless или ss
    inbound_id = int(data[2])

    logger.info(f"Выбран протокол: {protocol}, inbound_id: {inbound_id}")

    # Сохраняем выбор в FSM
    await state.update_data(protocol=protocol, inbound_id=inbound_id)

    # Запрашиваем email
    await callback.message.edit_text(
        "Введите email для VPN-клиента (или оставьте пустым для автогенерации):"
    )
    await state.set_state(AdminStates.add_client_email)


@router.message(AdminStates.add_client_email)
async def process_email_input(message: Message, state: FSMContext):
    """Обработка ввода email"""
    email_input = message.text.strip()

    # Если email не указан, генерируем случайный
    if not email_input:
        random_id = uuid.uuid4().hex[:8]
        email = f"user_{random_id}@razuber.xyz"
    else:
        # Проверяем формат email
        email_pattern = re.compile(r"^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$")
        if not email_pattern.match(email_input):
            await message.answer(
                "❌ Некорректный формат email. Пожалуйста, введите корректный email."
            )
            return
        email = email_input

    await state.update_data(email=email)
    await message.answer(
        f"Введите лимит трафика в ГБ для VPN-клиента (или введите 0 для безлимитного):"
    )
    await state.set_state(AdminStates.add_client_traffic)


@router.message(AdminStates.add_client_traffic)
async def process_traffic_input(message: Message, state: FSMContext):
    """Обработка ввода лимита трафика"""
    try:
        traffic_gb = int(message.text.strip())
        if traffic_gb < 0:
            raise ValueError("Traffic limit cannot be negative")

        await state.update_data(traffic_gb=traffic_gb)
        await message.answer("Введите срок действия в днях:")
        await state.set_state(AdminStates.add_client_days)
    except ValueError:
        await message.answer("❌ Пожалуйста, введите корректное число для лимита трафика.")


@router.message(AdminStates.add_client_days)
async def process_days_input(message: Message, state: FSMContext):
    """Обработка ввода срока действия"""
    try:
        days = int(message.text.strip())
        if days <= 0:
            raise ValueError("Days should be positive")

        # Получаем все данные из состояния
        data = await state.get_data()
        protocol = data.get("protocol")
        email = data.get("email")
        traffic_gb = data.get("traffic_gb")
        inbound_id = data.get("inbound_id")

        protocol_name = "VLESS" if protocol == "vless" else "Shadowsocks"

        # Создаем клавиатуру для подтверждения
        kb = create_confirmation_keyboard("Подтвердить", "confirm_add_client", "cancel_add_client")

        # Отображаем информацию для подтверждения
        await message.answer(
            f"📝 <b>Проверьте данные нового VPN-клиента:</b>\n\n"
            f"🔹 Протокол: <b>{protocol_name}</b>\n"
            f"🔹 Email: <b>{email}</b>\n"
            f"🔹 Лимит трафика: <b>{traffic_gb} ГБ</b>\n"
            f"🔹 Срок действия: <b>{days} дней</b>\n"
            f"🔹 ID inbound: <b>{inbound_id}</b>\n\n"
            f"Подтвердите создание клиента:",
            reply_markup=kb.as_markup(),
            parse_mode="HTML",
        )

        await state.update_data(days=days)
        await state.set_state(AdminStates.add_client_confirmation)
    except ValueError:
        await message.answer("❌ Пожалуйста, введите корректное число дней.")


@router.callback_query(
    StateFilter(AdminStates.add_client_confirmation), lambda c: c.data == "confirm_add_client"
)
async def confirm_add_client(callback: CallbackQuery, state: FSMContext):
    """Подтверждение создания клиента"""
    await callback.answer()

    try:
        data = await state.get_data()
        protocol = data.get("protocol")
        email = data.get("email")
        traffic_gb = data.get("traffic_gb")
        days = data.get("days")

        # Преобразуем протокол в формат для API
        vpn_protocol = "vless" if protocol == "vless" else "shadowsocks"

        # Ждем пока загрузится сообщение о создании
        await callback.message.edit_text("⏳ Создаем VPN-клиента, пожалуйста, подождите...")

        try:
            # Больше не проверяем соединение отдельно, т.к. это делается внутри create_vpn_account

            # Создаем VPN-аккаунт через API
            account_data = await create_vpn_account(
                user_id=callback.from_user.id,
                email=email,
                protocol=vpn_protocol,
                traffic_limit_gb=traffic_gb,
                days=days,
            )

            # Получаем полную информацию о клиенте из данных создания
            client_id = account_data.get("client_id")
            logger.info(f"Успешно создан VPN-клиент с ID: {client_id}")

            # Попытка получить данные через Subscription URL
            try:
                logger.info(
                    f"Попытка получения информации через Subscription URL для клиента {client_id}"
                )
                # Определяем ID inbound в зависимости от протокола
                inbound_id = 1  # По умолчанию для VLESS
                if vpn_protocol.lower() == "shadowsocks":
                    inbound_id = 2  # Для Shadowsocks

                # Ждем минимальное время на обновление данных
                logger.info("Минимальное ожидание обновления данных...")
                await asyncio.sleep(1)

                # Пробуем получить данные из всех возможных inbound
                sub_data = None
                for test_inbound_id in [1, 2, 3]:
                    logger.info(f"Попытка получения ссылки через inbound #{test_inbound_id}...")

                    # Получаем данные через API панели
                    try:
                        temp_data = await generate_vpn_key_for_admin(client_id, email)

                        if temp_data and ("direct_link" in temp_data or "link" in temp_data):
                            logger.info(f"Успешно получены данные из inbound #{test_inbound_id}")
                            sub_data = temp_data
                            break
                    except Exception as e:
                        logger.warning(
                            f"Ошибка при получении данных из inbound #{test_inbound_id}: {e}"
                        )
                        continue

                if sub_data:
                    logger.info(f"Получены данные через Subscription URL: {sub_data}")
                    # Обновляем данные аккаунта полученными из Subscription URL
                    account_data.update(sub_data)
                    logger.info(f"Данные аккаунта обновлены из Subscription URL")
                else:
                    logger.warning(
                        f"Не удалось получить данные через Subscription URL, используем локальные данные"
                    )
            except Exception as e:
                logger.error(f"Ошибка при получении данных через Subscription URL: {e}")

            # Логирование финальных данных
            logger.info(f"Финальные данные клиента:")
            logger.info(f"ID: {account_data.get('client_id')}")
            logger.info(f"Email: {account_data.get('email')}")
            logger.info(f"Ссылка: {account_data.get('direct_link', 'не задана')}")

        except StrictPanelApiError as e:
            # Обрабатываем ошибку подключения к панели
            logger.error(f"Ошибка при создании VPN-клиента: {e}")
            await callback.message.edit_text(
                f"❌ Не удалось создать VPN-клиента.\n"
                f"Ошибка: {e}\n"
                f"Пожалуйста, попробуйте позже или обратитесь к администратору."
            )
            return

        # Добавляем дополнительные параметры для Shadowsocks
        if vpn_protocol == "shadowsocks":
            # Генерируем пароль, если его нет
            password = account_data.get("password", "")
            if not password:
                password = str(uuid.uuid4()).replace("-", "")[:16]
                account_data["password"] = password

            # Стандартные параметры для shadowsocks
            server = "31.172.75.92"
            port = 14765
            method = "chacha20-ietf-poly1305"

            # Формируем ссылку для подключения
            ss_url_content = f"{method}:{password}@{server}:{port}"
            encoded_content = base64.b64encode(ss_url_content.encode()).decode()
            ss_url = f"ss://{encoded_content}?type=tcp#{email}"
            account_data["direct_link"] = ss_url

        # Добавляем дополнительную информацию для VLESS (ссылки и конфигурация)
        if vpn_protocol == "vless":
            # Настройки для VLESS (Reality)
            server = "31.172.75.92"
            port = 443
            client_uuid = account_data.get("client_id")
            public_key = "J7O8WbpqHE4mrSwOZZSb0zWc5Wx5ptCJzyfnVAi0pRw"
            server_name = "yahoo.com"
            short_id = "f7e8"
            spider_x = "/xWJGYAO8zsetAXk"

            # Формируем прямую ссылку для VLESS с обязательным flow=xtls-rprx-vision
            vless_link = f"vless://{client_uuid}@{server}:{port}?encryption=none&flow=xtls-rprx-vision&security=reality&sni={server_name}&fp=chrome&pbk={public_key}&sid={short_id}&spx={spider_x}&type=tcp&headerType=none#{email}"
            account_data["direct_link"] = vless_link

        # Получаем ID клиента
        client_id = account_data.get("id") or account_data.get("client_id")

        # Сохраняем данные аккаунта в хранилище клиентов
        from config import ClientStorage

        client_storage = ClientStorage()
        client_storage.add_client(
            {
                "client_id": client_id,
                "email": email,
                "protocol": vpn_protocol,
                "created_at": datetime.now().isoformat(),
                "account_data": account_data,
                "traffic_gb": traffic_gb,
                "days": days,
            }
        )

        # Создаем простое первое сообщение с ID клиента и базовой информацией
        simple_message = f"✅ <b>VPN-клиент успешно создан!</b>\n\n"
        simple_message += (
            f"<b>ID клиента:</b> <code>{client_id}</code>\n\n"
            f"🔹 Email: <code>{email}</code>\n"
            f"🔹 Протокол: <code>{vpn_protocol.upper()}</code>\n"
            f"⚙️ Трафик: <b>{traffic_gb} ГБ</b>\n"
            f"📅 Срок действия: <b>{days} дней</b>\n\n"
        )

        # Добавляем статус API в сообщение
        if account_data.get("created", False):
            api_status = "✅ <b>Клиент успешно добавлен в панель 3X-UI</b>"
        else:
            api_status = "⚠️ <b>Клиент создан в автономном режиме</b> (изменения не синхронизированы с панелью)"

        # Создаем клавиатуру с кнопкой для получения ключа
        get_key_kb = InlineKeyboardBuilder()
        get_key_kb.button(text="🔑 Получить ключ VPN", callback_data=f"admin_get_key_{client_id}")
        get_key_kb.adjust(1)

        # Сохраняем данные в state для последующего использования
        await state.update_data(client_id=client_id, account_data=account_data)

        # Данные клиента сохраняем в ClientStorage для последующего получения ключа
        from config import CREATED_CLIENTS

        # Создаем текущее время
        current_time = datetime.now().isoformat()

        # Формируем данные клиента для сохранения
        client_data = {
            "client_id": client_id,
            "email": email,
            "protocol": vpn_protocol,
            "created_at": current_time,
            "account_data": account_data,
            "traffic_gb": traffic_gb,
            "days": days,
        }

        # Сохраняем в хранилище
        CREATED_CLIENTS.add_client(client_data)

        # Теперь отправляем первое сообщение с ID клиента и кнопкой для получения ключа
        await callback.message.edit_text(
            simple_message + api_status, parse_mode="HTML", reply_markup=get_key_kb.as_markup()
        )

        # Очищаем состояние, т.к. шаг создания клиента завершен
        await state.clear()

        # Логирование успешного создания клиента
        logger.info(f"VPN клиент успешно создан: ID={client_id}, Email={email}")
    except Exception as e:
        logger.error(f"Ошибка при создании VPN-клиента: {e}")
        await callback.message.edit_text(
            f"❌ <b>Ошибка при создании VPN-клиента!</b>\n\n" f"Детали ошибки: {str(e)[:100]}..."
        )

    await state.clear()


@router.callback_query(
    StateFilter(AdminStates.add_client_confirmation), lambda c: c.data == "cancel_add_client"
)
async def cancel_add_client(callback: CallbackQuery, state: FSMContext):
    """Отмена создания клиента"""
    await callback.answer()
    await callback.message.edit_text("❌ Операция отменена.")
    await state.clear()


@router.callback_query(lambda c: c.data.startswith("refresh_link"))
async def refresh_client_link(callback: CallbackQuery, state: FSMContext):
    """Повторное получение ссылки для созданного клиента"""
    await callback.answer("⏳ Выполняется запрос к панели...")

    # Уведомляем пользователя о начале процесса
    await callback.message.edit_text(
        "⏳ <b>Повторно запрашиваем ссылку из панели...</b>\n\n"
        "Это может занять несколько секунд. Пожалуйста, подождите.",
        parse_mode="HTML",
    )

    try:
        # Получаем сохраненные данные клиента
        data = await state.get_data()
        client_id = data.get("client_id")
        email = data.get("email")
        protocol = data.get("protocol")
        traffic_gb = data.get("traffic_gb")
        days = data.get("days")

        logger.info(f"Повторный запрос ссылки для клиента: {email}, ID: {client_id}")

        # Определяем ID inbound в зависимости от протокола
        inbound_id = 1  # По умолчанию для VLESS
        if protocol.lower() == "shadowsocks":
            inbound_id = 2  # Для Shadowsocks

        # Делаем повторный запрос к Subscription URL с увеличенной задержкой
        direct_link = None

        # Создаем подключение для запроса
        ssl_context = ssl.create_default_context()
        ssl_context.check_hostname = False
        ssl_context.verify_mode = ssl.CERT_NONE
        connector = aiohttp.TCPConnector(ssl=ssl_context)
        timeout = aiohttp.ClientTimeout(total=15)

        # URL для получения данных о клиенте
        sub_url = f"http://31.172.75.92:2096/subavto/{inbound_id}"
        json_url = f"http://31.172.75.92:2096/jsonauto/{inbound_id}"

        logger.info(f"Запрос данных из: {json_url} и {sub_url}")

        # Пробуем получить ссылку
        async with aiohttp.ClientSession(connector=connector, timeout=timeout) as session:
            # Сначала проверяем JSON URL
            try:
                async with session.get(json_url) as response:
                    if response.status == 200:
                        json_data = await response.text()
                        logger.info(f"Получен ответ от JSON URL, размер: {len(json_data)} байт")

                        try:
                            # Преобразуем в JSON
                            config_data = json.loads(json_data)

                            # Ищем ссылку для нашего клиента
                            for line in config_data:
                                if isinstance(line, str):
                                    if (
                                        line.startswith("vless://") or line.startswith("ss://")
                                    ) and email.lower() in line.lower():
                                        logger.info(
                                            f"Найдена ссылка для клиента {email}: {line[:50]}..."
                                        )
                                        direct_link = line
                                        break
                        except Exception as e:
                            logger.warning(f"Ошибка при парсинге JSON: {e}")
            except Exception as e:
                logger.warning(f"Ошибка при запросе JSON URL: {e}")

            # Если не нашли через JSON, пробуем подписку
            if not direct_link:
                try:
                    async with session.get(sub_url) as response:
                        if response.status == 200:
                            sub_data = await response.text()
                            lines = sub_data.strip().split("\n")

                            for line in lines:
                                if (
                                    line.startswith("vless://") or line.startswith("ss://")
                                ) and email.lower() in line.lower():
                                    logger.info(
                                        f"Найдена ссылка через Subscription URL: {line[:50]}..."
                                    )
                                    direct_link = line
                                    break
                except Exception as e:
                    logger.warning(f"Ошибка при запросе Subscription URL: {e}")

        # Формируем ответное сообщение
        if direct_link:
            success_message = f"✅ <b>Ссылка успешно получена!</b>\n\n"

            # Добавляем базовую информацию
            success_message += (
                f"🔹 Email: <code>{email}</code>\n"
                f"🔹 Протокол: <code>{protocol.upper()}</code>\n"
                f"⚙️ Трафик: <b>{traffic_gb} GB</b>\n"
                f"📅 Срок действия: <b>{days} дней</b>\n\n"
                f"📋 <b>Прямая ссылка для импорта:</b>\n"
                f"<code>{direct_link}</code>"
            )

            # Отправляем сообщение с успешным результатом
            await callback.message.edit_text(success_message, parse_mode="HTML")
            # Сбрасываем состояние
            await state.clear()
        else:
            # Если не удалось получить ссылку, предлагаем попробовать снова
            fail_message = (
                "⚠️ <b>Не удалось получить ссылку из панели</b>\n\n"
                f"Клиент с email <code>{email}</code> создан в панели, "
                "но получить ссылку пока не удалось.\n\n"
                "Возможно, нужно подождать, пока данные обновятся в панели."
            )

            # Создаем клавиатуру с кнопкой повторного запроса
            keyboard = InlineKeyboardMarkup(
                inline_keyboard=[
                    [
                        InlineKeyboardButton(
                            text="🔄 Попробовать еще раз", callback_data="refresh_link"
                        )
                    ]
                ]
            )

            await callback.message.edit_text(fail_message, reply_markup=keyboard, parse_mode="HTML")
    except Exception as e:
        logger.error(f"Ошибка при повторном получении ссылки: {e}")
        error_message = (
            "❌ <b>Произошла ошибка при получении ссылки</b>\n\n" f"Детали ошибки: {str(e)[:100]}"
        )

        # Создаем клавиатуру с кнопкой повторного запроса
        keyboard = InlineKeyboardMarkup(
            inline_keyboard=[
                [InlineKeyboardButton(text="🔄 Попробовать еще раз", callback_data="refresh_link")]
            ]
        )

        await callback.message.edit_text(error_message, reply_markup=keyboard, parse_mode="HTML")


@router.callback_query(F.data.startswith("admin_get_key_"))
async def admin_get_vpn_key(callback: CallbackQuery, state: FSMContext):
    """Обработчик для получения ключа VPN администратором"""
    # Извлекаем ID клиента из callback_data
    client_id = callback.data.split("_")[-1]
    logger.info(f"Запрос на получение ключа для клиента с ID: {client_id}")

    # Получаем данные из хранилища клиентов
    from config import CREATED_CLIENTS

    clients = CREATED_CLIENTS.get_clients()

    # Ищем клиента по ID
    client_data = None
    for client in clients:
        if client.get("client_id") == client_id:
            client_data = client
            break

    # Если клиент не найден в локальном хранилище, пробуем получить из базы
    if not client_data:
        logger.warning(f"Клиент с ID {client_id} не найден в локальном хранилище")

        # Ищем подписку по ID клиента через базы данных
        from bot.database.operations import get_all_subscriptions

        subscriptions = await get_all_subscriptions()

        for sub in subscriptions:
            if sub.vpn_user_id == client_id:
                # Клиент найден в базе
                # Используем panel_service вместо strict_panel_api
                from bot.services.panel_service import get_vpn_link

                # Формируем email в новом формате user_id@username
                user = await get_user(sub.user_id)
                if user and user.username:
                    email = f"{sub.user_id}@{user.username}"
                else:
                    email = f"{sub.user_id}@user"
                direct_link = await get_vpn_link(email=email)

                if direct_link:
                    client_data = {
                        "client_id": client_id,
                        "email": email,
                        "account_data": {"direct_link": direct_link, "id": client_id},
                    }
                break

    if not client_data:
        # Клиент не найден ни в локальном хранилище, ни в базе
        await callback.message.edit_text(
            "❌ Не удалось найти информацию о клиенте. Попробуйте снова или создайте нового клиента.",
            reply_markup=InlineKeyboardBuilder()
            .button(text="🔄 Попробовать снова", callback_data=f"admin_get_key_{client_id}")
            .button(text="🏠 В главное меню", callback_data="admin_menu")
            .adjust(1)
            .as_markup(),
        )
        await callback.answer()
        return

    # Данные для отображения
    email = client_data.get("email", "")
    account_data = client_data.get("account_data", {})
    protocol = client_data.get("protocol", "vless").lower()

    # Отправляем сообщение о процессе получения ключа
    await callback.message.edit_text(
        "⏳ Получение VPN-ключа, пожалуйста, подождите...", reply_markup=None
    )

    # Пытаемся получить ссылку через API панели

    try:
        # Пробуем получить данные из всех inbound
        sub_data = None

        # Используем новый рабочий метод генерации ключей
        logger.info(f"Генерация VPN ключа для клиента {client_id}...")

        try:
            sub_data = await generate_vpn_key_for_admin(client_id, email)
            if sub_data:
                logger.info(f"Успешно сгенерирован VPN ключ для {email}")
        except Exception as e:
            logger.warning(f"Ошибка при генерации VPN ключа: {e}")
            sub_data = None

        if sub_data:
            # Если получили данные, обновляем информацию клиента
            account_data.update(sub_data)
            logger.info(f"Данные аккаунта обновлены из Subscription URL")
    except Exception as e:
        logger.error(f"Ошибка при получении данных через Subscription URL: {e}")

    # Формируем сообщение с ключом и информацией для подключения
    key_message = f"🔑 <b>VPN-ключ для клиента:</b>\n\n"

    # Проверяем, смогли ли получить ссылку из панели
    direct_link = account_data.get("direct_link", "")

    if protocol == "vless":
        # Данные для VLESS
        vless_id = account_data.get("id") or account_data.get("client_id") or client_id
        server_value = account_data.get("server", "31.172.75.92")
        port_value = account_data.get("port", 443)

        key_message += (
            f"📱 <b>Данные подключения VLESS:</b>\n"
            f"• ID: <code>{vless_id}</code>\n"
            f"• Сервер: <code>{server_value}</code>\n"
            f"• Порт: <code>{port_value}</code>\n"
            f"• Протокол: <code>VLESS</code>\n"
            f"• Шифрование: <code>none</code>\n"
            f"• Flow: <code>xtls-rprx-vision</code>\n"
            f"• Тип подключения: <code>tcp</code>\n"
            f"• Security: <code>reality</code>\n\n"
        )
    elif protocol == "shadowsocks":
        # Данные для Shadowsocks
        password = account_data.get("password", "")
        server = account_data.get("server", "31.172.75.92")
        port = account_data.get("port", 14765)
        method = account_data.get("method", "chacha20-ietf-poly1305")

        key_message += (
            f"📱 <b>Данные подключения Shadowsocks:</b>\n"
            f"• Сервер: <code>{server}</code>\n"
            f"• Порт: <code>{port}</code>\n"
            f"• Метод шифрования: <code>{method}</code>\n"
            f"• Пароль: <code>{password}</code>\n\n"
        )

    # Проверяем наличие прямой ссылки
    if direct_link:
        key_message += f"🔗 <b>Прямая ссылка:</b>\n<code>{direct_link}</code>\n\n"
    else:
        # Если прямой ссылки нет, просто отправляем информацию
        key_message += (
            "⚠️ <b>Прямая ссылка не найдена.</b> Используйте данные выше для ручной настройки.\n\n"
            "Вы можете попробовать получить ссылку позже, когда панель обновит данные."
        )

    # Отправляем сообщение с ключом и кнопкой обновления
    await callback.message.edit_text(
        key_message,
        parse_mode="HTML",
        reply_markup=InlineKeyboardBuilder()
        .button(text="🔄 Обновить ссылку", callback_data=f"admin_get_key_{client_id}")
        .button(text="🏠 В главное меню", callback_data="admin_menu")
        .adjust(1)
        .as_markup(),
    )

    await callback.answer()


@router.message(Command("delete_test_subs"))
@admin_only
async def cmd_delete_test_subs(message: Message, state: FSMContext, **kwargs):
    """
    Найти и удалить тестовые подписки
    """
    try:
        # Получаем все подписки
        all_subscriptions = await get_all_subscriptions()

        # Фильтруем только тестовые подписки (3 дня, 15 ГБ)
        from config import TEST_PERIOD_DAYS, TEST_PERIOD_TRAFFIC_GB

        test_subs = []

        for sub in all_subscriptions:
            # Проверяем, является ли подписка тестовой по признакам:
            # 1. Срок действия 3 дня (разница между expires_at и created_at)
            # 2. Лимит трафика 15 ГБ (в байтах)
            if sub.created_at and sub.expires_at:
                duration_days = (sub.expires_at - sub.created_at).days
                # Проверяем примерное соответствие тестовому периоду
                if (
                    duration_days == TEST_PERIOD_DAYS or duration_days == TEST_PERIOD_DAYS - 1
                ) and (sub.traffic_limit == TEST_PERIOD_TRAFFIC_GB * 1024 * 1024 * 1024):
                    test_subs.append(sub)

        if not test_subs:
            await message.answer("ℹ️ Тестовых подписок не найдено.")
            return

        # Выводим список найденных тестовых подписок
        test_subs_message = "🔍 <b>Найдены тестовые подписки:</b>\n\n"

        for i, sub in enumerate(test_subs, 1):
            # Получаем информацию о пользователе
            user = await get_user(sub.user_id)
            username = f"@{user.username}" if user and user.username else "Нет username"
            user_info = f"{user.full_name if user else 'Неизвестный'} ({username})"

            # Статус подписки
            status = "✅ Активна" if sub.is_active else "❌ Неактивна"

            # Дата создания и истечения
            created = sub.created_at.strftime("%d.%m.%Y") if sub.created_at else "Неизвестно"
            expires = sub.expires_at.strftime("%d.%m.%Y") if sub.expires_at else "Неизвестно"

            test_subs_message += (
                f"{i}. <b>{user_info}</b> | {status}\n"
                f"   📅 Создана: {created}, истекает: {expires}\n"
                f"   📊 Трафик: {sub.traffic_used / (1024*1024*1024):.2f} / {sub.traffic_limit / (1024*1024*1024):.2f} ГБ\n"
                f"   🔑 ID подписки: <code>{sub.id}</code>\n\n"
            )

        # Создаем клавиатуру для подтверждения удаления
        kb = InlineKeyboardBuilder()
        kb.row(
            InlineKeyboardButton(
                text="✅ Удалить все тестовые подписки", callback_data="delete_test_subs_confirm"
            )
        )
        kb.row(InlineKeyboardButton(text="❌ Отмена", callback_data="delete_test_subs_cancel"))

        # Сохраняем список ID подписок в состоянии
        test_sub_ids = [sub.id for sub in test_subs]
        await state.update_data(test_sub_ids=test_sub_ids)

        await message.answer(
            f"{test_subs_message}\n<b>Вы действительно хотите удалить все найденные тестовые подписки?</b>",
            reply_markup=kb.as_markup(),
            parse_mode="HTML",
        )

        # Устанавливаем состояние ожидания подтверждения
        await state.set_state(AdminStates.delete_test_sub_confirmation)

    except Exception as e:
        logger.error(f"Ошибка при поиске тестовых подписок: {e}")
        await message.answer(f"❌ Произошла ошибка: {str(e)[:100]}...")


@router.callback_query(StateFilter(AdminStates.delete_test_sub_confirmation))
async def process_delete_test_subs_action(callback: CallbackQuery, state: FSMContext):
    """Обработка действий с удалением тестовых подписок"""
    await callback.answer()

    if callback.data == "delete_test_subs_confirm":
        # Получаем список ID подписок из состояния
        data = await state.get_data()
        test_sub_ids = data.get("test_sub_ids", [])

        if not test_sub_ids:
            await callback.message.edit_text("❌ Не найдены ID подписок для удаления.")
            await state.clear()
            return

        # Счетчики для статистики
        success_count = 0
        error_count = 0

        # Удаляем все подписки из списка с правильным порядком
        import aiosqlite
        from config import DATABASE_FILE

        async with aiosqlite.connect(DATABASE_FILE) as db:
            await db.execute("PRAGMA foreign_keys = ON")

            for sub_id in test_sub_ids:
                try:
                    # Удаляем связанные записи в правильном порядке
                    await db.execute("DELETE FROM payments WHERE subscription_id = ?", (sub_id,))
                    # Пропускаем notifications, если таблица не существует
                    try:
                        await db.execute(
                            "DELETE FROM notifications WHERE subscription_id = ?", (sub_id,)
                        )
                    except:
                        pass
                    await db.execute(
                        "DELETE FROM traffic_usage WHERE subscription_id = ?", (sub_id,)
                    )
                    await db.execute("DELETE FROM subscriptions WHERE id = ?", (sub_id,))

                    success_count += 1
                    logger.info(f"Успешно удалена тестовая подписка {sub_id}")

                except Exception as e:
                    logger.error(f"Ошибка при удалении подписки {sub_id}: {e}")
                    error_count += 1

            await db.commit()

        # Формируем отчет
        result_message = (
            f"✅ <b>Результат удаления тестовых подписок:</b>\n\n"
            f"Всего подписок: {len(test_sub_ids)}\n"
            f"Успешно удалено: {success_count}\n"
            f"Ошибок: {error_count}"
        )

        await callback.message.edit_text(result_message, parse_mode="HTML")

    elif callback.data == "delete_test_subs_cancel":
        await callback.message.edit_text("❌ Удаление тестовых подписок отменено.")

    # Очищаем состояние
    await state.clear()


@router.message(F.text.startswith("/fix_traffic"))
@admin_only
async def cmd_fix_traffic(message: Message, **kwargs):
    """
    Исправить синхронизацию трафика для пользователя
    Использование: /fix_traffic user_id traffic_gb
    """
    try:
        args = message.text.split()
        if len(args) < 3:
            await message.answer(
                "📝 Использование: /fix_traffic user_id traffic_gb\nПример: /fix_traffic 35 0.44"
            )
            return

        user_id = int(args[1])
        traffic_gb = float(args[2])

        from bot.database.operations import get_user_subscription, update_traffic_usage

        # Находим подписку пользователя
        subscription = await get_user_subscription(user_id)
        if not subscription:
            await message.answer(f"❌ Подписка пользователя {user_id} не найдена")
            return

        # Обновляем трафик
        await update_traffic_usage(subscription.id, traffic_gb)

        remaining = subscription.traffic_limit - traffic_gb
        await message.answer(
            f"✅ Трафик обновлен для пользователя {user_id}:\n"
            f"📊 Использовано: {traffic_gb:.2f} ГБ\n"
            f"📊 Осталось: {remaining:.2f} ГБ из {subscription.traffic_limit} ГБ\n"
            f"🔄 Данные обновлены в личном кабинете"
        )

    except ValueError:
        await message.answer("❌ Неверный формат. Используйте числа для user_id и traffic_gb")
    except Exception as e:
        logger.error(f"Ошибка при исправлении трафика: {e}")
        await message.answer(f"❌ Ошибка: {e}")


def register_admin_handlers(dp):
    """Регистрация обработчиков административных команд"""
    dp.include_router(router)
